use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::Crypto::Config;

use Digest::SHA qw(sha256);

use Xyzzy -self;

field secret => sub { die "no CryptoSecret configured\n" };
sub set_cryptosecret {
	my $key = shift;
	my $len = int(length($key)*2/3);
	my @values = (substr($key, 0, $len), substr($key, -$len));
	$self->secret([map { sha256($_) } @values]);
}

package Xyzzy::Crypto;

use MIME::Base64;
use Encode;
use Digest::SHA qw(sha256);
use Crypt::OpenSSL::Random;
use Crypt::OpenSSL::Bignum;
use Time::HiRes qw(time);

use Clarity -self;

our @EXPORT_OK = qw(encode_base decode_base);
our @EXPORT_BASE = qw(encode_base decode_base);

field cfg;
field secret => sub { shift->cfg->secret };

sub random_bytes {
	die "Insufficient entropy\n"
		unless Crypt::OpenSSL::Random::random_status;
	return Crypt::OpenSSL::Random::random_bytes(@_);
}

sub random_hex {
	my $hex = unpack('H*', $self->random_bytes($@));
	$hex =~ tr/a-z/A-Z/;
	return $hex;
}

sub hmac {
	my $token = join('', map { my $str = $_; Encode::_utf8_off($str); $str } @_);
	foreach my $key (@{$self->secret}) {
		$token = sha256($key . $token);
	}
	return $token;
}

sub create_token_data {
	my ($hidden, @keys) = @_;

	foreach(@keys) {
		Encode::_utf8_off($_);
		die "Bad characters in request\n"
			unless index($_, "\0") < 0;
	}

	my $time = int(time * 1e6);
	my $salt = $self->random_bytes(16);
	my $token = pack('Q>a*a*', $time, $salt, join("\0", @keys));

	my $hash = $self->hmac($hidden, "\0", $token);

	return $time, $salt, $hash.$token;
}

sub create_token {
	my ($time, $salt, $raw) = $self->create_token_data(@_);

	my $encoded = encode_base64($raw, '');
	$encoded =~ s/=+$//;
	$encoded =~ tr|/+|._|;

	return $time, $salt, $encoded;
}

sub check_token_data {
	my ($hidden, $token, $expiry) = @_;

	my $hash = substr($token, 0, 32, '');
	die "Invalid token\n"
		unless length($hash) == 32;

	die "Invalid token\n"
		unless $self->hmac($hidden, "\0", $token) eq $hash;

	my ($time, $salt, @keys) = unpack('Q>a16(Z*)*', $token);

	$time /= 1e6;

	my $now = time;

	die "Invalid token\n"
		if $time > $now;

	die "Expired token\n"
		if $expiry && $time + $expiry < $now;

	foreach(@keys) {
		Encode::_utf8_on($_);
		Encode::_utf8_off($_) unless utf8::valid($_);
	}

	if(wantarray) {
		return $time, $salt, @keys;
	} else {
		return (shift @keys) // 1;
	}
}

sub check_token {
	my ($hidden, $token, $expiry) = @_;

	die "Invalid token\n"
		unless defined $token;

	die "Invalid token\n"
		unless $token =~ /^[a-zA-Z0-9_.]+$/;

	$token =~ tr|._|/+|;
	$token .= '=' x -(length($token) % -4);
	$token = decode_base64($token);

	return $self->check_token_data($hidden, $token, $expiry);
}

{
	my @chars = qw(
		0 1 2 3 4 5 6 7 8 9
		a b c d e f g h i j k l m n o p q r s t u v w x y z
		A B C D E F G H I J K L M N O P Q R S T U V W X Y Z
		- _
	);

	my %chars; @chars{@chars} = 0..@chars;

	my @cache;

	my $ctx = new Crypt::OpenSSL::Bignum::CTX;
	my $b = Crypt::OpenSSL::Bignum->zero;
	my $r = Crypt::OpenSSL::Bignum->zero;

	sub encode_base {
		my ($in, $base) = @_;
		my $inlen = length($in);
		my $outlen = $inlen * log(256) / log($base);

		my $a = Crypt::OpenSSL::Bignum->new_from_bin($in);
		my $bb = $cache[$base] //= Crypt::OpenSSL::Bignum->new_from_word($base);

		my $out = '';

		for(my $i = 0; $i < $outlen; $i++) {
			$a->div($bb, $ctx, $b, $r);
			$out .= $chars[$r->get_word];
			($a, $b) = ($b, $a);
		}

		return scalar reverse($out);
	}

	sub decode_base {
		my ($in, $base) = @_;
		my $inlen = length($in);
		return '' if $inlen > 256;
		my $outlen = $inlen * log($base) / log(256);

		my $a = Crypt::OpenSSL::Bignum->zero;
		my $bb = $cache[$base] //= Crypt::OpenSSL::Bignum->new_from_word($base);

		our @cache;

		for(my $i = 0; $i < $inlen; $i++) {
			my $chr = substr($in, $i, 1) // next;
			my $ord = $chars{$chr} // next;
			next unless $ord < $base;
			my $bord = $cache[$ord] //= Crypt::OpenSSL::Bignum->new_from_word($ord);
			$a->mul($bb, $ctx, $b);
			$b->add($bord, $a);
		}

		return substr($a->to_bin, 0, $outlen);
	}
}
