#! /usr/bin/perl

use strict;
use warnings FATAL => 'all';

use Test::More;

sub survive(&) {
	my $func = shift;
	if(eval { $func->(); 1 }) {
		return 1;
	} else {
		return fail($@);
	}
}

sub suicide(&$) {
	my ($func, $message) = @_;
	my $res = eval { $func->() // 'undef' };
	if(defined $res) {
		return fail("$message: should have died but returned '$res'");
	} else {
		return ok($@);
	}
}

sub carefully(&) {
	my $func = shift;
	my $res = eval { $func->() };
	return $res unless $@;
	warn $@;
	return;
}

BEGIN { use_ok('Xyzzy::Cookie') or BAIL_OUT('need Xyzzy::Cookie to run') }

new_ok('Xyzzy::Cookie') or BAIL_OUT('need a Xyzzy::Cookie object to run');

foreach my $r (
	[zxnrbl => 'session=zxnrbl; Version=1'],
	["\xEB" => 'session=%EB; Version=1'],
	["\x{3042}" => 'session=%E3%81%82; Version=1'],
) {
	my ($value, $expected, @extra) = @$r;
	my $cookie = new_ok('Xyzzy::Cookie', [session => $value, @extra]);
	carefully { is($cookie->toString, $expected, "string representation of session cookie") };
}

do {
	my $cookie = new Xyzzy::Cookie(session => 'DEADBEEF', BLAH => 'oiNk');
	like($cookie->toString, qr/Blah/, "setparameter works case-insensitively for key names");
	like($cookie->toString, qr/oiNk/, "setparameter works case-sensitively for key values");
	$cookie->unsetparameter('blAh');
	unlike($cookie->toString, qr/blah/i, "unsetparameter works case-insensitively");
	unlike($cookie->toString, qr/oiNk/, "unsetparameter works case-insensitively");
};

done_testing();
