/******************************************************************************

	address.h -- struct sockaddr bookkeeping
	Copyright (C) 2004  Wessel Dankers <wsl@uvt.nl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	$Id: address.h 7577 2005-05-17 08:09:47Z wsl $
	$URL: https://infix.uvt.nl/its-id/trunk/sources/fair/src/address.h $

******************************************************************************/

#ifndef ADDRESS_H
#define ADDRESS_H

#include <avl.h>
#include <sys/select.h>
#include <netdb.h>

#include "fair.h"
#include "chrono.h"
#include "worker.h"

typedef enum {
	ADDRESS_EVENT_LOAD,
	ADDRESS_EVENT_UNLOAD,
	ADDRESS_EVENT_STATECHANGE
} address_event_t;

struct address;

typedef void (*address_hook_t)(struct address *, address_event_t);

typedef struct address {
	/* for adding it to worker_t->addr */
	avl_node_t node;
	/* for adding it to worker_t->prio */
	avl_node_t prio;
	/* timeout timer */
	chrono_t *timer;
	/* re-enable address after errors */
	chrono_t *whip;
	/* callback for state change notification */
	address_hook_t func;
	void *data;
	socklen_t len;
	u16 nconn;
	/* timed out */
	bool dead;
	/* benched because of errors */
	bool disabled;
	/* number of errors that have occurred */
	u8 errors;
} address_t;

typedef struct address_string {
	char host[NI_MAXHOST];
	char serv[NI_MAXSERV];
} address_string_t;

#define address_sa(addr) ((const struct sockaddr *)((const address_t *)(addr)+1))

extern address_t *address_new(const struct sockaddr *, socklen_t, address_hook_t, void *);
extern bool address_authorized(const struct sockaddr *, socklen_t);
extern int address_cmp_addr(const address_t *, const address_t *);
extern int address_cmp_prio(const address_t *, const address_t *);
extern bool address_string(address_string_t *, const address_t *);
extern bool address_string_sa(address_string_t *, const struct sockaddr *, socklen_t);
extern address_t *address_byaddr(avl_tree_t *, const struct sockaddr *, socklen_t);

extern void address_load(address_t *);
extern void address_unload(address_t *);
extern void address_working(address_t *);
extern void address_broken(address_t *);
extern void address_update(address_t *);
extern unsigned int address_count(void);

#endif
