/******************************************************************************

	carrousel.c -- main loadbalancer program
	Copyright (C) 2004  Wessel Dankers <wsl@uvt.nl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	$Id: carrousel.c 7718 2005-05-27 08:24:55Z wsl $
	$URL: https://infix.uvt.nl/its-id/trunk/sources/fair/src/carrousel.c $

******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <pwd.h>
#include <grp.h>

#include "fair.h"
#include "protocol.h"
#include "error.h"
#include "sock.h"
#include "conn.h"
#include "chrono.h"
#include "fdcopy.h"
#include "worker.h"
#include "address.h"
#include "init.h"
#include "conf.h"

typedef struct relay {
	fd_t *src, *dst;
	fdcopy_t *from, *to;
	address_t *addr;
	chrono_t *cr;
	connector_t *cn;
	size_t attempts;
	address_string_t str_from;
	address_string_t str_to;
} relay_t;

static const relay_t relay_0 = {0};

static unsigned int relay_count = 0;

static void relay_delete(relay_t *rl) {
	unless(rl) return;

	if(rl->from)
		fdcopy_delete(rl->from);
	if(rl->to)
		fdcopy_delete(rl->to);
	if(rl->src)
		fd_close(rl->src);
	if(rl->dst)
		fd_close(rl->dst);
	if(rl->cr)
		chrono_delete(rl->cr);
	if(rl->cn)
		connector_delete(rl->cn);
	if(rl->addr)
		address_unload(rl->addr);
	if(conf_Debug)
		memset(rl, 'A', sizeof *rl);
	free(rl);
	relay_count--;
}

static void throughput_to(fdcopy_t *fdc, fdcopy_event_t evt, int len) {
	relay_t *rl;
	assert(fdc);
	rl = fdc->data;
	assert(rl);
	assert(rl->cr);

	if(evt == FDCOPY_EVENT_DONE) {
		if(!rl->from)
			return relay_delete(rl);
		assert(rl->src);
		shutdown(rl->src->fd, SHUT_RD);
		assert(rl->dst);
		shutdown(rl->dst->fd, SHUT_WR);
		assert(rl->to);
		fdcopy_delete(rl->to);
		rl->to = NULL;
	}
	chrono_after(rl->cr, conf_IdleTimeout);
}

static void throughput_from(fdcopy_t *fdc, fdcopy_event_t evt, int len) {
	relay_t *rl;
	assert(fdc);
	rl = fdc->data;
	assert(rl);
	assert(rl->cr);

	if(evt == FDCOPY_EVENT_DONE) {
		if(!rl->to)
			return relay_delete(rl);
		assert(rl->dst);
		shutdown(rl->dst->fd, SHUT_RD);
		assert(rl->src);
		shutdown(rl->src->fd, SHUT_WR);
		assert(rl->from);
		fdcopy_delete(rl->from);
		rl->from = NULL;
	}
	chrono_after(rl->cr, conf_IdleTimeout);
}

static void outgoing_tcp(connector_t *cn, fd_t *fd);

static bool relay_connect(relay_t *rl) {
	address_t *addr;

	unless(rl) return FALSE;

	if(rl->attempts--) {
		addr = worker_bestaddress();
		if(addr) {
			address_string(&rl->str_to, addr);
			if(conf_Debug)
				syslog(LOG_INFO, "Connecting to %s %s",
					rl->str_to.host, rl->str_to.serv);
			address_load(rl->addr = addr);
			chrono_after(rl->cr, conf_ConnectTimeout);
			if(rl->cn)
				connector_delete(rl->cn);
			rl->cn = connector_new(addr, outgoing_tcp, rl);
			if(rl->cn)
				return TRUE;
			else
				syslog(LOG_ERR, "Setting up connection failed: %m");
		} else {
			syslog(LOG_ERR, "No available workers; dropping connection");
		}
	} else {
		syslog(LOG_ERR, "No attempts left; dropping connection");
	}
	relay_delete(rl);
	return FALSE;
}

static void relay_timeout(chrono_t *cr) {
	relay_t *rl;

	unless(cr) return;
	rl = cr->data;
	unless(rl) return;

	if(rl->cn) {
		errno = ETIMEDOUT;
		outgoing_tcp(rl->cn, NULL);
	} else {
		syslog(LOG_WARNING, "Closing idle connection from %s %s to %s %s",
			rl->str_from.host, rl->str_from.serv, rl->str_to.host, rl->str_to.serv);
		relay_delete(rl);
	}
}

static void outgoing_tcp(connector_t *cn, fd_t *fd) {
	relay_t *rl;
	address_t *addr;

	unless(cn) return;
	rl = cn->data;
	unless(rl) return;

	if(rl->cn)
		connector_delete(rl->cn);
	rl->cn = NULL;

	if(fd) {
		rl->dst = fd;
		rl->from = fdcopy_new(rl->src, fd, throughput_from, rl);
		rl->to = fdcopy_new(fd, rl->src, throughput_to, rl);
		address_string(&rl->str_to, rl->addr);
		syslog(LOG_INFO, "Connection from %s %s put through to %s %s",
			rl->str_from.host, rl->str_from.serv, rl->str_to.host, rl->str_to.serv);
		address_working(rl->addr);
		chrono_after(rl->cr, conf_IdleTimeout);
	} else {
		addr = rl->addr;
		if(addr) {
			rl->addr = NULL;
			syslog(LOG_ERR, "connect(): %m while connecting to %s %s",
				rl->str_to.host, rl->str_to.serv);
			address_broken(addr);
			address_unload(addr);
		}
		relay_connect(rl);
	}
}

static void incoming_tcp(void *_, int fd, const struct sockaddr *sa, socklen_t sa_len) {
	relay_t *rl;

	relay_count++;
	rl = xalloc(sizeof *rl);
	*rl = relay_0;

	address_string_sa(&rl->str_from, sa, sa_len);
	syslog(LOG_INFO, "Connection from %s %s",
		rl->str_from.host, rl->str_from.serv);

	rl->cr = chrono_new(relay_timeout, rl);
	rl->src = fd_new(fd);
	rl->attempts = conf_ConnectAttempts;

	relay_connect(rl);
}

static void klokje(chrono_t *cr) {
	struct tm *tm;
	time_t walltime = now / STAMP_TICKS;
	stamp_t next = (STAMP_TICKS/4) - now % (STAMP_TICKS/4);

	if(next < STAMP_TICKS/100)
		next += (STAMP_TICKS/4);

	if(cr)
		chrono_after(cr, next);

	tm = localtime(&walltime);
	eprintf("\r%04d-%02d-%02d %02d:%02d:%02d.%06d cr:%u rl:%u cn:%u fd:%u fdc:%u lst:%u wrk:%u addr:%u\033[K\r",
		tm->tm_year+1900, tm->tm_mon+1, tm->tm_mday,
		tm->tm_hour, tm->tm_min, tm->tm_sec,
		(int)(now % STAMP_TICKS),
		chrono_count(), relay_count, connector_count(), fd_count(),
		fdcopy_count(), listener_count(), worker_count(), address_count());
}

static bool strsane(const char *s, size_t len) {
	const char *e;
	for(e = s + len; s < e; s++)
		if(!isprint(*s))
			return FALSE;
	return TRUE;
}

static void incoming_udp(fd_t *fd, fd_event_t evt) {
	struct sockaddr_storage _sa;
	struct sockaddr *sa = (struct sockaddr *)&_sa;
	socklen_t len = sizeof _sa;
	byte buf[1024];
	char *hostname;
	ssize_t r;
	loadping_t *p = (loadping_t *)buf;
	worker_t *w;
	int load;

	r = recvfrom(fd->fd, buf, sizeof buf - 1, 0, sa, &len);
	if(r < sizeof *p)
		return;
	if(p->version != PROTOCOL_VERSION)
		return;
	if(!p->hostlen)
		return;
	if(r != sizeof *p + p->hostlen)
		return;

	hostname = buf + sizeof *p;
	if(memchr(hostname, '\0', p->hostlen))
		return;
	hostname[p->hostlen] = '\0';
	load = ntohs(p->capacity);

	w = worker_byname(hostname);
	if(!w && strsane(hostname, p->hostlen) && address_authorized(sa, len))
		w = worker_new(hostname);
	if(w)
		worker_update(w, load, sa, len);
}

int main(int argc, char **argv) {
	const char *err;
	int opt;
	char *config = NULL;

	while((opt = getopt(argc, argv, "c:")) != EOF) {
		switch(opt) {
			case 'c':
				config = strdup(optarg);
				break;
			default:
				syslog_exit(LOG_CRIT, "Unknown option. Program exit.");
				break;
		}
	}

	openlog("carrousel", LOG_NDELAY|LOG_PID, LOG_DAEMON);

	if(!init_signals())
		syslog_exit(LOG_CRIT, "sigaction(): %m. Program exit.");

	stamp_sync();

	if(!conf_load(config, FALSE))
		syslog_exit(LOG_CRIT, "Error reading config file. Program exit.");

	init_limits();

	if(conf_Debug)
		chrono_after(chrono_new(klokje, NULL), 0LL);

	err = listener_new_tcp(conf_BalancerService, incoming_tcp, NULL);
	if(err)
		syslog_exit(LOG_CRIT, "Error listening on TCP port %s: %s", conf_BalancerService, err);

	err = listener_new_udp(conf_TransponderService, incoming_udp, NULL);
	if(err)
		syslog_exit(LOG_CRIT, "Error listening on UDP port %s: %s", conf_TransponderService, err);

	if(!conf_Debug && !init_daemon())
		syslog_exit(LOG_CRIT, "while backgrounding: %m. Exiting.");

	if(conf_DropPrivs && !init_privs())
		syslog_exit(LOG_CRIT, errno == ENOENT
			? "init_privs(): user/group ID not found"
			: "init_privs(): %m");

	syslog(LOG_INFO, "carrousel started, listening on %s/tcp and %s/udp",
		conf_BalancerService, conf_TransponderService);

	for(;;) {
		fd_select(chrono_events());
		if(init_hupped) {
			init_hupped = FALSE;
			if(conf_reload(NULL))
				syslog(LOG_INFO, "Config reloaded.");
			else
				syslog(LOG_ERR, "Errors in config file, not reloaded");
		}
	}

	return 0;
}
