/******************************************************************************

	init.c -- set up the UNIX environment
	Copyright (C) 2004  Wessel Dankers <wsl@uvt.nl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	$Id: init.c 6337 2004-11-26 14:48:00Z wsl $
	$URL: https://infix.uvt.nl/its-id/trunk/sources/fair/src/init.c $

******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "fair.h"
#include "error.h"
#include "conf.h"

#include "init.h"

bool init_hupped = FALSE;

static void init_log_exit(int sig) {
	syslog(LOG_CRIT, "Received signal %d. Program exit.", sig);
	exit(128 + sig);
}

static void init_sighup(int sig) {
	syslog(LOG_INFO, "Received SIGHUP, scheduling config file reload.");
	init_hupped = TRUE;
}

bool init_signals(void) {
	static struct sigaction sa = {{0}};
	static int ignore[] = {
		SIGIO,
		SIGURG,
		SIGCONT,
		SIGPIPE,
		SIGALRM,
		SIGUSR1,
		SIGUSR2,
		SIGTTIN,
		SIGTTOU,
		SIGWINCH,
		SIGVTALRM
	};
	static int traplog[] = {
		SIGINT,
		SIGQUIT,
		SIGSEGV,
		SIGILL,
		SIGBUS,
		SIGABRT,
		SIGTERM,
		SIGXCPU
	};
	int i;

	sa.sa_handler = SIG_IGN;
	i = sizeof ignore / sizeof *ignore;
	while(i)
		if(sigaction(ignore[--i], &sa, NULL) == -1)
			return FALSE;

	sa.sa_handler = init_log_exit;
	i = sizeof traplog / sizeof *traplog;
	while(i)
		if(sigaction(traplog[--i], &sa, NULL) == -1)
			return FALSE;

	sa.sa_handler = init_sighup;
	if(sigaction(SIGHUP, &sa, NULL) == -1)
		return FALSE;

	return TRUE;
}

void init_limits(void) {
	struct rlimit rlim;
	int i;

	for(i = 10; i < 16; i++) {
		rlim.rlim_cur = rlim.rlim_max = 1 << i;
		setrlimit(RLIMIT_NOFILE, &rlim);
	}

	rlim.rlim_cur = rlim.rlim_max = RLIM_INFINITY;
	setrlimit(RLIMIT_NOFILE, &rlim);
	setrlimit(RLIMIT_STACK, &rlim);
	setrlimit(RLIMIT_FSIZE, &rlim);
	if(conf_Debug)
		setrlimit(RLIMIT_CORE, &rlim);
}

bool init_privs(void) {
	struct passwd *user;
	struct group *group;

	assert(conf_UserID);
	user = getpwnam(conf_UserID);
	if(!user)
		return FALSE;

	assert(conf_GroupID);
	group = getgrnam(conf_GroupID);
	if(!group)
		return FALSE;

	if(setgid(group->gr_gid) == -1)
		return FALSE;

	if(initgroups(user->pw_name, group->gr_gid) == -1)
		return FALSE;

	if(setuid(user->pw_uid) == -1)
		return FALSE;

	return TRUE;
}

bool init_daemon() {
	FILE *f;
	int err;
	pid_t pid;

	err = daemon(FALSE, FALSE);
	if(err)
		return FALSE;

	pid = getpid();
	if(pid == -1)
		return FALSE;

	assert(conf_PidFile);
	f = fopen(conf_PidFile, "w");
	if(!f)
		return FALSE;
	fprintf(f, "%ld\n", (long)pid);
	fclose(f);

	return TRUE;
}
