<?php

/**
 * CAS+LDAP authentication/authorization backend
 *
 * @license	GPL 2 (http://www.gnu.org/licenses/gpl.html)
 * @author	Andreas Gohr <andi@splitbrain.org>
 * @author	Chris Smith <chris@jalakaic.co.uk>
 * @author	Wessel Dankers <wsl@uvt.nl>
 * @author	Casper Gielen <cgielen@uvt.nl>
 *
 * $Id: auth.php 45260 2016-06-15 10:39:56Z cgielen $
 * $URL: https://svn.uvt.nl/its-id/trunk/sources/dokuwiki-cas/auth.php $
 */

require_once 'CAS.php';

class auth_plugin_authcasldap extends DokuWiki_Auth_Plugin {
	var $cnf = null;
	var $con = null;
	var $bound = 0; // 0: anonymous, 1: user, 2: superuser

	/**
	 * Constructor
	 */
	public function __construct() {
		parent::__construct();

		global $conf;
		$this->cnf = $conf['plugin']['ldap'];

		// ldap extension is needed
		if(!function_exists('ldap_connect')) {
			if($this->cnf['debug'])
				msg("LDAP err: PHP LDAP extension not found.", -1, __LINE__, __FILE__);
			$this->success = false;
			return;
		}

		$cas = $conf['plugin']['cas'];
		phpCAS::client(CAS_VERSION_2_0, $cas['host'], $cas['port'], $cas['path']);
		phpCAS::setCasServerCACert($cas['cert']);

		// SSO (CAS)
		$this->cando['logoff'] = false;
		$this->cando['external'] = true;
		$conf['userlogout'] = false;

		if(empty($this->cnf['groupkey']))
			$this->cnf['groupkey'] = 'cn';

		// try to connect
		if(!$this->_openLDAP())
			$this->success = false;

		// auth_casldap currently just handles authentication, so no
		// capabilities are set
	}

	/**
	 * Do all authentication [ OPTIONAL ]
	 *
	 * Set $this->cando['external'] = true when implemented
	 *
	 * If this function is implemented it will be used to
	 * authenticate a user - all other DokuWiki internals
	 * will not be used for authenticating, thus
	 * implementing the functions below becomes optional.
	 
	 * The function can be used to authenticate against third
	 * party cookies or Apache auth mechanisms and replaces
	 * the auth_login() function
	 *
	 * The function will be called with or without a set
	 * username. If the Username is given it was called
	 * from the login form and the given credentials might
	 * need to be checked. If no username was given it
	 * the function needs to check if the user is logged in
	 * by other means (cookie, environment).
	 *
	 * The function needs to set some globals needed by
	 * DokuWiki like auth_login() does.
	 *
	 * @see auth_login()
	 * @author	Andreas Gohr <andi@splitbrain.org>
	 *
	 * @param	string	$user	 Username
	 * @param	string	$pass	 Cleartext Password
	 * @param	bool	$sticky Cookie should not expire
	 * @return	bool			 true on successful auth
	 */
	function trustExternal($user, $pass, $sticky = false) {
		global $USERINFO;
		global $conf;
		// make sure it's really a boolean
		$sticky = $sticky ? true : false;

		phpCAS::forceAuthentication();
		$user = phpCAS::getUser();

		// make logininfo globally available
		$_SERVER['REMOTE_USER'] = $user;
		$USERINFO = $this->getUserData($user); // FIXME move all references to session

		// set cookie
		$pass= PMA_blowfish_encrypt($pass, auth_cookiesalt());
		$cookie = base64_encode("$user|$sticky|$pass");
		if($sticky)
			$time = time()+60*60*24*365; //one year
		setcookie(DOKU_COOKIE, $cookie, $time, '/');

		// set session
		$_SESSION[DOKU_COOKIE]['auth']['user'] = $user;
		$_SESSION[DOKU_COOKIE]['auth']['pass'] = $pass;
		$_SESSION[DOKU_COOKIE]['auth']['buid'] = auth_browseruid();
		$_SESSION[DOKU_COOKIE]['auth']['info'] = $USERINFO;
		return true;
	}

	/*
	 * Check user+password
	 *
	 * Checks if the given user exists and the given
	 * plaintext password is correct by trying to bind
	 * to the LDAP server
	 *
	 * @author	Andreas Gohr <andi@splitbrain.org>
	 * @return	bool
	 */
	function checkPass($user, $pass) {
		// SSO (CAS) already checked password
		die("internal error");
		return false;
	}

	/**
	 * Return user info
	 *
	 * Returns info about the given user needs to contain
	 * at least these fields:
	 *
	 * name string	full name of the user
	 * mail string	email addres of the user
	 * grps array	list of groups the user is in
	 *
	 * This LDAP specific function returns the following
	 * addional fields:
	 *
	 * dn	string	distinguished name (DN)
	 * uid	string	Posix User ID
	 *
	 * @author	Andreas Gohr <andi@splitbrain.org>
	 * @author	Trouble
	 * @author	Dan Allen <dan.j.allen@gmail.com>
	 * @auhtor	<evaldas.auryla@pheur.org>
	 * @return	array containing user data or false
	 */
	function getUserData($user) {
		global $conf;
		if(!$this->_openLDAP())
			return false;

		// force superuser bind if wanted and not bound as superuser yet
		if($this->cnf['binddn'] && $this->cnf['bindpw'] && $this->bound < 2) {
			// use superuser credentials
			if(!@ldap_bind($this->con, $this->cnf['binddn'], $this->cnf['bindpw'])) {
				if($this->cnf['debug'])
					msg('LDAP bind as superuser: '.htmlspecialchars(ldap_error($this->con)), 0, __LINE__, __FILE__);
				return false;
			}
			$this->bound = 2;
		}
		// with no superuser creds we continue as user or anonymous here

		$info['user'] = $user;
		$info['server'] = $this->cnf['server'];

		//get info for given user
		$base = $this->_makeFilter($this->cnf['usertree'], $info);
		if(!empty($this->cnf['userfilter']))
			$filter = $this->_makeFilter($this->cnf['userfilter'], $info);
		else
			$filter = "(ObjectClass=*)";

		$sr = @ldap_search($this->con, $base, $filter);
		$result = @ldap_get_entries($this->con, $sr);
		if($this->cnf['debug'])
			msg('LDAP user search: '.htmlspecialchars(ldap_error($this->con)), 0, __LINE__, __FILE__);

		// Don't accept more or less than one response
		if($result['count'] != 1)
			return false; //user not found

		$user_result = $result[0];
		ldap_free_result($sr);

		// general user info
		$info['dn'] = $user_result['dn'];
		$info['mail'] = $user_result['mail'][0];
		$info['name'] = $user_result['cn'][0];
		$info['grps'] = array();

		// overwrite if other attribs are specified.
		if(is_array($this->cnf['mapping'])) {
			foreach($this->cnf['mapping'] as $localkey => $key) {
				if(is_array($key)) {
					// use regexp to clean up user_result
					list($key, $regexp) = each($key);
					foreach($user_result[$key] as $idx => $grp) {
						if($idx === 'count')
							continue;
						if(preg_match($regexp, $grp, $match)) {
							if($localkey == 'grps')
								$info[$localkey][] = $match[1];
							else
								$info[$localkey] = $match[1];
						}
					}
				} else {
					$info[$localkey] = $user_result[$key][0];
				}
			}
		}
		$user_result = array_merge($info, $user_result);

		// get groups for given user if grouptree is given
		if($this->cnf['grouptree'] && $this->cnf['groupfilter']) {
			$base = $this->_makeFilter($this->cnf['grouptree'], $user_result);
			$filter = $this->_makeFilter($this->cnf['groupfilter'], $user_result);

			$sr = @ldap_search($this->con, $base, $filter, array($this->cnf['groupkey']));
			if(!$sr) {
				msg("LDAP: Reading group memberships failed", -1);
				if($this->cnf['debug'])
					msg('LDAP group search: '.htmlspecialchars(ldap_error($this->con)), 0, __LINE__, __FILE__);
				return false;
			}
			$result = ldap_get_entries($this->con, $sr);
			ldap_free_result($sr);

			foreach($result as $grp) {
				if(!empty($grp[$this->cnf['groupkey']][0])) {
					if($this->cnf['debug'])
						msg('LDAP usergroup: '.htmlspecialchars($grp[$this->cnf['groupkey']][0]), 0, __LINE__, __FILE__);
					$info['grps'][] = $grp[$this->cnf['groupkey']][0];
				}
			}
		}

		// always add the default group to the list of groups
		if(!in_array($conf['defaultgroup'], $info['grps']))
			$info['grps'][] = $conf['defaultgroup'];


		// extra groups from file
		if ($conf['extragroups']) {
			$lines = file($conf['extragroups']);
			$userline = array_filter($lines,
				function($line) use($info) {
					return preg_match("/^" . $info['user'] . ":/", $line);
				});
			$userline = preg_replace('/#.*$/', '', reset($userline)); //ignore comments
			$userline = trim($userline);
			$userrow = explode(":", $userline, 2);
			if (!empty($userrow[1])) {
				$extragroups = explode(",", $userrow[1]);
				if($this->cnf['debug'])
					msg('LDAP extra groups: '.htmlspecialchars(implode(",", $extragroups)),0,__LINE__,__FILE__);
				$info['grps'] = array_merge($info['grps'], $extragroups);
			}
		}
		return $info;
	}

	/**
	 * Make LDAP filter strings.
	 *
	 * Used by auth_getUserData to make the filter
	 * strings for grouptree and groupfilter
	 *
	 * filter	 string ldap search filter with placeholders
	 * placeholders array	array with the placeholders
	 *
	 * @author	Troels Liebe Bentsen <tlb@rapanden.dk>
	 * @return	string
	 */
	function _makeFilter($filter, $placeholders) {
		preg_match_all("/%{([^}]+)/", $filter, $matches, PREG_PATTERN_ORDER);
		//replace each match
		foreach ($matches[1] as $match) {
			//take first element if array
			if(is_array($placeholders[$match])) {
				$value = $placeholders[$match][0];
			} else {
				$value = $placeholders[$match];
			}
			$filter = str_replace('%{'.$match.'}', $value, $filter);
		}
		return $filter;
	}

	/**
	 * Opens a connection to the configured LDAP server and sets the wanted
	 * option on the connection
	 *
	 * @author	Andreas Gohr <andi@splitbrain.org>
	 */
	function _openLDAP() {
		if($this->con) return true; // connection already established

		$this->bound = 0;

		$port = ($this->cnf['port']) ? $this->cnf['port'] : 389;
		$this->con = @ldap_connect($this->cnf['server'], $port);
		if(!$this->con) {
			msg("LDAP: couldn't connect to LDAP server", -1);
			return false;
		}

		//set protocol version and dependend options
		if($this->cnf['version']) {
			if(!@ldap_set_option($this->con, LDAP_OPT_PROTOCOL_VERSION, 
								 $this->cnf['version'])) {
				msg('Setting LDAP Protocol version '.$this->cnf['version'].' failed', -1);
				if($this->cnf['debug'])
					msg('LDAP version set: '.htmlspecialchars(ldap_error($this->con)), 0, __LINE__, __FILE__);
			} else {
				//use TLS (needs version 3)
				if($this->cnf['starttls']) {
					if(!@ldap_start_tls($this->con)) {
						msg('Starting TLS failed', -1);
						if($this->cnf['debug'])
							msg('LDAP TLS set: '.htmlspecialchars(ldap_error($this->con)), 0, __LINE__, __FILE__);
					}
				}
				// needs version 3
				if(isset($this->cnf['referrals'])) {
					if(!@ldap_set_option($this->con, LDAP_OPT_REFERRALS, 
					 $this->cnf['referrals'])) {
						msg('Setting LDAP referrals to off failed', -1);
						if($this->cnf['debug'])
							msg('LDAP referal set: '.htmlspecialchars(ldap_error($this->con)), 0, __LINE__, __FILE__);
					}
				}
			}
		}

		//set deref mode
		if($this->cnf['deref']) {
			if(!@ldap_set_option($this->con, LDAP_OPT_DEREF, $this->cnf['deref'])) {
				msg('Setting LDAP Deref mode '.$this->cnf['deref'].' failed', -1);
				if($this->cnf['debug'])
					msg('LDAP deref set: '.htmlspecialchars(ldap_error($this->con)), 0, __LINE__, __FILE__);
			}
		}

		return true;
	}
}
