# $Id: Alias.pm 47136 2017-11-02 13:44:40Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/kiki/lib/UvT/Kiki/Database/Alias.pm $

package UvT::Kiki::Database::Alias;

use Email::Address;
use JSON;

use Clarity -self;

use constant type => 'alias';

field db;

field id;
field localpart;
field domain;
field addressbook;
field destinations => sub {
	my $self = shift;
	return $self->db->destinations_for($self->id);
};

field referrers => [];

field original => sub { shift->clone(original => undef) };

sub clone {
	my $class = ref $self;
	return $class->new(
		id => $self->id,
		localpart => $self->localpart,
		domain => $self->domain,
		addressbook => $self->addressbook,
		destinations => $self->destinations,
		referrers => $self->referrers,
		@_
	);
}

sub full {
	return $self->localpart . '@' . $self->domain->name;
}

sub unique {
	my $db = $self->db;
	my $a = $db->fetch_alias($self->full);
	return 1 unless $a;
	return 0 unless $a->can('id');
	my $id = $self->id;
	return 1 if defined $id && $id == $a->id;
	return 0;
}

sub update_localpart {
	my $newlocalpart = shift;
	return [error => 'empty-localpart'] if $newlocalpart eq '';
	if(my $original = $self->original) {
		my $oldlocalpart = $original->localpart;

		return [ok => 'unchanged', $newlocalpart]
			if $newlocalpart eq $oldlocalpart;

		if(lc($newlocalpart) eq lc($oldlocalpart)) {
			$self->localpart($newlocalpart);
			return [ok => 'case', $newlocalpart];
		}
	}

	return [error => 'malformed-localpart']
		if $newlocalpart !~ /^[a-z0-9_]+(?:[.-][a-z0-9_]+)*$/ai;

	return [error => 'localpart-too-long']
		if length($newlocalpart) > 64;

	$self->localpart($newlocalpart);
	return [ok => 'changed', $newlocalpart];
}

sub update_domain {
	my $newdomain = shift;
	return [error => 'empty-domain'] if $newdomain eq '';

	$newdomain = $self->db->domain_by_name($newdomain);
	return [error => 'unknown-domain'] unless $newdomain;

	if(my $original = $self->original) {
		my $olddomain = $original->domain;
		return [ok => 'unchanged', $olddomain->name]
			if $newdomain == $olddomain;
	}

	$self->domain($newdomain);
	return [ok => 'changed', $newdomain->name];
}

sub update_addressbook {
	my $newaddressbook = shift;
	return [error => 'empty-addressbook'] unless defined $newaddressbook;

	if(my $original = $self->original) {
		my $oldaddressbook = $original->addressbook;
		return [ok => 'unchanged', $oldaddressbook ? JSON::true : JSON::false]
			if $newaddressbook == $oldaddressbook;
	}

	$self->addressbook($newaddressbook);
	return [ok => 'changed', $newaddressbook ? JSON::true : JSON::false];
}

sub update_destinations {
	my $destinations = shift;

	if(my $original = $self->original) {
		$original->destinations;
	}

	my $db = $self->db;

	my @notes;
	my %uniq;
	my $fail;

	foreach(@$destinations) {
		if($_ eq '') {
			# leeg
			push @notes, [ok => 'empty', s/^\s*//ra];
			next;
		} elsif(/^\d{6}$/) {
			# anr, check of-ie bestaat
			my $p = $db->person_for_anr($_);
			if($p) {
				my $addr = $p->full;
				$uniq{$addr} = $p;
				push @notes, [ok => 'anr', $addr];
			} else {
				$fail = 1;
				push @notes, [error => 'unknown-anr'];
			}
		} elsif(/\@/) {
			my ($x, $y) = Email::Address->parse($_);
			if(defined $y) {
				$fail = 1;
				push @notes, [error => 'multiple'];
				next;
			}

			unless(defined $x && $x->original eq $_) {
				$fail = 1;
				push @notes, [error => 'invalid-address'];
				next;
			}

			if(my $p = $db->person_for_mailbox($x->address)) {
				my $addr = $p->full;
				$uniq{$addr} = $p;
				push @notes, [ok => 'person', $addr];
			} elsif(my $a = $db->fetch_alias($x->address)) {
				my $addr = $a->full;
				$uniq{$addr} = $a;
				push @notes, [ok => $a->type, $addr];
			} elsif($db->domain_by_name($x->host)) {
				$fail = 1;
				push @notes, [error => 'unknown-alias'];
			} else {
				my $addr = $x->address;
				$addr =~ s/\@.*/\L$&/;
				if($db->hostresolvable($addr)) {
					my $e = new UvT::Kiki::Database::External(db => $db, address => $addr);
					$uniq{$addr} = $e;
					push @notes, [ok => 'external', $addr];
				} else {
					$fail = 1;
					push @notes, [error => 'unknown-domain'];
				}
			}
		} else {
			# localpart, check of-ie bestaat en uniek is
			my $res = $db->fetch_alias_by_localpart($_);
			if(@$res < 1) {
				$fail = 1;
				push @notes, [error => 'unknown-alias'];
			} elsif(@$res > 1) {
				$fail = 1;
				push @notes, [error => 'localpart-not-unique'];
			} else {
				my ($a) = @$res;
				my $addr = $a->full;
				$uniq{$addr} = $a;
				push @notes, [ok => $a->type, $addr];
			}
		}
	}

	unless($fail) {
		if(%uniq) {
			$self->destinations([values %uniq]);
		} else {
			return [[error => 'empty-destinations']];
		}
	}

	return \@notes;
}

sub remove {
	if($self->db->has_referrers($self)) {
		return ['error', 'has-referrers'];
	} else {
		$self->db->remove_alias($self);
		return ['ok', 'removed'];
	}
}

sub expand {
	$self->destinations;
}

sub store {
	if($self->id) {
		$self->db->update_alias($self);
	} else {
		$self->db->create_alias($self);
	}
}

# A textual representation of how this entry is different from an older version.
# If the older version is undef, a pseudo-diff is generated.
sub diff {
	my $full = $self->full;
	my $ab = $self->addressbook;
	my @destinations = map { $_->full } @{$self->destinations};
	my $diff = $ab ? "$full (in address book):\n" : "$full (not in address book):\n";

	if(my $old = $self->original) {
		$diff = $old->addressbook
			? $ab
				? "$full (in address book)\n"
				: "$full (no longer in address book)\n"
			: $ab
				? "$full (now in address book)\n"
				: "$full (not in address book)\n";

		my %destinations; @destinations{@destinations} = ();
		my @old_destinations = map { $_->full } @{$old->destinations};
		my %old_destinations; @old_destinations{@old_destinations} = ();

		foreach my $destination (sort { lc($a) cmp lc($b) || $a cmp $b } @old_destinations) {
			$diff .= "-\t$destination\n" unless exists $destinations{$destination};
		}

		foreach my $destination (sort { lc($a) cmp lc($b) || $a cmp $b } @old_destinations) {
			$diff .= " \t$destination\n" if exists $destinations{$destination};
		}

		foreach my $destination (sort { lc($a) cmp lc($b) || $a cmp $b } @destinations) {
			$diff .= "+\t$destination\n" unless exists $old_destinations{$destination};
		}
	} else {
		foreach my $destination (sort { lc($a) cmp lc($b) || $a cmp $b } @destinations) {
			$diff .= " \t$destination\n";
		}
	}

	return $diff;
}

sub toString {
	return $self->full . ": "
		. join(', ', sort { lc($a) cmp lc($b) } map { $_->full } @{$self->destinations})
		. ($self->addressbook ? " (in address book)" : " (not in address book)");
}

sub json_boolean() {
	my $val = shift;
	return JSON::null unless defined $val;
	return $val ? JSON::true : JSON::false;
}

sub TO_JSON {
	my $destinations = $self->destinations;
	my %destinations;
	foreach my $destination (@$destinations) {
		push @{$destinations{$destination->type}}, $destination->full;
	}
	return {
#		type => $self->type,
		address => $self->full,
		destinations => \%destinations,
		addressbook => json_boolean($self->addressbook),
	};
}
