# Subclass of Module::Build that sets module versions during build.
#
# This is a small custom subclass of Module::Build that overrides the
# process_pm_files sub and process_script_files sub to set the $VERSION of
# each module to match the distribution version.

package Module::Build::SetVersion;

use 5.010;
use autodie;
use strict;
use warnings;

use Carp qw(croak);
use File::Spec;
use File::Temp;
use Readonly;

use parent qw(Module::Build);

our $VERSION = '1.00';

# Regex matching the line setting $VERSION in a module.
Readonly my $VERSION_REGEX => qr{
    \A (                        # capture the leading text (1)
        \s* (?: our | my ) \s*  #   optional leading our or my
        \$VERSION \s* = \s*     #   $VERSION setting
    )                           # end part being preserved
    '? \S+ '? ;                 # version setting with optional quotes
    \s* \z                      # trailing whitespace
}xms;


# filter the input file to the output file, replacing the first instance of
# the value of $VERSION with the dist_version from the module configuration.
sub _filter_file_version {
    my ($self, $in, $out) = @_;
    my $version = $self->dist_version;

    # Find the first line setting $VERSION and update it.
    while (defined(my $line = <$in>)) {
        if ($line =~ s{ $VERSION_REGEX }{$1'$version';\n}xms) {
            print {$out} $line or croak("Cannot write to output file: $!");
            last;
        }
        print {$out} $line or croak("Cannot write to output file: $!");
    }

    # Copy the remaining input to the output verbatim.
    while (defined(my $line = <$in>)) {
        print {$out} $line or croak("Cannot write to output file: $!");
    }

    # Flush output before returning since we're about to copy the file.
    $out->flush;
    return;
}


# A replacement for the default process_pm_files Module::Build method that
# sets the version number of each file to match the dist_version.
sub process_pm_files {
    my ($self, $ext) = @_;

    # find pm files
    my $method = "find_${ext}_files";
    my $files = $self->can($method) ? $self->$method() : $self->_find_file_by_type($ext,  'lib');

    # iterate pm files
    foreach my $source_file (sort keys %$files) {
        my $destination_file = File::Spec->catfile($self->blib, $files->{$source_file});

        # check if source file is newer than the destination file
        if (!$self->up_to_date($source_file, $destination_file)) {

            # create temporary file in which the version will be replaced with the distribution version
            open(my $in, q{<}, $source_file);
            my $temp_file = File::Temp->new;
            $self->_filter_file_version($in, $temp_file);
            close($in);

            # copy temporary file to destination file
            $self->copy_if_modified(from => $temp_file->filename, to => $destination_file);
        }
    }
}

# A replacement for the default process_script_files Module::Build method that
# sets the version number of each file to match the dist_version.
sub process_script_files {
    my $self = shift;

    # find script files
    my $files = $self->find_script_files;
    # return when no scripts are found
    return unless keys %$files;

    # create blib script directory
    my $script_dir = File::Spec->catdir($self->blib, 'script');
    File::Path::mkpath( $script_dir );

    # iterate script files
    foreach my $source_file (sort keys %$files) {
        my ($volume, $directories, $script_name) = File::Spec->splitpath($source_file);
        my $destination_file = File::Spec->catfile($script_dir, $script_name);

        # check if source file is newer than the destination file
        if (!$self->up_to_date($source_file, $destination_file)) {

            # create temporary file in which the version will be replaced with the distribution version
            open(my $in, q{<}, $source_file);
            my $temp = File::Temp->new;
            $self->_filter_file_version($in, $temp);
            close($in);

            # copy temporary file to destination file
            my $result = $self->copy_if_modified(from => $temp->filename, to => $destination_file) or next;

            $self->fix_shebang_line($result) unless $self->is_vmsish;
            $self->make_executable($result);
        }
    }
}

1;