# $Id: Client.pm 34594 2011-04-14 07:49:33Z wsl $
# $URL: https://svn.uvt.nl/its-id/branches/sources/aselect-perl-2/lib/Aselect/Client.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package Aselect::Client;

use Encode;
use IO::Socket::INET;
use Aselect::Util;

use Exporter 'import';

our @EXPORT = qw(aselect_authenticate aselect_verify_credentials aselect_verify_ticket aselect_attributes aselect_kill_ticket);
our @EXPORT_OK = qw(aselect_connect aselect_comm);
our %EXPORT_TAGS = (all => \@EXPORT_OK);

our $timeout = 10;
our $debug;

sub aselect_read {
	my $fh = shift;
	my $rfds = '';

	vec($rfds, fileno($fh), 1) = 1;
	my $efds = $rfds;

	my $ret = '';

	for(;;) {
		my ($rout, $eout) = ($rfds, $efds);
		my $num = select($rout, undef, $eout, $timeout);
		die "select() failed: $!\n"
			if $num == -1;
		die "timeout waiting for read\n"
			if $num == 0;
		my $buf;
		my $r = sysread($fh, $buf, 65536);
		die "error reading from socket: $!\n"
			unless defined $r;
		last if $r == 0;
		$ret .= $buf;
	}
	return decode_utf8($ret, Encode::FB_CROAK);
}

sub aselect_write {
	my $fh = shift;
	my $msg = shift;
	my $wfds = '';

	vec($wfds, fileno($fh), 1) = 1;
	my $efds = $wfds;

	my $buf = encode_utf8($msg);
	my $len = length($buf);
	my $off = 0;

	for(;;) {
		my ($wout, $eout) = ($wfds, $efds);
		my $num = select(undef, $wout, $eout, $timeout);
		die "select() failed: $!\n"
			if $num == -1;
		die "timeout waiting for write: $!\n"
			if $num == 0;
		my $r = syswrite($fh, $buf, $len - $off, $off);
		die "error writing to socket: $!\n"
			unless defined $r;
		last if $r == 0;
		$off += $r;
		last if $off == $len;
	}
}

sub aselect_connect {
	my $msg = shift;

	my $agent = new IO::Socket::INET(
		Proto => 'tcp',
		PeerAddr => 'localhost',
		PeerPort => 1495,
		Blocking => 0,
		Timeout => $timeout
	);

	die "can't connect to agent: $!\n"
		unless defined $agent;

	binmode($agent)
		or die "can't set socket binmode: $!\n";
	warn "send: $msg\n" if $debug;
	aselect_write($agent, $msg);
	$msg = aselect_read($agent);
	warn "recv: $msg\n" if $debug;

	close($agent)
		or die "can't close socket: $!\n";

	return $msg;
}

sub aselect_comm {
	my $msg = aselect_join(@_) . "\015\012";

	my $reply = aselect_connect($msg);

	my %res = aselect_split($reply);
	my $result_code = delete $res{result_code};
	die "missing result_code in agent reply\n"
		unless defined $result_code;
	die "result_code=$result_code\n"
		if $result_code ne '0000';
	return wantarray ? %res : \%res;
}

sub aselect_authenticate {
	my %args = (request => 'authenticate');
	@args{qw(app_id app_url)} = @_;
	my $res = aselect_comm(%args);
	my $as_url = delete $res->{as_url};
	die "missing as_url in agent reply\n" unless defined $as_url;
	return aselect_url($as_url, $res);
}

sub aselect_verify_credentials {
	my %args = (request => 'verify_credentials');
	@args{qw(rid aselect_credentials)} = @_;
	die "malformed credentials\n" unless aselect_valid(%args);
	my $res = aselect_comm(%args);
	my @fields = qw(uid organization ticket);
	my %ret; @ret{@fields} = @$res{@fields};
	my $ret = aselect_join(%ret);
	if(wantarray) {
		my @ret = ($ret);
		my $attrs = $res->{attributes};
		if(defined $attrs) {
			push @ret, aselect_decode($attrs);
		}
		return @ret;
	}
	return aselect_join($ret);
}

sub aselect_verify_ticket {
	my %args = (request => 'verify_ticket');
	my %ticket = aselect_split(shift);
	my @fields = qw(uid organization ticket);
	@args{@fields} = @ticket{@fields};
	$args{ticket_hash} = $ticket{ticket_hash}
		if defined $ticket{ticket_hash};
	die "malformed ticket\n" unless aselect_valid(%args);
	aselect_comm(%args);
	return @args{@fields};
}

sub aselect_attributes {
	my %args = (request => 'attributes');
	my %ticket = aselect_split(shift);
	my @fields = qw(uid organization ticket);
	@args{@fields} = @ticket{@fields};
	die "malformed ticket\n" unless aselect_valid(%args);
	my $res = aselect_comm(%args);
	return aselect_decode($res->{attributes});
}

use Data::Dumper;

sub aselect_kill_ticket {
	my %args = (request => 'kill_ticket');
	my %ticket = aselect_split(shift);
	my @fields = qw(uid organization ticket);
	@args{@fields} = @ticket{@fields};
	die "malformed ticket\n" unless aselect_valid(%args);
	my $res = aselect_comm(%args);
	return;
}

1;
