# $Id: Request.pm 36089 2011-11-23 19:23:06Z wsl $
# $URL: https://svn.uvt.nl/its-id/branches/sources/aselect-perl-2/lib/Aselect/UI/Request.pm $

use strict;
use warnings FATAL => 'all';
use utf8;

package Aselect::UI::Request;

use Aselect::UI::Error;

use Aselect::Request -self;

field gssapi => sub { shift->cfg->gssapi };

# create a new session, either anonymous or with uid
sub create_session {
	my $uid = shift;
	my $ip = $self->remote_addr;
	my ($exp, $salt, $token) = $self->crypto->create_token('s'.$ip, $uid // ());
	return new Aselect::UI::Ticket(token => $token, uid => $uid, salt => $salt, expiration => $exp);
}

# session found in input (or a new one if no valid session was found)
field cur_session => sub {
	my $self = shift;
	if(my $cookie = $self->cookie('session')) {
		my $ip = $self->remote_addr;
		if(my ($exp, $salt, $uid) = eval { $self->crypto->check_token('s'.$ip, $cookie, $self->cfg->session_timeout) }) {
			return new Aselect::UI::Ticket(token => $cookie, uid => $uid, salt => $salt, expiration => $exp);
		}
		warn "session: $@" if $@;
	}
	# if no session exists, create an anonymous one
	return undef;
};

# session that will be used in output (override in derived classes)
field new_session => sub {
	my $self = shift;
	return $self->cur_session // $self->create_session;
};

sub uid { $self->new_session->uid }

sub create_nonce {
	return scalar $self->crypto->create_token('n'.$self->new_session->token);
}

param nonce => sub {
	my $self = shift;
	$_ = eval { $self->crypto->check_token('n'.$self->cur_session->token, $_, $self->cfg->request_timeout); 1 };
};

bool_param raw;

field cur_warn => sub {
	return bool_string(shift->cookie('warn')) // 'false';
};

*new_warn = *cur_warn;

field cur_remember => sub {
	return bool_string(shift->cookie('remember')) // 'true';
};

*new_remember = *cur_remember;

field cur_spnego => sub {
	return shift->cookie('spnego') // 'auto';
};

*new_spnego = *cur_spnego;

field lang => sub {
	my $self = shift;

	my @supported = @{$self->cfg->languages};
	return undef unless @supported;
	my %supported; @supported{@supported} = ();

	if(my $lang = $self->cookie('lang')) {
		foreach my $preferred (split(qr{\s*/\s*}, lc($lang))) {
			return $preferred if exists $supported{$preferred};
		}
	}

	my $h = lc($self->http('Accept-Language'));
	$h =~ s/\s//g;

	my $curlang = $supported[0];
	my $curlevel = 0;

	foreach my $q (split(',', $h)) {
		my $level = 1;
		my ($lang, @attrs) = split(';', $q);
		$lang =~ s/-.*//;
		next unless exists $supported{$lang};
		foreach my $a (@attrs) {
			my ($key, $val) = split('=', $a, 2);
			next unless $key eq 'q';
			next unless defined $val;
			next unless $val =~ /^\d+(?:\.\d+)?$/;
			$level = $val;
		}
		if($level > $curlevel) {
			$curlang = $lang;
			$curlevel = $level;
		}
	}

	return $curlang;
};

sub error {
	my ($type, $exc, %args) = @_;
	die $exc if UNIVERSAL::isa($exc, 'Xyzzy::Response');
	warn $exc if $exc;
	my $doc = new Aselect::UI::Error(req => $self, type => $type, args => \%args);
	return $doc->response;
}
