# $Id: Aselect.pm 37889 2012-09-28 10:04:21Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package Aselect;

use Xyzzy::Directory;

use Aselect::Resources;
use Aselect::UI::Attributes;
use Aselect::UI::Cancel;
use Aselect::UI::Combined;
use Aselect::UI::Giveup;
use Aselect::UI::Login;
use Aselect::UI::Logout;
use Aselect::UI::Password;
use Aselect::UI::Session;
use Aselect::UI::Settings;
use Aselect::UI::Static;
use Aselect::UI::Status;
use Aselect::UI::Success;
use Aselect::UI::Windshield;
use Aselect::WS::RequestAuthentication;
use Aselect::WS::Server;
use Aselect::WS::ValidateCASv1;
use Aselect::WS::ValidateCASv2;
use Aselect::WS::VerifyCredentials;

use Xyzzy -self;
use Aselect::Config -mixin;
use Aselect::LDAP::Config -mixin;
use Aselect::GSSAPI::Config -mixin;
use Aselect::Crypto::Config -mixin;
use Aselect::UI::SPNEGO::Config -mixin;

sub handler {
	my $rsc = new Aselect::Resources(cfg => $self);

	my $server = new Aselect::WS::Server(cfg => $rsc, requests => {
		authenticate => new Aselect::WS::RequestAuthentication(cfg => $rsc),
		verify_credentials => new Aselect::WS::VerifyCredentials(cfg => $rsc),
	});

	my $status = new Aselect::UI::Status(cfg => $rsc);
	my $static = new Aselect::UI::Static(cfg => $rsc);
	my $success = new Aselect::UI::Success(cfg => $rsc);
	my $credentials = $self->kerberos_principal
		? new Aselect::UI::Combined(cfg => $rsc, success => $success)
		: new Aselect::UI::Password(cfg => $rsc, success => $success);

	my $root = new Xyzzy::Directory(cfg => $rsc,
		handler => $static,
		fallback => $static,
		subdirs => {
			login => new Aselect::UI::Login(cfg => $rsc, methods => [
				new Aselect::UI::Cancel(cfg => $rsc),
				new Aselect::UI::Session(cfg => $rsc, success => $success),
				$credentials,
				new Aselect::UI::Giveup(cfg => $rsc),
			]),
			status => $status,
			settings => new Aselect::UI::Settings(cfg => $rsc),
			attributes => new Aselect::UI::Attributes(cfg => $rsc),
			logout => new Aselect::UI::Logout(cfg => $rsc),
			server => $server,
			aselectserver => new Xyzzy::Directory(cfg => $rsc,
				handler => $status,
				subdirs => { server => $server },
			),
			validate => new Aselect::WS::ValidateCASv1(cfg => $rsc),
			serviceValidate => new Aselect::WS::ValidateCASv2(cfg => $rsc),
		}
	);
	return new Aselect::UI::Windshield(cfg => $rsc, handler => $root);
}

__END__

=encoding utf8

=head1 NAME

Aselect – Single Sign On for web applications

=head1 DESCRIPTION

A module for use with the Xyzzy FastCGI framework to deliver
a seamless Single Sign On experience connecting LDAP, Kerberos
(including Microsoft's Active Directory) with A-Select and CAS
aware applications.

=head1 USAGE

Sample configuration file:

	#! /usr/bin/perl /usr/lib/cgi-bin/xyzzy

	# $Id: Aselect.pm 37889 2012-09-28 10:04:21Z wsl $
	# $URL: https://svn.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect.pm $

	Application Aselect

	StylesheetDir /etc/aselect-perl/stylesheets

	LDAPServer ldap://ldap.example.org
	LDAPBase dc=example,dc=org
	LDAPUsername cn=aselect,dc=example,dc=org
	LDAPPassword hunter2
	LDAPCApath /etc/ssl/certs

	KerberosPrincipal HTTP/host.example.org@DOMAIN.EXAMPLE.ORG

	AselectSecret zxnrbl

	AselectServerID sso.example.org
	AselectCookieDomain sso.example.org

	AselectAttributes basic
		uid $uid
		anr $employeeNumber
		mail $mail

	AselectRequestor foobar basic
	AselectRequestor website basic

	AselectAttributes federated
		urn:mace:dir:attribute-def:uid $uid
		urn:mace:dir:attribute-def:employeeNumber $employeeNumber
		urn:mace:dir:attribute-def:mail $mail
		urn:mace:dir:attribute-def:sn $sn
		urn:mace:dir:attribute-def:cn $sn, $gn
		urn:mace:dir:attribute-def:givenName $gn
		urn:mace:dir:attribute-def:eduPersonEntitlement $eduPersonEntitlement
		urn:mace:dir:attribute-def:eduPersonPrincipalName $eduPersonPrincipalName

	AselectRequestor federation federated
