# $Id: Request.pm 39141 2013-03-27 08:48:54Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect/UI/Request.pm $

use strict;
use warnings FATAL => 'all';
use utf8;

package Aselect::UI::Request;

use Aselect::UI::Error;
use Aselect::UI::Ticket;
use Xyzzy::Util qw(parse_bool);

use Aselect::Request -self;

field gssapi => sub { shift->cfg->gssapi };

# create a new session, either anonymous or with uid
sub create_session {
	my $uid = shift;
	my $ip = $self->remote_addr;
	my ($exp, $salt, $token) = $self->crypto->create_token('s'.$ip, $uid // ());
	return new Aselect::UI::Ticket(token => $token, uid => $uid, salt => $salt, expiration => $exp, cfg => $self->cfg);
}

# session found in input (or a new one if no valid session was found)
field cur_session => sub {
	my $self = shift;
	if(my $cookie = $self->cookie('session')) {
		my $ip = $self->remote_addr;
		if(my ($exp, $salt, $uid) = eval { $self->crypto->check_token('s'.$ip, $cookie, $self->cfg->session_timeout) }) {
			return new Aselect::UI::Ticket(token => $cookie, uid => $uid, salt => $salt, expiration => $exp, cfg => $self->cfg);
		}
		warn "session: $@" if $@;
	}
	# if no session exists, create an anonymous one
	return undef;
};

# session that will be used in output (override in derived classes)
field new_session => sub {
	my $self = shift;
	return $self->cur_session // $self->create_session;
};

param nonce => sub {
	my $self = shift;
	$_ = eval { $self->cur_session->check_nonce($_, $self->cfg->request_timeout); 1 };
};

bool_param raw;

field gateway => sub {
	my $self = shift;
	# It is RECOMMENDED that CAS implementations ignore the "gateway" parameter if "renew" is set.
	return undef if $self->renew;
	my ($val) = $self->param('gateway')
		or return undef;
	return eval { parse_bool($val) } // 1;
};

field cur_warn => sub {
	my $warn = shift->cookie('warn');
	return 'true' if defined $warn && $warn eq 'true';
	return 'false';
};

*new_warn = *cur_warn;

field cur_remember => sub {
	my $remember = shift->cookie('remember');
	return 'false' if defined $remember && $remember eq 'false';
	return 'true';
};

*new_remember = *cur_remember;

field cur_spnego => sub {
	my $cookie = shift->cookie('spnego');
	if(defined $cookie) {
		my $spnego = eval { parse_bool($cookie) };
		return bool_string($spnego) if defined $spnego;
		warn $@ if $@;
	}
	return 'auto';
};

*new_spnego = *cur_spnego;

field lang => sub {
	my $self = shift;

	my @supported = @{$self->cfg->languages};
	return undef unless @supported;
	my %supported; @supported{@supported} = ();

	if(my $lang = $self->param('lang')) {
		$lang = lc $lang;
		return $lang if exists $supported{$lang};
	}

	if(my $lang = $self->cookie('lang')) {
		foreach my $preferred (split(qr{\s*/\s*}, lc($lang))) {
			return $preferred if exists $supported{$preferred};
		}
	}

	my $h = lc($self->http('Accept-Language') // '');
	$h =~ s/\s//g;

	my $curlang = $supported[0];
	my $curlevel = 0;

	foreach my $q (split(',', $h)) {
		my $level = 1;
		my ($lang, @attrs) = split(';', $q);
		$lang =~ s/-.*//;
		next unless exists $supported{$lang};
		foreach my $a (@attrs) {
			my ($key, $val) = split('=', $a, 2);
			next unless $key eq 'q';
			next unless defined $val;
			next unless $val =~ /^\d+(?:\.\d+)?$/;
			$level = $val;
		}
		if($level > $curlevel) {
			$curlang = $lang;
			$curlevel = $level;
		}
	}

	return $curlang;
};

sub errorpage {
	my ($type, $exc, %args) = @_;
	die $exc if UNIVERSAL::isa($exc, 'Xyzzy::Response');
	warn $exc if $exc;
	my $doc = new Aselect::UI::Error(req => $self, type => $type, args => \%args);
	return $doc->response;
}
