/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */

/* 
 * $Id: mod_aselect_filter.c,v 1.10 2006/03/01 08:56:39 remco Exp $
 * 
 * Changelog:
 * $Log: mod_aselect_filter.c,v $
 * Revision 1.10  2006/03/01 08:56:39  remco
 * Added option "url=..." in aselect_filter_add_secure_app, which forces the A-Select Server and filter to redirect to a fixed url after a succesful authentication.
 *
 * Revision 1.9  2005/09/13 14:15:31  remco
 * Fixed small bugs found during system test:
 * - verify_ticket uri is now sent according to API
 * - if rule upload failes filter does not start
 *
 * Revision 1.8  2005/09/12 13:50:49  remco
 * Added authorization support (untested)
 *
 * Revision 1.7  2005/05/23 15:40:42  leon
 * html logout button changed in logout bar template
 *
 * Revision 1.5  2005/05/11 08:20:50  remco
 * Fixed bug with option "remote-organization"
 *
 * Revision 1.4  2005/05/04 13:20:02  remco
 * - Added "remote_organization" config directive
 * - Changed directive "extra" to "extra_parameters"
 *
 * Revision 1.3  2005/05/04 08:55:00  remco
 * - Added application config directive "extra="
 * - Filter now logs if incoming message from Agent is too large
 *
 * Revision 1.2  2005/05/02 09:21:15  remco
 * - Changed parsing of application options
 * - Added 3 new application options: uid, language, and country
 * - Changed internal application storage
 * - Fixed handling of errors: some errors incorrectly resulted in re-authentication. Error response is now the same as in the ISAPI filter.
 *
 * Revision 1.1  2005/04/13 09:44:29  remco
 * RC1 of integrated apache 1.3 & 2.0 filter
 *
 */

#include "asf_common.h"

// -----------------------------------------------------
// Exports
// -----------------------------------------------------
#ifdef APACHE_13_ASELECT_FILTER
module MODULE_VAR_EXPORT    aselect_filter_module;
#else
module AP_MODULE_DECLARE_DATA aselect_filter_module;
#endif


//static handler_rec      aselect_filter_handlers[];
static const command_rec    aselect_filter_cmds[];


// -----------------------------------------------------
// Functions 
// -----------------------------------------------------

int             aselect_filter_upload_authz_rules(PASELECT_FILTER_CONFIG pConfig, server_rec *pServer, pool *pPool, PASELECT_APPLICATION pApp);
char *          aselect_filter_verify_ticket(request_rec *pRequest, pool *pPool, PASELECT_FILTER_CONFIG pConfig, char *pcTicket, char *pcUID, char *pcOrganization, char *pcAttributes);
char *          aselect_filter_kill_ticket(request_rec *pRequest, pool *pPool, PASELECT_FILTER_CONFIG pConfig, char *pcTicket );
char *          aselect_filter_auth_user(request_rec *pRequest, pool *pPool, PASELECT_FILTER_CONFIG pConfig, char *pcAppUrl );
char *          aselect_filter_verify_credentials(request_rec *pRequest, pool *pPool, PASELECT_FILTER_CONFIG pConfig, char *pcRID, char *pcCredentials );
static int      aselect_filter_handler(request_rec *pRequest );
static void *   aselect_filter_create_config(pool *pPool, server_rec *pServer );
static int      aselect_filter_verify_config(PASELECT_FILTER_CONFIG pConfig );
static const char * aselect_filter_set_agent_address(cmd_parms *parms, void *mconfig, const char *arg );
static const char * aselect_filter_set_agent_port(cmd_parms *parms, void *mconfig, const char *arg );
static const char * aselect_filter_add_secure_app(cmd_parms *parms, void *mconfig, const char *arg1, const char *arg2, const char *arg3);
static const char * aselect_filter_add_authz_rule(cmd_parms *parms, void *mconfig, const char *arg1, const char *arg2, const char *arg3);
static const char * aselect_filter_set_html_error_template(cmd_parms *parms, void *mconfig, const char *arg );
static const char * aselect_filter_set_redirection_mode(cmd_parms *parms, void *mconfig, const char *arg );
static const char * aselect_filter_set_use_aselect_bar(cmd_parms *parms, void *mconfig, const char *arg );


//
// Called once during the module initialization phase.
// can be used to setup the filter configuration 
//

#ifdef APACHE_13_ASELECT_FILTER
void
aselect_filter_init(server_rec *pServer, pool *pPool)
#else
int
aselect_filter_init(apr_pool_t *pconf, apr_pool_t *plog, apr_pool_t *pPool, server_rec *pServer)
#endif
{
    int i;
    PASELECT_APPLICATION pApp;
    char pcMessage[2048];
    PASELECT_FILTER_CONFIG  pConfig = (PASELECT_FILTER_CONFIG) 
        ap_get_module_config(pServer->module_config, &aselect_filter_module);

    TRACE1("aselect_filter_init: %x", pServer);

    if (pConfig)
    {
        ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, pServer,
                "ASELECT_FILTER:: initializing");

        ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, pServer,
            ap_psprintf(pPool, "ASELECT_FILTER:: A-Select Agent running on: %s:%d", 
                    pConfig->pcASAIP, pConfig->iASAPort));

        if (pConfig->bUseASelectBar)
            ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, pServer,
                "ASELECT_FILTER:: configured to use the a-select bar");

        if (pConfig->iRedirectMode == ASELECT_FILTER_REDIRECT_TO_APP)
            ap_log_error( APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, pServer,
                "ASELECT_FILTER:: configured to redirect to application entry point" );
        else if (pConfig->iRedirectMode == ASELECT_FILTER_REDIRECT_FULL)
            ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, pServer,
                "ASELECT_FILTER:: configured to redirect to user entry point");

        ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, pServer,
            "ASELECT_FILTER:: configured to use the a-select bar");

        for (i=0; i<pConfig->iAppCount; i++)
        {
            pApp = &pConfig->pApplications[i];
            ap_snprintf(pcMessage, sizeof(pcMessage),
                "ASELECT_FILTER:: added %s at %s %s%s",
                pApp->pcAppId,
                pApp->pcLocation,
                pApp->bEnabled ? "" : "(disabled)",
                pApp->bForcedLogon ? "(forced logon)" : "");
            ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, 
                pServer, pcMessage);
            if (aselect_filter_upload_authz_rules(pConfig, pServer, pPool, pApp))
            {
                ap_snprintf(pcMessage, sizeof(pcMessage), 
                    "ASELECT_FILTER:: registered %d authZ rules for %s",
                    pApp->iRuleCount, pApp->pcAppId);
                ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, 
                    pServer, pcMessage);
            }
            else
#ifdef APACHE_20_ASELECT_FILTER
                return -1;
#else
                pConfig->bConfigError = 1;
#endif
        }
    }
    TRACE("aselect_filter_init: done");
#ifdef APACHE_20_ASELECT_FILTER
    return 0;
#endif
}


// Upload authorization rule for a single application
int
aselect_filter_upload_authz_rules(PASELECT_FILTER_CONFIG pConfig, 
    server_rec *pServer, pool *pPool, PASELECT_APPLICATION pApp)
{
    int i, length;
    char *pRequest;
    char *pAppId;
    char pRuleId[16];
    char *pcResponse;
    int iRet;
    
    if (pApp->iRuleCount == 0) return 1;
    
    // Calculate size of request
    pAppId = aselect_filter_url_encode(pPool, pApp->pcAppId);
    length = 64 + strlen(pAppId);
    for (i=0; i<pApp->iRuleCount; i++)
    {
        length += 32 + strlen(pApp->pTargets[i]) + 
            strlen(pApp->pConditions[i]);
    }
    
    // Create request
    pRequest = (char *)ap_palloc(pPool, length);
    if (pRequest)
    {
        strcpy(pRequest, "request=set_authorization_rules&app_id=");
        strcat(pRequest, pAppId);
        for (i=0; i<pApp->iRuleCount; i++)
        {
            ap_snprintf(pRuleId, sizeof(pRuleId), "r%d", i);
            strcat(pRequest, "&rules%5B%5D=");
            strcat(pRequest, pRuleId);
            strcat(pRequest, "%3B");
            strcat(pRequest, pApp->pTargets[i]);
            strcat(pRequest, "%3B");
            strcat(pRequest, pApp->pConditions[i]);
        }
        strcat(pRequest, "\r\n");
        
        TRACE1("aselect_filter_upload_authz_rules: sending: %s", pRequest);
        pcResponse = aselect_filter_send_request(pServer, 
                pPool,
                pConfig->pcASAIP,
                pConfig->iASAPort,
                pRequest,
                strlen(pRequest));
        if (pcResponse == NULL)
        {
            // Could not send request, error already logged
            return 0;
        }
        TRACE1("aselect_filter_upload_authz_rules: response: %s", pcResponse);
        iRet = aselect_filter_get_error(pPool, pcResponse);
        if (iRet != 0)
        {
            ap_log_error(APLOG_MARK, APLOG_NOERRNO|APLOG_NOTICE, pServer, 
                ap_psprintf(pPool, "ASELECT_FILTER:: Agent returned "
                    "error %s while uploading authorization rules", 
                    pcResponse));
            return 0;
        }
    }
    else
    {
        TRACE("aselect_filter_upload_authz_rules: Out of memory");
        return 0;
    }
    return 1;
}

//
// Request validation of a ticket
//
char *
aselect_filter_verify_ticket(request_rec *pRequest, 
                pool *pPool,
                PASELECT_FILTER_CONFIG pConfig,
                char *pcTicket,
                char *pcUID,
                char *pcOrganization,
                char *pcAttributes)
{
    char *          pcSendMessage;
    int             ccSendMessage;
    char *          pcResponse;
    AP_SHA1_CTX     ctxSHA1;
    char            cSHA1[SHA_DIGESTSIZE];
    char            pcSHA1[64];
    char            *pcURI;
    
    TRACE("aselect_filter_verify_ticket");
    
    //
    // Create the message
    //
    if (*pcAttributes)
    {
        ap_SHA1Init(&ctxSHA1);
        ap_SHA1Update(&ctxSHA1, pcAttributes, strlen(pcAttributes));
        ap_SHA1Final(cSHA1, &ctxSHA1);
        aselect_filter_bytes_to_hex(cSHA1, sizeof(cSHA1), pcSHA1);
    }
    else
        *pcSHA1 = 0;

    pcURI = pRequest->uri + strlen(pConfig->pCurrentApp->pcLocation);
    pcURI = aselect_filter_url_encode(pPool, pcURI);
    
    pcSendMessage = ap_psprintf(pPool, 
        "request=verify_ticket&ticket=%s&app_id=%s&uid=%s&organization=%s&attributes_hash=%s&request_uri=%s&ip=%s\r\n", 
        pcTicket, pConfig->pCurrentApp->pcAppId, pcUID, pcOrganization, 
        pcSHA1, pcURI, pRequest->connection->remote_ip);
    ccSendMessage = strlen(pcSendMessage);

    TRACE2("request(%d): %s", ccSendMessage, pcSendMessage);
    pcResponse = aselect_filter_send_request(pRequest->server, 
                        pPool,
                        pConfig->pcASAIP,
                        pConfig->iASAPort,
                        pcSendMessage,
                        ccSendMessage);

    return pcResponse;
}

//
// Kills the ticket
//
char *
aselect_filter_kill_ticket(request_rec *pRequest,
                                pool *pPool,
                                PASELECT_FILTER_CONFIG pConfig,
                                char *pcTicket)
{
    char            *pcSendMessage;
    int             ccSendMessage;
    char            *pcResponse;

    TRACE("aselect_filter_kill_ticket");
    
    //
    // Create the message
    //
    pcSendMessage = ap_psprintf(pPool, "request=kill_ticket&ticket=%s&app_id=%s\r\n", 
        aselect_filter_url_encode(pPool, pcTicket), 
        aselect_filter_url_encode(pPool, pConfig->pCurrentApp->pcAppId));
    ccSendMessage = strlen(pcSendMessage);

    TRACE2("request(%d): %s", ccSendMessage, pcSendMessage);

    pcResponse = aselect_filter_send_request(pRequest->server,
                    pPool,
                    pConfig->pcASAIP,
                    pConfig->iASAPort,
                    pcSendMessage,
                    ccSendMessage);

    return pcResponse;
}

//
// Request an authentication of a user
//
char *
aselect_filter_auth_user(request_rec *pRequest,
            pool *pPool,
            PASELECT_FILTER_CONFIG pConfig,
            char *pcAppUrl)
{
    char    *pcSendMessage;
    int     ccSendMessage;
    char    *pcResponse;

    TRACE("aselect_filter_auth_user");

    //
    // Create the message
    //
    pcSendMessage = ap_psprintf(pPool, "request=authenticate&app_url=%s&app_id=%s&forced_logon=%s%s%s%s%s%s\r\n", 
        aselect_filter_url_encode(pPool, pcAppUrl), 
        aselect_filter_url_encode(pPool, pConfig->pCurrentApp->pcAppId),
        pConfig->pCurrentApp->bForcedLogon ? "true" : "false",
        pConfig->pCurrentApp->pcUid,
        pConfig->pCurrentApp->pcCountry,
        pConfig->pCurrentApp->pcLanguage,
        pConfig->pCurrentApp->pcRemoteOrg,
        pConfig->pCurrentApp->pcExtra);
    ccSendMessage = strlen(pcSendMessage);

    TRACE2("request(%d): %s", ccSendMessage, pcSendMessage);

    if ((pcResponse = aselect_filter_send_request(pRequest->server,
                            pPool,
                            pConfig->pcASAIP,
                            pConfig->iASAPort,
                            pcSendMessage,
                            ccSendMessage)))
    {
        TRACE1("response message: %s", pcResponse);
    }
    else
    {
        //
        // socket error occured, take appropiate action
        //
        pcResponse = NULL;
    }

    return pcResponse;
}

//
// Verify the credentials
// sends the RID and the credentials to the Aselect Agent for verification
//
char *
aselect_filter_verify_credentials(request_rec *pRequest,
            pool *pPool,
            PASELECT_FILTER_CONFIG pConfig,
            char *pcRID,
            char *pcCredentials)
{
    char    *pcSendMessage;
    int     ccSendMessage;
    char    *pcResponse;

    TRACE("aselect_filter_verify_credentials");

    //
    // Create the message
    //
    pcSendMessage = ap_psprintf(pPool, "request=verify_credentials&rid=%s&aselect_credentials=%s\r\n", 
        aselect_filter_url_encode(pPool, pcRID), 
        aselect_filter_url_encode(pPool, pcCredentials));
    ccSendMessage = strlen(pcSendMessage);

    TRACE2("request(%d): %s", ccSendMessage, pcSendMessage);
    if ((pcResponse = aselect_filter_send_request(pRequest->server,
                            pPool,
                            pConfig->pcASAIP,
                            pConfig->iASAPort,
                            pcSendMessage,
                            ccSendMessage)))
    {
        TRACE1("response message: %s", pcResponse);
    }
    else
    {
        //
        // could not send request to A-Select Agent
        //
        pcResponse = NULL;
    }

    return pcResponse;
}


static int
aselect_filter_verify_config(PASELECT_FILTER_CONFIG pConfig)
{
    if (pConfig->bConfigError)
        return ASELECT_FILTER_ERROR_CONFIG;
    if (pConfig->pcASAIP)
    {
        if (strlen(pConfig->pcASAIP) <= 0)
            return ASELECT_FILTER_ERROR_CONFIG;
    }
    else
        return ASELECT_FILTER_ERROR_CONFIG;

    if (pConfig->iASAPort == 0)
        return ASELECT_FILTER_ERROR_CONFIG;

    if (pConfig->iAppCount == 0)
        return ASELECT_FILTER_ERROR_CONFIG;

    if (pConfig->pcErrorTemplate)
    {
        if (strlen(pConfig->pcErrorTemplate) <= 0)
            return ASELECT_FILTER_ERROR_CONFIG;
    }
    else
        return ASELECT_FILTER_ERROR_CONFIG;
    
    return ASELECT_FILTER_ERROR_OK;
}


//
// Main handler, will handle cookie checking and redirection
//
static int
aselect_filter_handler(request_rec *pRequest)
{
    int             iRet = FORBIDDEN;
    int             iError = ASELECT_FILTER_ERROR_OK;
    int             iAction = ASELECT_FILTER_ACTION_ACCESS_DENIED;
    table           *headers_in = pRequest->headers_in;
    table           *headers_out = pRequest->headers_out;
    PASELECT_FILTER_CONFIG  pConfig;
    char            *pcTicketIn;
    char            *pcTicketOut = NULL;
    char            *pcUIDIn;
    char            *pcUIDOut = NULL;
    char            *pcOrganizationIn;
    char            *pcOrganizationOut = NULL;
    char            *pcAttributesIn;
    char            *pcTicket;
    char            *pcCredentials;
    char            *pcRID;
    char            *pcAppUrl;
    char            *pcCookie;
    char            *pcCookie2;
    char            *pcCookie3;
    char            *pcCookie4;
    char            *pcASUrl;
    char            *pcASelectServer;
    char            *pcResponseVT;
    char            *pcResponseAU;
    char            *pcResponseCred;
    pool            *pPool;
    char            *pcUrl;
    char            *pcRequest;
    char            *pcASelectAppURL;
    char            *pcASelectServerURL;
    char            *pcResponseKill;
    char            *pcTmp;
    char            *pcTmp2;
    char            *pcStrippedParams;
    char            *pcAttributes = NULL;
    int             bFirstParam;

#ifdef ASELECT_FILTER_TRACE
    TRACE("");
    TRACE("----------------------------------------------------------- aselect_filter_handler");
    TRACE1("GET %s", pRequest->uri);
    TRACE("-----------------------------------------------------------");
    ap_table_do(aselect_filter_print_table, pRequest, headers_in, NULL);
#endif

    //
    // Select which pool to use
    //
    
#ifdef APACHE_13_ASELECT_FILTER
    if ((pPool = ap_make_sub_pool(pRequest->pool)) == NULL)
#else
    if ((apr_pool_create(&pPool, pRequest->pool)) != APR_SUCCESS)
#endif
    {
        //
        // could not allocate pool
        //
        TRACE("aselect_filter_handler:: could not allocate memory pool");
        ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, pRequest,
             "ASELECT_FILTER:: could allocate memory pool");
        return iRet;
    }

    //
    // NOTE: the application ticket is a cookie, check if the browser can handle cookies else we
    // run into a loop
    // check cookie, no cookie, validate_user, set cookie, check cookie, no cookie.... and so on
    //
    
    //
    // Read config data
    //
    if (!(pConfig = (PASELECT_FILTER_CONFIG) 
        ap_get_module_config(pRequest->server->module_config, &aselect_filter_module)))
    {
        //
        // Something went wrong, access denied
        //
        TRACE("could not get module config data");
        ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, pRequest,
            "ASELECT_FILTER:: could not retrieve configuration data");

        //
        // Cleanup
        //
        ap_destroy_pool(pPool);
        return iRet;
    }
    else
    {
        //
        // Verify configuration
        //
        if (aselect_filter_verify_config(pConfig) != ASELECT_FILTER_ERROR_OK)
        {
            TRACE("invalid configuration data");
            ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_ERR, pRequest,
                "ASELECT_FILTER:: invalid configuration data");
            return iRet;
        }
    }

    //
    // Check if we are in a protected dir
    // this function point the global pConfig->pCurrentApp to the requested app
    //
    if (aselect_filter_verify_directory(pPool, pConfig, pRequest->uri) == ASELECT_FILTER_ERROR_FAILED)
    {
        //
        // not in a protected dir
        // this should not be possible
        // but we let the request through anyway
        //
        TRACE1("\"%s\" is not a protected dir (or is disabled)", pRequest->uri);
        ap_destroy_pool(pPool);   
        return DECLINED;
    }
    else
    {
        TRACE2("\"%s\" is a protected dir, app_id: %s", pRequest->uri, pConfig->pCurrentApp->pcAppId);
    }

    //
    // Retrieve the remote_addr
    //
    if (pRequest->connection->remote_ip) 
    {
        TRACE1("remote_ip: %s", pRequest->connection->remote_ip);
    }

    //
    // Check for application ticket
    //
    if ((pcTicketIn = aselect_filter_get_cookie(pPool, headers_in, "aselectticket=")))
    {
        TRACE1("aselect_filter_handler: found ticket: %s", pcTicketIn);
        
        //
        // Check for user ID
        //
        if ((pcUIDIn = aselect_filter_get_cookie(pPool, headers_in, "aselectuid=")))
        {
            TRACE1("aselect_filter_handler: found uid: %s", pcUIDIn);
            
            //
            // Check for Organization 
            //
            if ((pcOrganizationIn = aselect_filter_get_cookie(pPool, headers_in, "aselectorganization=")))
            {
                TRACE1("aselect_filter_handler: found organization: %s", pcOrganizationIn);

                //
                // Check attributes
                //
                if ((pcAttributesIn = aselect_filter_get_cookie(pPool, headers_in, "aselectattributes=")))
                {
                    TRACE1("aselect_filter_handler: found attributes: %s", pcAttributesIn);
                    //
                    // Validate ticket
                    //
                    if ((pcResponseVT = aselect_filter_verify_ticket(pRequest, pPool, pConfig, pcTicketIn, pcUIDIn, pcOrganizationIn, pcAttributesIn)))
                    {
                        iError = aselect_filter_get_error(pPool, pcResponseVT);
        
                        if (iError == ASELECT_FILTER_ASAGENT_ERROR_OK)
                        {
                            //
                            // User has ticket, ACESS GRANTED
                            //
                            iAction = ASELECT_FILTER_ACTION_ACCESS_GRANTED;
                            TRACE("aselect_filter_handler: ACCESS_GRANTED");
                        }
                        else
                        {
                            if (iError == ASELECT_SERVER_ERROR_TGT_NOT_VALID ||
                                iError == ASELECT_SERVER_ERROR_TGT_EXPIRED ||
                                iError == ASELECT_SERVER_ERROR_TGT_TOO_LOW ||
                                iError == ASELECT_FILTER_ASAGENT_ERROR_TICKET_INVALID ||
                                iError == ASELECT_FILTER_ASAGENT_ERROR_TICKET_EXPIRED ||
                                iError == ASELECT_FILTER_ASAGENT_ERROR_UNKNOWN_TICKET)
                            {
                                iError = ASELECT_FILTER_ERROR_OK;
                                iAction = ASELECT_FILTER_ACTION_VERIFY_CREDENTIALS;
                            }
                            else
                            {
                                TRACE1("aselect_filter_verify_ticket FAILED (%d)", iError);
                                ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, pRequest,
                                    ap_psprintf(pPool, "ASELECT_FILTER::aselect_filter_verify_ticket FAILED (%d)", iError));
                            }
                        }
                    }
                }
                else
                {
                    //
                    // Ticket verification failed, check if credentials is valid (if credentials is present)
                    //
                    iError = ASELECT_FILTER_ASAGENT_ERROR_CORRUPT_ATTRIBUTES;
                }
            }
            else
            {
                //
                // Could not find a inst-id, check if the user has credentials 
                //
                iAction = ASELECT_FILTER_ACTION_VERIFY_CREDENTIALS;
            }
        }
        else
        {
            //
            // Could not find a user-id, check if the user has credentials 
            //
            iAction = ASELECT_FILTER_ACTION_VERIFY_CREDENTIALS;
        }
    }
    else
    {
        //
        // Could not find a application ticket, check if the user has credentials
        //
        iAction = ASELECT_FILTER_ACTION_VERIFY_CREDENTIALS;
    }

    if (iAction == ASELECT_FILTER_ACTION_VERIFY_CREDENTIALS)
    {
        TRACE1("ARGUMENTS: %s", pRequest->args);

        //
        // Check for user credentials 
        //
        if ((pcCredentials = aselect_filter_get_param(pPool, pRequest->args, "aselect_credentials=", "&", TRUE)))
        {
            TRACE1("aselect_credentials: %s", pcCredentials);

            if ((pcRID = aselect_filter_get_param(pPool, pRequest->args, "rid=", "&", TRUE)))
            {
                //
                // Found credentials, now verify them, if ok it returns a ticket
                //
                if ((pcResponseCred = aselect_filter_verify_credentials(pRequest, pPool, pConfig, pcRID, pcCredentials)))
                {
                    iError = aselect_filter_get_error(pPool, pcResponseCred);
                    if (iError == ASELECT_FILTER_ERROR_INTERNAL)
                        iError = ASELECT_FILTER_ERROR_AGENT_RESPONSE;

                    if (iError == ASELECT_FILTER_ASAGENT_ERROR_OK)
                    {
                        //
                        // User credentials are ok, set application ticket and let user through
                        //
                        if ((pcTicketOut = aselect_filter_get_param(
                            pPool, pcResponseCred, "ticket=", "&", TRUE)) != NULL)
                        {
                            //
                            // Save Uid
                            //
                            if ((pcUIDOut = aselect_filter_get_param(pPool, pcResponseCred, "uid=", "&", TRUE)))
                            {
                                if ((pcOrganizationOut = aselect_filter_get_param(pPool, pcResponseCred, "organization=", "&", TRUE)))
                                {
                                    pcAttributes = aselect_filter_get_param(pPool, pcResponseCred, "attributes=", "&", TRUE);
                                    if (pcAttributes)
                                        pcAttributes = aselect_filter_base64_decode(pPool, pcAttributes);
                                    iAction = ASELECT_FILTER_ACTION_SET_TICKET;
                                }
                                else
                                {
                                    TRACE1("could not find organization in response: %s", pcResponseCred);
                                    iError = ASELECT_FILTER_ERROR_AGENT_RESPONSE;
                                }
                            }
                            else
                            {
                                TRACE1("could not find uid in response: %s", pcResponseCred);
                                iError = ASELECT_FILTER_ERROR_AGENT_RESPONSE;
                            }
                        }
                        else
                        {
                            //
                            // Could not find ticket in response
                            //
                            TRACE1("could not find ticket in response: %s", pcResponseCred);
                            iError = ASELECT_FILTER_ERROR_AGENT_RESPONSE;
                        }
                    }
                    else
                    {
                        TRACE1("aselect_filter_verify_tgt FAILED (%d)", iError);
                        if (iError == ASELECT_SERVER_ERROR_TGT_NOT_VALID ||
                            iError == ASELECT_SERVER_ERROR_TGT_EXPIRED ||
                            iError == ASELECT_SERVER_ERROR_TGT_TOO_LOW ||
                            iError == ASELECT_FILTER_ASAGENT_ERROR_TICKET_INVALID ||
                            iError == ASELECT_FILTER_ASAGENT_ERROR_TICKET_EXPIRED ||
                            iError == ASELECT_FILTER_ASAGENT_ERROR_UNKNOWN_TICKET)
                        {
                            iAction = ASELECT_FILTER_ACTION_AUTH_USER;
                            iError = ASELECT_FILTER_ERROR_OK;
                        }
                    }
                }
                else
                {
                    iError = ASELECT_FILTER_ERROR_AGENT_NO_RESPONSE;
                }
            }
            else
            {
                //
                // Could not find a RID, authenticate user
                //
                iAction = ASELECT_FILTER_ACTION_AUTH_USER;
            }
        }
        else
        {
            //
            // No Credentials present, authenticate user
            //
            iAction = ASELECT_FILTER_ACTION_AUTH_USER;
        }
    }

    //
    // if we do not have an error then
    // act according to iAction
    //
    if (iError == ASELECT_FILTER_ERROR_OK)
    {
        //
        // Act according to action
        //
        switch(iAction)
        {
            case ASELECT_FILTER_ACTION_ACCESS_GRANTED:

                //
                // User was granted access
                //
                TRACE("ASELECT_FILTER_ACTION_ACCESS_GRANTED");

                //
                // check for requests such as show_aselect_bar and kill_ticket
                //
                if (pRequest->args)
                {
                    if ((pcRequest = aselect_filter_get_param(pPool, pRequest->args, "request=", "&", TRUE)))
                    {
                        if (strstr(pcRequest, "aselect_show_bar") && pConfig->bUseASelectBar)
                        {
                            //
                            // must show the the aselect_bar
                            //
                            if ((pcASelectAppURL = aselect_filter_get_param(pPool, pRequest->args, "aselect_app_url=", "&", TRUE)))
                                iRet = aselect_filter_gen_barhtml(pPool, pRequest, pConfig, pcASelectAppURL);
                            else
                                iRet = aselect_filter_gen_barhtml(pPool, pRequest, pConfig, pConfig->pCurrentApp->pcLocation);
                        }
                        else if (strstr(pcRequest, "aselect_generate_bar"))
                        {
                            //
                            // return the bar html content
                            //
                            pRequest->content_type = "text/html";

                            ap_send_http_header(pRequest);
                            ap_rprintf(pRequest, ASELECT_LOGOUT_BAR, pConfig->pCurrentApp->pcLocation);
                            iRet = DONE;
                        }
                        else if (strstr(pcRequest, "aselect_kill_ticket"))
                        {
                            //
                            // kill the user ticket
                            //
                            if ((pcTicket = aselect_filter_get_cookie(pPool, headers_in, "aselectticket=")))
                            {
                                if ((pcResponseKill = aselect_filter_kill_ticket(pRequest, pPool, pConfig, pcTicket)))
                                {
                                        iError = aselect_filter_get_error(pPool, pcResponseKill);

                                        if (iError == ASELECT_FILTER_ASAGENT_ERROR_OK)
                                        {
                                            //
                                            // Successfully killed the ticket, now redirect to the aselect-server
                                            //
                                            if ((pcASelectServerURL = aselect_filter_get_cookie(pPool, headers_in, "aselectserverurl=")))
                                            {
                                                pRequest->content_type = "text/html";
                                                
                                                pcCookie = ap_psprintf(pPool, "%s=; version=1; max-age=0; path=%s", "aselectticket", pConfig->pCurrentApp->pcLocation);
                                                ap_table_add(headers_out, "Set-Cookie", pcCookie);
                                                pcCookie = ap_psprintf(pPool, "%s=; version=1; max-age=0; path=%s", "aselectuid", pConfig->pCurrentApp->pcLocation);
                                                ap_table_add(headers_out, "Set-Cookie", pcCookie);
                                                pcCookie = ap_psprintf(pPool, "%s=; version=1; max-age=0; path=%s", "aselectorganization", pConfig->pCurrentApp->pcLocation);
                                                ap_table_add(headers_out, "Set-Cookie", pcCookie);
                                                pcCookie = ap_psprintf(pPool, "%s=; version=1; max-age=0; path=%s", "aselectattributes", pConfig->pCurrentApp->pcLocation);
                                                ap_table_add(headers_out, "Set-Cookie", pcCookie);

                                                ap_send_http_header(pRequest);
                                                ap_rprintf(pRequest, ASELECT_FILTER_CLIENT_REDIRECT, 
                                                    pcASelectServerURL, pcASelectServerURL);

                                                iRet = DONE;
                                        }
                                        else
                                        {
                                            iError = ASELECT_FILTER_ERROR_NO_SUCH_COOKIE;
                                            TRACE1("aselect_filter_get_cookie(aselectserverurl) FAILED: %d", iError);
                                            ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, pRequest,
                                                ap_psprintf(pPool, "ASELECT_FILTER::aselect_filter_get_cookie(aselectserverurl) FAILED: %d", iError));
                                        }
                                    }
                                    else
                                    {
                                        TRACE1("aselect_filter_kill_ticket FAILED: %d", iError);
                                        ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, pRequest,
                                            ap_psprintf(pPool, "ASELECT_FILTER::aselect_filter_kill_ticket FAILED: %d", iError));
                                    }
                                }
                                else
                                {
                                    iError = ASELECT_FILTER_ERROR_INTERNAL;
                                    TRACE1("aselect_filter_kill_ticket FAILED: %d", iError);
                                    ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, pRequest,
                                        ap_psprintf(pPool, "ASELECT_FILTER::aselect_filter_kill_ticket FAILED: %d", iError));
                                }
                            }
                            else
                            {
                                //
                                // Could not find ticket to kill
                                //
                                iError = ASELECT_FILTER_ERROR_NO_SUCH_COOKIE;
                                TRACE1("aselect_filter_get_cookie(aselectticket) FAILED: %d", iError);
                                ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, pRequest,
                                    ap_psprintf(pPool, "ASELECT_FILTER::aselect_filter_get_cookie(aselectticket) FAILED: %d", iError));
                            }
                        }
                        else
                        {
                            //
                            // Nothing interesting in the request=param, so continue as normal
                            //
                            iRet = DECLINED;
                        }
                    }
                    else
                    {
                        //
                        // No arguments we are intereste in, so continue as normal
                        //
                        iRet = DECLINED;
                    }
                }
                else
                {
                    iRet = DECLINED;
                }

            break;

            case ASELECT_FILTER_ACTION_AUTH_USER:
            
                //
                // user does not have a valid CREDENTIALS and must be authenticated by the ASelect Server
                // Contact ASelect Agent to find the users ASelect Server
                //      

                TRACE("ASELECT_FILTER_ACTION_AUTH_USER");
                TRACE1("iRedirectMode: %d", pConfig->iRedirectMode);
                if (*pConfig->pCurrentApp->pcRedirectURL)
                {
                    TRACE1("Using fixed app_url: %s", pConfig->pCurrentApp->pcRedirectURL);
                    if (pRequest->args != NULL)
                    {
                      if (strchr(pConfig->pCurrentApp->pcRedirectURL, '?'))
                          pcAppUrl = ap_psprintf(pPool, "%s&%s", pConfig->pCurrentApp->pcRedirectURL,
                              pRequest->args);
                      else
                          pcAppUrl = ap_psprintf(pPool, "%s?%s", pConfig->pCurrentApp->pcRedirectURL,
                              pRequest->args);
                    }
                    else
                      pcAppUrl = pConfig->pCurrentApp->pcRedirectURL;
                }
                else
                {
                  if (pConfig->iRedirectMode == ASELECT_FILTER_REDIRECT_FULL)
                  {
                      if (pRequest->args != NULL)
                      {
                          pcUrl = ap_psprintf(pPool, "%s?%s", pRequest->uri, pRequest->args);
                          pcAppUrl = ap_construct_url(pPool, pcUrl, pRequest);
                      }
                      else
                          pcAppUrl = ap_construct_url(pPool, pRequest->uri, pRequest);
                  }
                  else
                      pcAppUrl = ap_construct_url(pPool, pConfig->pCurrentApp->pcLocation, pRequest);
                }
                
                TRACE1("app_url: %s", pcAppUrl);

                if ((pcResponseAU = aselect_filter_auth_user(pRequest, pPool, pConfig, pcAppUrl)))
                {
                    iError = aselect_filter_get_error(pPool, pcResponseAU);
                }

                if (iError == ASELECT_FILTER_ASAGENT_ERROR_OK)
                {
                    TRACE1("response: %s", pcResponseAU);
                    //
                    // build the redirection URL from the response
                    //
                    if ((pcRID = aselect_filter_get_param(pPool, pcResponseAU, "rid=", "&", TRUE)))
                    {
                        if ((pcASelectServer = aselect_filter_get_param(pPool, pcResponseAU, "a-select-server=", "&", TRUE)))
                        {
                            if ((pcASUrl = aselect_filter_get_param(pPool, pcResponseAU, "as_url=", "&", TRUE)))
                            {
                                iRet = aselect_filter_gen_top_redirect(pPool, pRequest, pcASUrl, pcASelectServer, pcRID);
                            }
                            else{
                                iError = ASELECT_FILTER_ERROR_AGENT_RESPONSE;
                            }
                        }
                        else{
                            iError = ASELECT_FILTER_ERROR_AGENT_RESPONSE;
                        }
                    }
                    else{
                        iError = ASELECT_FILTER_ERROR_AGENT_RESPONSE;
                    }
                }
                else
                {
                    TRACE1("aselect_filter_auth_user FAILED (%d)", iError);
                    ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_WARNING, pRequest,
                        ap_psprintf(pPool, "ASELECT_FILTER::aselect_filter_auth_user FAILED (%d)", iError));
                }
            
                break;

            case ASELECT_FILTER_ACTION_SET_TICKET:

                //
                // Generate & set A-Select cookies
                //

                TRACE3("ASELECT_FILTER_ACTION_SET_TICKET: %s - %s - %s", pcTicketOut, pcUIDOut, pcOrganizationOut);

                pcCookie = ap_psprintf(pPool, "%s=%s; version=1; path=%s", "aselectticket", 
                    pcTicketOut, pConfig->pCurrentApp->pcLocation);
                TRACE1("Set-Cookie: %s", pcCookie);
                ap_table_add(headers_out, "Set-Cookie", pcCookie); 
                pcCookie2 = ap_psprintf(pPool, "%s=%s; version=1; path=%s", "aselectuid", 
                    pcUIDOut, pConfig->pCurrentApp->pcLocation);
                TRACE1("Set-Cookie: %s", pcCookie2);
                ap_table_add(headers_out, "Set-Cookie", pcCookie2); 
                pcCookie3 = ap_psprintf(pPool, "%s=%s; version=1; path=%s", "aselectorganization", 
                    pcOrganizationOut, pConfig->pCurrentApp->pcLocation);
                TRACE1("Set-Cookie: %s", pcCookie3);
                ap_table_add(headers_out, "Set-Cookie", pcCookie3); 
                pcCookie4 = ap_psprintf(pPool, "%s=%s; version=1; path=%s", "aselectattributes", 
                    (pcAttributes == NULL) ? "" : pcAttributes, 
                    pConfig->pCurrentApp->pcLocation);
                TRACE1("Set-Cookie: %s", pcCookie4);
                ap_table_add(headers_out, "Set-Cookie", pcCookie4); 

                pcStrippedParams  = NULL;
                bFirstParam = TRUE;
                pcTmp = strtok(pRequest->args, "?&");
                while (pcTmp != NULL)
                {
                    if ((pcTmp2 = strstr(pcTmp, "aselect_credentials")) == NULL) 
                    {
	                    if ((pcTmp2 = strstr(pcTmp, "rid")) == NULL) 
	                    {
     	                    if ((pcTmp2 = strstr(pcTmp, "a-select-server")) == NULL) 
	                        {
    	                        if (bFirstParam)
    	                        {
    	                            pcStrippedParams = ap_psprintf(pPool, "?%s", pcTmp);
    	                            bFirstParam = FALSE;
    	                        }
    	                        else
    	                            pcStrippedParams = ap_psprintf(pPool, "%s&%s", pcStrippedParams, pcTmp);
    	                     }
	                    }
                    }
                    pcTmp = strtok(NULL, "?&");
                }
                if (pcStrippedParams != NULL)
                    pRequest->args = pcStrippedParams;
                else
                    pRequest->args = "";
                TRACE1("----->pRequest->args %s",pRequest->args);
                                                                         
                iRet = aselect_filter_gen_authcomplete_redirect(
                    pPool, pRequest, pConfig);
                break;

            case ASELECT_FILTER_ACTION_ACCESS_DENIED:
                TRACE("ACCESS_DENIED");

            default:

                //
                // Access is denied or unknown action
                // ACCESS DENIED
                //
            break;
        }
    }
    TRACE1("iError %ld", iError);

    if (iError != ASELECT_FILTER_ERROR_OK)
        if (aselect_filter_gen_error_page(pPool, pRequest, iError, pConfig->pcErrorTemplate) == ASELECT_FILTER_ERROR_OK)
            iRet = DONE;

    TRACE("destroying memory pool");

    //
    // Cleanup
    //
    ap_destroy_pool(pPool);
    TRACE1("returning %d", iRet);

    return iRet;
}

//
// Use to create the per server configuration data
//
static void*
aselect_filter_create_config(pool *pPool, server_rec *pServer)
{
    PASELECT_FILTER_CONFIG  pConfig = NULL;

#ifdef ASELECT_FILTER_TRACE
    aselect_filter_trace("aselect_filter_create_config");
#endif
    if ((pConfig = (PASELECT_FILTER_CONFIG) ap_palloc(pPool, sizeof(ASELECT_FILTER_CONFIG))))
    {
        memset(pConfig, 0, sizeof(ASELECT_FILTER_CONFIG));
    }
    else
    {
#ifdef ASELECT_FILTER_TRACE
            aselect_filter_trace("aselect_filter_create_config::ERROR:: could not allocate memory for pConfig");
#endif
    }

    return pConfig;
}

static const char *
aselect_filter_set_agent_address(cmd_parms *parms, void *mconfig, const char *arg)
{
    PASELECT_FILTER_CONFIG  pConfig = (PASELECT_FILTER_CONFIG) ap_get_module_config(parms->server->module_config, &aselect_filter_module);

    if (pConfig)
    {
        if ((pConfig->pcASAIP = ap_pstrdup(parms->pool, arg)))
        {
#ifdef ASELECT_FILTER_TRACE
            aselect_filter_trace("aselect_filter_set_agent_address:: ip: %s", pConfig->pcASAIP);
#endif
        }
        else
        {
            return "A-Select ERROR: Internal error when setting A-Select IP";
        }
    }
    else
    {
        return "A-Select ERROR: Internal error when setting A-Select IP";
    }

    return NULL;
}

static const char *
aselect_filter_set_agent_port(cmd_parms *parms, void *mconfig, const char *arg)
{
    PASELECT_FILTER_CONFIG  pConfig = (PASELECT_FILTER_CONFIG) ap_get_module_config(parms->server->module_config, &aselect_filter_module);
    char                    *pcASAPort;

    if (pConfig)
    {
        if ((pcASAPort = ap_pstrdup(parms->pool, arg)))
        {
#ifdef ASELECT_FILTER_TRACE
            aselect_filter_trace("aselect_filter_set_agent_port:: port: %s", pcASAPort);
#endif
            pConfig->iASAPort = atoi(pcASAPort);
        }
        else
        {
            return "A-Select ERROR: Internal error when setting A-Select port";
        }
    }
    else
    {
        return "A-Select ERROR: Internal error when setting A-Select port";
    }

    return NULL;
}

static const char *
aselect_filter_add_authz_rule(cmd_parms *parms, void *mconfig, const char *arg1, const char *arg2, const char *arg3)
{
    int i;
    PASELECT_APPLICATION pApp;
    PASELECT_FILTER_CONFIG  pConfig = (PASELECT_FILTER_CONFIG) 
        ap_get_module_config(parms->server->module_config, &aselect_filter_module);

    if (pConfig)
    {
        for (i=0; i<pConfig->iAppCount; i++)
        {
            if (strcmp(pConfig->pApplications[i].pcAppId, arg1) == 0)
                break;
        }
        if (i >= pConfig->iAppCount)
        {
            TRACE1("aselect_filter_add_secure_app: Unknown app_id \"%s\"", arg1);
            return "A-Select ERROR: Unknown application ID in authorization rule";
        }
        pApp = &pConfig->pApplications[i];
        if (pApp->iRuleCount == ASELECT_FILTER_MAX_RULES_PER_APP)
            return "A-Select ERROR: Maximum amount of authorization rules per application exceeded";
        pApp->pTargets[pApp->iRuleCount] = aselect_filter_url_encode(parms->pool, arg2);
        pApp->pConditions[pApp->iRuleCount] = aselect_filter_url_encode(parms->pool, arg3);
        TRACE3("aselect_filter_add_secure_app: app=%s, target=%s, added condition \"%s\"",
            arg1, arg2, arg3);
        ++(pApp->iRuleCount);
    }
    else
    {
        return "A-Select ERROR: Internal error: missing configuration object";
    }

    return NULL;
}

static const char *
aselect_filter_add_secure_app(cmd_parms *parms, void *mconfig, const char *arg1, const char *arg2, const char *arg3)
{
    static char *_empty = "";
    const char *pcTok, *pcEnd;
    PASELECT_APPLICATION pApp;
    PASELECT_FILTER_CONFIG  pConfig = (PASELECT_FILTER_CONFIG) 
        ap_get_module_config(parms->server->module_config, &aselect_filter_module);

    if (pConfig)
    {
        if (pConfig->iAppCount < ASELECT_FILTER_MAX_APP)
        {
            pApp = &pConfig->pApplications[pConfig->iAppCount];
            memset(pApp, 0, sizeof(ASELECT_APPLICATION));
            pApp->bEnabled = 1;
            pApp->pcUid = _empty;
            pApp->pcLanguage = _empty;
            pApp->pcCountry = _empty;
            pApp->pcExtra = _empty;
            pApp->pcRemoteOrg = _empty;
            pApp->pcRedirectURL = _empty;
            if ( (pApp->pcLocation = ap_pstrdup(parms->pool, arg1)) &&
                 (pApp->pcAppId = ap_pstrdup(parms->pool, arg2)) )
            {
                if (strcmp(arg3, "none") != 0 &&
                    strcmp(arg3, "default") != 0)
                {
                    pcTok = arg3;
                    while (*pcTok)
                    {
                        TRACE1("Parsing application options token %s", pcTok);
                        if (strncmp(pcTok, "forced-logon", 12) == 0 ||
                            strncmp(pcTok, "forced_logon", 12) == 0)
                        {
                            pcTok += 12;
                            pApp->bForcedLogon = 1;
                        }
                        else
                        if (strncmp(pcTok, "disabled", 8) == 0)
                        {
                            pcTok += 8;
                            pApp->bEnabled = 0;
                        }
                        else
                        if (strncmp(pcTok, "uid=", 4) == 0)
                        {
                            pcTok += 4;
                            if ((pcEnd = strchr(pcTok, ',')))
                                pApp->pcUid = ap_pstrndup(parms->pool, pcTok, pcEnd-pcTok);
                            else
                                pApp->pcUid = ap_pstrdup(parms->pool, pcTok);
                            pcTok += strlen(pApp->pcUid);
                            pApp->pcUid = ap_psprintf(parms->pool, "&uid=%s",
                                aselect_filter_url_encode(parms->pool, pApp->pcUid));
                        }
                        else
                        if (strncmp(pcTok, "language=", 9) == 0)
                        {
                            pcTok += 9;
                            if ((pcEnd = strchr(pcTok, ',')))
                                pApp->pcLanguage = ap_pstrndup(parms->pool, pcTok, pcEnd-pcTok);
                            else
                                pApp->pcLanguage = ap_pstrdup(parms->pool, pcTok);
                            pcTok += strlen(pApp->pcLanguage);
                            pApp->pcLanguage = ap_psprintf(parms->pool, "&language=%s",
                                aselect_filter_url_encode(parms->pool, pApp->pcLanguage));
                        }
                        else
                        if (strncmp(pcTok, "country=", 8) == 0)
                        {
                            pcTok += 8;
                            if ((pcEnd = strchr(pcTok, ',')))
                                pApp->pcCountry = ap_pstrndup(parms->pool, pcTok, pcEnd-pcTok);
                            else
                                pApp->pcCountry = ap_pstrdup(parms->pool, pcTok);
                            pcTok += strlen(pApp->pcCountry);
                            pApp->pcCountry = ap_psprintf(parms->pool, "&country=%s",
                                aselect_filter_url_encode(parms->pool, pApp->pcCountry));
                        }
                        else
                        if (strncmp(pcTok, "remote_organization=", 20) == 0 ||
                            strncmp(pcTok, "remote-organization=", 20) == 0)
                        {
                            pcTok += 20;
                            if ((pcEnd = strchr(pcTok, ',')))
                                pApp->pcRemoteOrg = ap_pstrndup(parms->pool, pcTok, pcEnd-pcTok);
                            else
                                pApp->pcRemoteOrg = ap_pstrdup(parms->pool, pcTok);
                            pcTok += strlen(pApp->pcRemoteOrg);
                            pApp->pcRemoteOrg = ap_psprintf(parms->pool, "&remote_organization=%s",
                                aselect_filter_url_encode(parms->pool, pApp->pcRemoteOrg));
                        }
                        else
      if (strncmp(pcTok, "url=", 4) == 0)
      {
          pcTok += 4;
                            if ((pcEnd = strchr(pcTok, ',')))
                                pApp->pcRedirectURL = ap_pstrndup(parms->pool, pcTok, pcEnd-pcTok);
                            else
                                pApp->pcRedirectURL = ap_pstrdup(parms->pool, pcTok);
          pcTok += strlen(pApp->pcRedirectURL);
      }
                        else    
                        if (strncmp(pcTok, "extra_parameters=", 17) == 0 ||
                            strncmp(pcTok, "extra-parameters=", 17) == 0)
                        {
                            pcTok += 17;
                            if ((pcEnd = strchr(pcTok, ',')))
                                pApp->pcExtra = ap_pstrndup(parms->pool, pcTok, pcEnd-pcTok);
                            else
                                pApp->pcExtra = ap_pstrdup(parms->pool, pcTok);
                            pcTok += strlen(pApp->pcExtra);
                            pApp->pcExtra = ap_psprintf(parms->pool, "&%s",
                                pApp->pcExtra);
                        }
                        else
                        {
                            // Unknown option
                            return ap_psprintf(parms->pool,
                                "A-Select ERROR: Unknown option in application %s near \"%s\"", 
                                pApp->pcAppId, pcTok);
                        }
                        if (*pcTok)
                        {
                            if (*pcTok != ',')
                                return ap_psprintf(parms->pool,
                                    "A-Select ERROR: Error in application %s options, near \"%s\"", 
                                    pApp->pcAppId, pcTok);
                            ++pcTok;
                        }                        
                    }
                }
                // Success!
                pConfig->iAppCount++;
            }
            else
            {
                return "A-Select ERROR: Out of memory while adding applications";
            }
        }
        else
        {
            return "A-Select ERROR: Reached max possible application IDs";
        }
    }
    else
    {
        return "A-Select ERROR: Internal error: missing configuration object";
    }

    return NULL;
}

static const char *aselect_filter_set_redirection_mode(cmd_parms *parms, void *mconfig, const char *arg)
{
    PASELECT_FILTER_CONFIG  pConfig = (PASELECT_FILTER_CONFIG) ap_get_module_config(parms->server->module_config, &aselect_filter_module);
    char *pcMode;
    if (pConfig){
        if ((pcMode = ap_pstrdup(parms->pool, arg))){
            if (strcasecmp(pcMode, "app") == 0){
                                pConfig->iRedirectMode = ASELECT_FILTER_REDIRECT_TO_APP;
                        }
                        else if (strcasecmp(pcMode, "full") == 0){
                                pConfig->iRedirectMode = ASELECT_FILTER_REDIRECT_FULL;
                        }
                        else{
                return "A-Select ERROR: Invalid argument to aselect_filter_set_redirect_mode";
            }
#ifdef ASELECT_FILTER_TRACE
            aselect_filter_trace("aselect_filter_set_redirect_mode:: %d (%s)", 
                pConfig->iRedirectMode,
                (pConfig->iRedirectMode == ASELECT_FILTER_REDIRECT_TO_APP)
                ? "app" : "full");
#endif
        }
        else{
            return "A-Select ERROR: Internal error while setting redirect_mode";
        }
    }
    return NULL;
}


static const char *
aselect_filter_set_html_error_template(cmd_parms *parms, void *mconfig, const char *arg)
{
        PASELECT_FILTER_CONFIG  pConfig = (PASELECT_FILTER_CONFIG) ap_get_module_config(parms->server->module_config, &aselect_filter_module);
        FILE    *file;
        fpos_t  fpos;
        int     ipos;
        char    cLine[100];
        char    *pcFile;

        if (pConfig)
        {
                if ((pcFile = ap_pstrdup(parms->pool, arg)))
                {
#ifdef ASELECT_FILTER_TRACE
                        aselect_filter_trace("aselect_filter_set_html_error_template:: %s", pcFile);
#endif
                        if ((file = fopen(pcFile, "r")))
                        {
                                if (fseek(file, 0, SEEK_END) == 0)
                                {
                                        if (fgetpos(file, &fpos) == 0)
                                        {
                                                memcpy(&ipos, &fpos, sizeof(ipos));

#ifdef ASELECT_FILTER_TRACE
                                                aselect_filter_trace("aselect_filter_set_html_error_template:: size of template: %d", ipos);
#endif

                                                if (fseek(file, 0, SEEK_SET) == 0)
                                                {
                                                        if ((pConfig->pcErrorTemplate = (char*) ap_palloc(parms->pool, ipos)))
                                                        {
                                                                memset(pConfig->pcErrorTemplate, 0, sizeof(ipos));

                                                                while (fgets(cLine, sizeof(cLine), file) != NULL)
                                                                        strcat(pConfig->pcErrorTemplate, cLine);
                                                        }
                                                        else
                                                        {
#ifdef ASELECT_FILTER_TRACE
                                                                aselect_filter_trace("aselect_filter_set_html_error_template:: failed to allocate mem for pConfig->pcErrorTemplate");
#endif
                                                                fclose(file);

                                                                return "A-Select ERROR: Internal error when setting html_error_template file";
                                                        }
                                                }
                                                else
                                                {
#ifdef ASELECT_FILTER_TRACE
                                                        aselect_filter_trace("aselect_filter_set_html_error_template:: fseek(SEEK_SET) failed");
#endif
                                                        fclose(file);

                                                        return "A-Select ERROR: Internal error when setting html_error_template file";
                                                }
                                        }
                                        else
                                        {
                                                fclose(file);

#ifdef ASELECT_FILTER_TRACE
                                                aselect_filter_trace("aselect_filter_set_html_error_template:: fgetpos failed");
#endif

                                                return "A-Select ERROR: Internal error when setting html_error_template file";
                                        }
                                }
                                else
                                {
#ifdef ASELECT_FILTER_TRACE
                                        aselect_filter_trace("aselect_filter_set_html_error_template:: fseek(SEEK_END) failed");
#endif
                                        fclose(file);

                                        return "A-Select ERROR: Internal error when setting html_error_template file";
                                }

                                fclose(file);
                        }
                        else
                        {
#ifdef ASELECT_FILTER_TRACE
                                aselect_filter_trace("aselect_filter_set_html_error_template:: fopen failed");
#endif
                                return "A-Select ERROR: Could not open html_error_template";
                        }
                }
                else
                {
                        return "A-Select ERROR: Internal error when setting html_error_template file";
                }
        }
        else
        {
                return "A-Select ERROR: Internal error when setting html_error_template file";
        }

        return NULL;
}


static const char *
aselect_filter_set_use_aselect_bar(cmd_parms *parms, void *mconfig, const char *arg)
{
        PASELECT_FILTER_CONFIG  pConfig = (PASELECT_FILTER_CONFIG) ap_get_module_config(parms->server->module_config, &aselect_filter_module);
        char                    *pcUseASelectBar;

        if (pConfig)
        {
                if ((pcUseASelectBar = ap_pstrdup(parms->pool, arg)))
                {
#ifdef ASELECT_FILTER_TRACE
                        aselect_filter_trace("aselect_filter_set_use_aselect_bar:: %s", pcUseASelectBar);
#endif
                        pConfig->bUseASelectBar = FALSE;

                        if (strcasecmp(pcUseASelectBar, "1") == 0)
                                pConfig->bUseASelectBar = TRUE;
                }
                else
                {
                        return "A-Select ERROR: Internal error when setting use_aselect_bar";
                }
        }
        else
        {
                return "A-Select ERROR: Internal error when setting use_aselect_bar";
        }

        return NULL;
}


#ifdef APACHE_13_ASELECT_FILTER
//
// Registered handlers that can be called for this module
//
static handler_rec aselect_filter_handlers[] =
{
    { "aselect_filter_handler", aselect_filter_handler },
    {NULL}
};

//
// Registered cmds to call from httpd.conf
//
static const command_rec
aselect_filter_cmds[] = 
{
    { "aselect_filter_set_agent_address", 
        aselect_filter_set_agent_address, 
        NULL,
        RSRC_CONF,
        TAKE1,
        "Usage aselect_filter_set_agent_address <ip or dns name of A-Select Agent>, example: aselect_filter_set_agent_address \"localhost\"" },
    { "aselect_filter_set_agent_port", 
        aselect_filter_set_agent_port, 
        NULL,
        RSRC_CONF,
        TAKE1,
        "Usage aselect_filter_set_agent_port <port of A-Select Agent>, example: aselect_filter_set_agent_port \"1495\"" },
    { "aselect_filter_add_secure_app",
        aselect_filter_add_secure_app,
        NULL,
        RSRC_CONF,
        TAKE3,
        "Usage aselect_filter_add_secure_app <location> <application id> <flags>, example: aselect_filter_add_secure_app \"///secure///\" \"app1\" \"forced-logon\"" },
    { "aselect_filter_add_authz_rule",
        aselect_filter_add_authz_rule,
        NULL,
        RSRC_CONF,
        TAKE3,
        "Usage aselect_filter_add_authz_rule <application id> <target uri> <condition>, example: aselect_filter_add_authz_rule \"app1\" \"*\" \"role=student\"" },
    { "aselect_filter_set_html_error_template",
        aselect_filter_set_html_error_template,
        NULL,
        RSRC_CONF,
        TAKE1,
        "Usage aselect_filter_set_html_error_template <path to template html page>, example: aselect_filter_set_html_error_template \"///usr//local//apache//aselect//error.html\"" },
    { "aselect_filter_set_use_aselect_bar",
        aselect_filter_set_use_aselect_bar,
        NULL,
        RSRC_CONF,
        TAKE1,
        "Usage aselect_filter_set_use_aselect_bar <on or off>, example: aselect_filter_set_use_aselect_bar on" },
    { "aselect_filter_set_redirect_mode",
        aselect_filter_set_redirection_mode,
        NULL,
        RSRC_CONF,
        TAKE1,
        "Usage aselect_filter_redirect_mode <app | full>, example: aselect_filter_redirect_mode \"app\"" },
    { NULL }
};

//
// Main export structure containing all the entry points for this module
//
module MODULE_VAR_EXPORT aselect_filter_module =
{
    STANDARD_MODULE_STUFF,
    aselect_filter_init,    // module initializer
    NULL,                   // per-dir config creator
    NULL,                   // dir config merger
    aselect_filter_create_config,   // server config creator
    NULL,                   // server config merger
    aselect_filter_cmds,    // command table
    NULL,                   // [9] content handlers
    NULL,                   // [2] URI-to-filename translation
    NULL,                   // [5] check/validate user_id
    NULL,                   // [6] check user_id is valid here
    aselect_filter_handler, // [4] check access by host address
    NULL,                   // [7] MIME type checker/setter
    NULL,                   // [8] fixups
    NULL,                   // [10] logger
    NULL,                   // [3] header parser
    NULL,                   // process initialization
    NULL,                   // process exit/cleanup
    NULL                    // [1] post read_request handling
};

#else // #ifdef APACHE_13_ASELECT_FILTER

//
// Registered cmds to call from httpd.conf
//
static const command_rec
aselect_filter_cmds[] = 
{
    AP_INIT_TAKE1( "aselect_filter_set_agent_address", 
        aselect_filter_set_agent_address, 
        NULL,
        RSRC_CONF,
        "Usage aselect_filter_set_agent_address <ip or dns name of A-Select Agent>, example: aselect_filter_set_agent_address \"localhost\"" ),
    AP_INIT_TAKE1( "aselect_filter_set_agent_port", 
        aselect_filter_set_agent_port, 
        NULL,
        RSRC_CONF,
        "Usage aselect_filter_set_agent_port <port of A-Select Agent>, example: aselect_filter_set_agent_port \"1495\"" ),
    AP_INIT_TAKE3( "aselect_filter_add_secure_app",
        aselect_filter_add_secure_app,
        NULL,
        RSRC_CONF,
        "Usage aselect_filter_add_secure_app <location> <application id> <flags>, example: aselect_filter_add_secure_app \"///secure///\" \"app1\" \"forced-logon\"" ),
    AP_INIT_TAKE3( "aselect_filter_add_authz_rule",
        aselect_filter_add_authz_rule,
        NULL,
        RSRC_CONF,
        "Usage aselect_filter_add_authz_rule <application id> <target uri> <condition>, example: aselect_filter_add_authz_rule \"app1\" \"*\" \"role=student\"" ),
    AP_INIT_TAKE1( "aselect_filter_set_html_error_template",
        aselect_filter_set_html_error_template,
        NULL,
        RSRC_CONF,
        "Usage aselect_filter_set_html_error_template <path to template html page>, example: aselect_filter_set_html_error_template \"///usr//local//apache//aselect//error.html\"" ),
    AP_INIT_TAKE1( "aselect_filter_set_use_aselect_bar",
        aselect_filter_set_use_aselect_bar,
        NULL,
        RSRC_CONF,
        "Usage aselect_filter_set_use_aselect_bar <on or off>, example: aselect_filter_set_use_aselect_bar on" ),
    AP_INIT_TAKE1("aselect_filter_set_redirect_mode",
        aselect_filter_set_redirection_mode,
        NULL,
        RSRC_CONF,
        "Usage aselect_filter_redirect_mode <app | full>, example: aselect_filter_redirect_mode \"app\""),
    { NULL }
};

void *
aselect_filter_create_server_config( apr_pool_t *pPool, server_rec *pServer )
{
    PASELECT_FILTER_CONFIG  pConfig = NULL;

#ifdef ASELECT_FILTER_TRACE
    aselect_filter_trace( "aselect_filter_create_config" );
#endif
    if( ( pConfig = ( PASELECT_FILTER_CONFIG ) apr_palloc( pPool, sizeof( ASELECT_FILTER_CONFIG ) ) ) )
    {
        memset( pConfig, 0, sizeof( ASELECT_FILTER_CONFIG ) );
    }
    else
    {
#ifdef ASELECT_FILTER_TRACE
        aselect_filter_trace( "aselect_filter_create_config::ERROR:: could not allocate memory for pConfig" );
#endif
        pConfig = NULL;
    }

    return pConfig;
}

void
aselect_filter_register_hooks( apr_pool_t *p )
{

#ifdef ASELECT_FILTER_TRACE
    aselect_filter_trace( "aselect_filter_register_hooks" );
#endif
    ap_hook_post_config( aselect_filter_init, NULL, NULL, APR_HOOK_MIDDLE );
    ap_hook_access_checker( aselect_filter_handler, NULL, NULL, APR_HOOK_MIDDLE );
}

module AP_MODULE_DECLARE_DATA
aselect_filter_module =
{
    STANDARD20_MODULE_STUFF,
    NULL,                   // per-directory config creator
    NULL,                   // dir config merger
    aselect_filter_create_server_config,    // server config creator
    NULL,                   // server config merger
    aselect_filter_cmds,            // command table
    aselect_filter_register_hooks,      //  set up other request processing hooks
};

#endif // #ifdef APACHE_13_ASELECT_FILTER

