/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */

package org.aselect.server.cross.selectorhandler;

import java.io.IOException;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Hashtable;
import java.util.logging.Level;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletResponse;

import org.aselect.server.config.ASelectConfigManager;
import org.aselect.server.cross.CrossASelectManager;
import org.aselect.server.cross.ISelectorHandler;
import org.aselect.server.elo.ELO;
import org.aselect.server.elo.ELOFactory;
import org.aselect.server.elo.IELOStorage;
import org.aselect.server.log.ASelectSystemLogger;
import org.aselect.server.sam.ASelectSAMAgent;
import org.aselect.server.session.SessionManager;
import org.aselect.system.error.Errors;
import org.aselect.system.exception.ASelectException;
import org.aselect.system.sam.agent.SAMResource;

/**
 * Cross A-Select Selector handler that redirects to the WAYF page.
 * <br><br>
 * <b>Description:</b>
 * The user will be redirected to a configured WAYF page location.
 * <br>
 * This handler also supports ELO servers:
 * <ul>
 *  <li>Tries to retrieve the ELO url from a notification cookie (elo)</li>
 *  <li>If the 'remote_idp' parameter returned by the WAYF page contains an 
 *  ELO ID it will redirect to that ELO</li>
 * </ul> 
 * <br><br>
 * @author Alfa & Ariss
 */
public class RedirectSelectorHandler implements ISelectorHandler
{
    private final static String MODULE = "RedirectSelectorHandler";
    private final static String COOKIE_NOTIFICATION = "elo";
    private final static String SESSION_ELO_ID = "elo_id";
    
    private ASelectConfigManager _configManager;
    private ASelectSystemLogger _systemLogger;
    private SessionManager _sessionManager;
    private ASelectSAMAgent _samAgent;
    private IELOStorage _eloStorage;
    
    private SAMResource _samActiveResource;
    private String _sTargetResourceGroup;
    private String _sTargetUrl;
    private Hashtable _htRemoteServers;
    

    /**
     * Initialization of the Handler.
     * <br>
     * @see org.aselect.server.cross.ISelectorHandler#init(java.lang.Object)
     */
    public void init(Object oHandlerConfig) throws ASelectException
    {
        String sMethod = "init()";
        try
        {
            _systemLogger = ASelectSystemLogger.getHandle();
            _configManager = ASelectConfigManager.getHandle();
            _sessionManager = SessionManager.getHandle();
            _samAgent = ASelectSAMAgent.getHandle();
            _eloStorage = ELOFactory.getHandle().getEloStore();
            
            CrossASelectManager crossASelectManager = CrossASelectManager.getHandle();
            _htRemoteServers = crossASelectManager.getRemoteServers();
            
            Object oTarget = null;
            try
            {
                oTarget = _configManager.getSection(oHandlerConfig, "target");
            }
            catch (ASelectException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, "No 'target' section found in configuration");
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            try
            {
                _sTargetResourceGroup = _configManager.getParam(oTarget, "resourcegroup");
            }
            catch (ASelectException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "No 'resourcegroup' item found in 'target' section in configuration");
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            getTargetUrl();//for test
        }
        catch (ASelectException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, 
                "Could not initialize the default selector handler", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR, e);
        }
    }

    /**
     * Retrieves the remote A-Select Server id. 
     * <br>
     * <ul>
     *  <li>The server id can be in the request as 'remote_idp'</li>
     *  <li>If no server id found in the request it checks if a notification cookie is set containing the id</li>
     *  <li>If no server id found in the cookie it redirects the user to a remote WAYF page</li>
     * </ul>
     * <br>
     * @see org.aselect.server.cross.ISelectorHandler#getRemoteServerId(java.util.Hashtable, javax.servlet.http.HttpServletResponse, java.io.PrintWriter)
     */
    public Hashtable getRemoteServerId(Hashtable htServiceRequest,
        HttpServletResponse servletResponse, PrintWriter pwOut)
    		throws ASelectException
    {
        String sMethod = "getRemoteServerId()";
        Hashtable htResult = null;
        try
        {
            String sRemoteId = (String)htServiceRequest.get("remote_idp");
            if (sRemoteId != null)
                htResult = resolveIDP(servletResponse, htServiceRequest, sRemoteId);
            else
            {
                ELO oELO = getELOFromCookie(htServiceRequest);
                if (oELO != null)
                {//ELO has the function of an SP
                    String sRid = (String)htServiceRequest.get("rid");
                    if (sRid == null)
                    {
                        _systemLogger.log(Level.FINE, MODULE, sMethod, "No required 'rid' parameter in request");
                        throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
                    }
                    
                    Hashtable htSession = _sessionManager.getSessionContext(sRid);
                    if (htSession == null)
                    {
                        _systemLogger.log(Level.FINE, MODULE, sMethod, 
                            "Session expired or no session found with rid: " + sRid);
                        throw new ASelectException(Errors.ERROR_ASELECT_SERVER_SESSION_EXPIRED);
                    }
                    
                    //update session with ELO id, needed by the cookiemonster
                    htSession.put(SESSION_ELO_ID, oELO.getID());
                    
                    _sessionManager.updateSession(sRid, htSession);
                    
                    redirectToELO(servletResponse, sRid, oELO);
                }
                else
                    redirectToWAYF(servletResponse, htServiceRequest);
            }
        }
        catch (ASelectException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, 
                "Could not resolve remote server", e);
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
        }
        return htResult;
    }
    
    private ELO getELOFromCookie(Hashtable htServiceRequest) throws ASelectException
    {
        String sMethod = "getELOFromCookie()";
        ELO oELO = null;
        String sCookie = getNotificationCookie(htServiceRequest);
        if (sCookie != null)
        {
            try
            {
                new URL(sCookie);
            }
            catch (MalformedURLException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "Cookie value is not a URL: " + sCookie);
                throw new ASelectException(Errors.ERROR_ASELECT_CONFIG_ERROR);
            }
            
            oELO = _eloStorage.getEloByURL(sCookie);
            if (oELO == null)
            {
                _systemLogger.log(Level.FINE, MODULE, sMethod, 
                    "Cookie value contains a non existing ELO URL: " + sCookie);
            }
        }
        
        return oELO;
    }

    private void redirectToWAYF(HttpServletResponse response, Hashtable htServiceRequest)
        throws ASelectException
    {
        String sMethod = "redirectToWAYF()";
        StringBuffer sbRedirect = null;
        try
        {
            String sRid = (String)htServiceRequest.get("rid");
            if (sRid == null)
            {
                _systemLogger.log(Level.FINE, MODULE, sMethod, "No required 'rid' parameter in request");
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
            }
            
            Hashtable htSession = _sessionManager.getSessionContext(sRid);
            if (htSession == null)
            {
                _systemLogger.log(Level.FINE, MODULE, sMethod, 
                    "Session expired or no session found with rid: " + sRid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_SESSION_EXPIRED);
            }
            
            String sRequestor = (String)htSession.get("app_id");
            if (sRequestor == null)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "No 'app_id' found in session with rid: " + sRid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
            }
            
            String sTargetUrl = getTargetUrl();
            
            sbRedirect = new StringBuffer(sTargetUrl);
            
            if (sTargetUrl.indexOf('?') == -1)
                sbRedirect.append("?");
            else
                sbRedirect.append("&");
            
            sbRedirect.append("rid=").append(sRid);
            sbRedirect.append("&requestor=").append(sRequestor);
                        
            _systemLogger.log(Level.FINER, MODULE, sMethod,
                    "Performing redirect: " + sbRedirect.toString());
            try
            {
                response.sendRedirect(sbRedirect.toString());
            }
            catch (IOException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "Could not send redirect: " + sbRedirect.toString(), e);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
            }
        }
        catch (ASelectException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, 
                "Could not resolve remote server id", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
    }
    
    private void redirectToELO(HttpServletResponse response, 
        String sRid, ELO oELO) throws ASelectException
    {
        String sMethod = "redirectToELO()";
        StringBuffer sbRedirect = null;
        try
        {
            String sELOUrl = oELO.getURL();
            
            sbRedirect = new StringBuffer(sELOUrl);
            if (sELOUrl.indexOf('?') == -1)
                sbRedirect.append("?");
            else
                sbRedirect.append("&");
            sbRedirect.append("rid=").append(sRid);
                        
            _systemLogger.log(Level.FINER, MODULE, sMethod,
                    "Performing redirect: " + sbRedirect.toString());
            try
            {
                response.sendRedirect(sbRedirect.toString());
            }
            catch (IOException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "Could not send redirect: " + sbRedirect.toString(), e);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
            }
        }
        catch (ASelectException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, 
                "Could not resolve remote server id", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
    }
    
    /*
     * verify response from wayf
     */
    private Hashtable resolveIDP(HttpServletResponse response, 
        Hashtable htServiceRequest, String sRemoteId) 
        throws ASelectException
    {
        String sMethod = "resolveIDP()";
        Hashtable htResult = null;
        try
        {
            if (sRemoteId.equalsIgnoreCase(""))
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "Empty 'remote_idp' in response");
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
            }
            
            if (_htRemoteServers.containsKey(sRemoteId))
            {
                htResult = new Hashtable();
                htResult.put("organization_id", sRemoteId);
            }
            else
            {
                ELO oELO = _eloStorage.getEloByID(sRemoteId);
                if (oELO == null)
                {
                    _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                        "Value 'remote_idp' in response is not an ELO id: " + sRemoteId);
                    throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
                }
                //ELO has the function of an IDP
                
                String sRid = (String)htServiceRequest.get("rid");
                if (sRid == null)
                {
                    _systemLogger.log(Level.FINE, MODULE, sMethod, "No required 'rid' parameter in request");
                    throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
                }
                
                Hashtable htSession = _sessionManager.getSessionContext(sRid);
                if (htSession == null)
                {
                    _systemLogger.log(Level.FINE, MODULE, sMethod, 
                        "Session expired or no session found with rid: " + sRid);
                    throw new ASelectException(Errors.ERROR_ASELECT_SERVER_SESSION_EXPIRED);
                }
                
                //update session with ELO id, needed by the cookiemonster
                htSession.put(SESSION_ELO_ID, oELO.getID());
                
                _sessionManager.updateSession(sRid, htSession);
                
                redirectToELO(response, sRid, oELO);
            }
        }
        catch (ASelectException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, 
                "Could not resolve remote server id", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
        return htResult;
    }

    private String getNotificationCookie(Hashtable htServiceRequest)
    {
        String sMethod = "getNotificationCookie()";
        String sValue = null;
        
        Cookie[] aCookies = (Cookie[])htServiceRequest.get("aselect_cookies");
        if (aCookies == null)
        {
            _systemLogger.log(Level.FINER, MODULE, sMethod, 
                "No cookies found in request");
        }
        else
        {
            for (int i = 0; i < aCookies.length; i++)
            {
                Cookie oCookie = aCookies[i];
                if (COOKIE_NOTIFICATION.equalsIgnoreCase(oCookie.getName()))
                {
                    sValue = oCookie.getValue();
                    break;
                }
            }
        }
        
        return sValue;
    }
    
    private String getTargetUrl() throws ASelectException
    {
        String sMethod = "getTargetUrl()";
        try
        {
            if (_sTargetUrl == null || _samActiveResource == null || !_samActiveResource.live())
            {
                _samActiveResource = _samAgent.getActiveResource(_sTargetResourceGroup);
                
                Object oResource = _samActiveResource.getAttributes();
                
                try
                {
                    _sTargetUrl = _configManager.getParam(oResource, "url");
                }
                catch(ASelectException e)
                {
                    _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                        "No 'url' configuration item found in resourcegroup with id: " + _sTargetResourceGroup);
                    throw new ASelectException(Errors.ERROR_ASELECT_CONFIG_ERROR);
                }
                
                try
                {
                    new URL(_sTargetUrl);
                }
                catch(MalformedURLException e)
                {
                    _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                        "Configured 'url' item in resourcegroup is not a valid URL: " + _sTargetUrl, e);
                    throw new ASelectException(Errors.ERROR_ASELECT_CONFIG_ERROR);
                }
            }
        }
        catch (ASelectException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, 
                "Could not retrieve target url from resourcegroup: " + 
                    _sTargetResourceGroup, e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
        return _sTargetUrl;
    }
    
}