/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */
package org.aselect.server.elo.handler;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;

import org.aselect.server.elo.ELO;
import org.aselect.server.elo.IELOStorage;
import org.aselect.system.configmanager.ConfigManager;
import org.aselect.system.error.Errors;
import org.aselect.system.exception.ASelectConfigException;
import org.aselect.system.exception.ASelectException;
import org.aselect.system.exception.ASelectStorageException;
import org.aselect.system.logging.SystemLogger;

/**
 * The Elo store that retrieves its information from the A-Select configuration file.
 * <br><br>
 * <b>Concurrency issues:</b> <br> - <br>
 * @author Alfa & Ariss
 */
public class ConfigStorage implements IELOStorage
{
    private static final String MODULE = "ConfigStorage";
    
    private Map _mELOsIDKeys = null;
    private Map _mELOsURLKeys = null;
    private SystemLogger _systemLogger;
    
    /**
     * Default constructor.
     * <br><br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     */
    public ConfigStorage()
    {
        _mELOsIDKeys = new HashMap();
        _mELOsURLKeys = new HashMap();
    }
    
    /**
     * Does not really do anything.
     * <br><br>
     * @see org.aselect.server.elo.IELOStorage#destroy()
     */
    public void destroy() throws ASelectException
    {
        //does not have to do anything
    }

    /**
     * returns an Elo object associated with the given ID.
     * <br><br>
     * @see org.aselect.server.elo.IELOStorage#getEloByID(java.lang.String)
     */
    public ELO getEloByID(String sId) throws ASelectStorageException
    {
        if (_mELOsIDKeys == null) return null;
        return (ELO) _mELOsIDKeys.get(sId);
    }
    

    /**
     * returns an Elo object associated with the given URL.
     * <br><br>
     * @see org.aselect.server.elo.IELOStorage#getEloByURL(java.lang.String)
     */
    public ELO getEloByURL(String sURL) throws ASelectStorageException
    {
        if (_mELOsURLKeys == null) return null;
        return (ELO) _mELOsURLKeys.get(sURL);
    }

    /**
     * Initializes the Elo store. It requires the rules from the A-Select configuration
     * specifying the Elos.
     * <br><br>
     * @see org.aselect.server.elo.IELOStorage#init(java.lang.Object, org.aselect.system.configmanager.ConfigManager, org.aselect.system.logging.SystemLogger)
     */
    public void init(Object oConfigSection, ConfigManager oConfigManager,
        SystemLogger systemLogger) throws ASelectException
    {
        _systemLogger = systemLogger;
        
        String sMethod = "init()";
        
        Object oELOs = null;
        try
        {
            oELOs = oConfigManager.getSection(oConfigSection, "elos");
        }
        catch (ASelectConfigException e)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod
                , "No config item 'elos' in 'elostore' section found", e);
            throw new ASelectException (Errors.ERROR_ASELECT_INIT_ERROR, e);
        }
        
        int iDefaultLevel = -1;
        String sDefaultLevel = null;
        try
        {
            sDefaultLevel = oConfigManager.getParam(oELOs, "authsp_level");
        }
        catch (ASelectException e)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod
                , "No optional (default level) parameter 'authsp_level' in 'elo' section found");
        }
        
        if (sDefaultLevel != null)
        {   
            try
            {
                iDefaultLevel = Integer.parseInt(sDefaultLevel);
            }
            catch(NumberFormatException nfe)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod
                    , "Parameter 'authsp_level' in 'elo' section is not an integer value: " 
                    + sDefaultLevel, nfe);
                throw new ASelectException (Errors.ERROR_ASELECT_INIT_ERROR);
            }
        }
        
        String sDefaultShowTemplate = "FALSE";
        try
        {
            sDefaultShowTemplate = oConfigManager.getParam(oELOs, "show_template");
        }
        catch (ASelectConfigException e)
        {
            _systemLogger.log(Level.CONFIG, MODULE, sMethod
                , "No optional (default show template) parameter 'show_template' in 'elos' section found");
        }
        
        boolean bDefaultShowTemplate = false;
        if ("TRUE".equalsIgnoreCase(sDefaultShowTemplate))
            bDefaultShowTemplate = true;
        else if (!"FALSE".equalsIgnoreCase(sDefaultShowTemplate))
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod
                , "Invalid parameter 'show_template' in 'elos' section found (must be true or false): " 
                + sDefaultShowTemplate);
            throw new ASelectException (Errors.ERROR_ASELECT_INIT_ERROR);
        }
        
        Object oELO = null;
        try
        {
            oELO = oConfigManager.getSection(oELOs, "elo");
        }
        catch (ASelectConfigException e)
        {
            _systemLogger.log(Level.CONFIG, MODULE, sMethod
                , "Not even one config item 'elo' in 'elos' section found", e);
        }
        
        while (oELO != null)
        {
            String sEloID = null;
            try
            {
                sEloID = oConfigManager.getParam(oELO, "id");
            }
            catch (ASelectConfigException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod
                    , "No parameter 'id' in 'elo' section found");
                throw new ASelectException (Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            String sEloURL = null;
            try
            {
                sEloURL = oConfigManager.getParam(oELO, "url");
            }
            catch (ASelectConfigException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod
                    , "No parameter 'url' in 'elo' section found");
                throw new ASelectException (Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            String sEloSS = null;
            try
            {
                sEloSS = oConfigManager.getParam(oELO, "shared_secret");
            }
            catch (ASelectConfigException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod
                    , "No parameter 'shared_secret' in 'elo' section found");
                throw new ASelectException (Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            int iLevel = -1;
            
            String sLevel = null;
            try
            {
                sLevel = oConfigManager.getParam(oELO, "authsp_level");
            }    
            catch (ASelectConfigException e)
            {
                if (iDefaultLevel < 0)
                {
                    _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                        "No parameter 'authsp_level' in 'elo' section with id: " + sEloID);
                    throw new ASelectException (Errors.ERROR_ASELECT_INIT_ERROR);
                }
                iLevel = iDefaultLevel;
            }
            
            if (sLevel != null)
            {
                try
                {
                    iLevel = Integer.parseInt(sLevel);
                }
                catch(NumberFormatException nfe)
                {
                    _systemLogger.log(Level.WARNING, MODULE, sMethod
                        , "Parameter 'authsp_level' in 'elo' section is not an integer value: " 
                        + sLevel, nfe);
                    throw new ASelectException (Errors.ERROR_ASELECT_INIT_ERROR);
                }
            }
            
            StringBuffer sbInfo = new StringBuffer();
            sbInfo.append("Using authsp_level '");
            sbInfo.append(iLevel);
            sbInfo.append("' for ELO with id: ");
            sbInfo.append(sEloID);
            _systemLogger.log(Level.INFO, MODULE, sMethod, 
                sbInfo.toString());
            
            if (_mELOsIDKeys.containsKey(sEloID))
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod
                    , "duplicate elo ID found in section 'elos': " + sEloID);
                throw new ASelectException (Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            if (_mELOsURLKeys.containsValue(sEloURL))
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod
                    , "duplicate elo URL found in section 'elos': " + sEloURL);
                throw new ASelectException (Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            boolean bShowTemplate = false;
            try
            {
                String sShowTemplate = oConfigManager.getParam(oELO, "show_template");
                if ("TRUE".equalsIgnoreCase(sShowTemplate))
                    bShowTemplate = true;
                else if (!"FALSE".equalsIgnoreCase(sShowTemplate))
                {
                    _systemLogger.log(Level.WARNING, MODULE, sMethod
                        , "Invalid parameter 'show_template' in 'elo' section found (must be true or false): " + sShowTemplate);
                    throw new ASelectException (Errors.ERROR_ASELECT_INIT_ERROR);
                }
            }
            catch (ASelectConfigException e)
            {
                bShowTemplate = bDefaultShowTemplate;
                _systemLogger.log(Level.CONFIG, MODULE, sMethod
                    , "No optional parameter 'show_template' in 'elo' section found, using default: " 
                    + bShowTemplate);
            }
            
            ELO elo = new ELO(sEloID, sEloSS, sEloURL, iLevel, bShowTemplate);
            _mELOsIDKeys.put(sEloID, elo);
            _mELOsURLKeys.put(sEloURL, elo);
            
            oELO = oConfigManager.getNextSection(oELO);
        }
    }
}
