/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */

package org.aselect.server.processor;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Vector;
import java.util.logging.Level;

import org.aselect.server.log.ASelectSystemLogger;

/**
 * NameMultiValue pair <br>
 * <br>
 * <b>Description:</b><br>
 * NameValue pair. Contains a name and value. <br>
 * <br>
 * <b>Concurrency issues:</b> <br>
 * None <br>
 */
public class NMVPair
{
    private final static String MODULE = "NMVPair";
    private ASelectSystemLogger _systemLogger;
    private String _sName;
    private List _listValues;

    /**
     * Create a new {@link NMVPair} <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param name
     *            The name.
     */
    public NMVPair (String name)
    {
        _systemLogger = ASelectSystemLogger.getHandle();
        _sName = name;
        _listValues = new LinkedList();
    }

    /**
     * Create a new {@link NMVPair} <br>
     * <b>Description:</b> <br>
     * Create a new {@link NMVPair}. With no value. <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param name name as String
     * @param value value as String
     */
    public NMVPair (String name, String value)
    {
        _systemLogger = ASelectSystemLogger.getHandle();
        _sName = name;
        _listValues = new LinkedList();
        _listValues.add(value);
    }

    /**
     * Create a new {@link NMVPair} with a default value. <br>
     * <b>Description:</b> Create a new {@link NMVPair} with a default value.
     * <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param name
     *            The name of the pair
     * @param value
     *            The value of the pair.
     */
    public NMVPair (String name, Vector value)
    {
        _systemLogger = ASelectSystemLogger.getHandle();
        _sName = name;
        _listValues = new LinkedList();
        _listValues.addAll(value);
    }

    /**
     * Add to the mixed hashtable the value(s) of the current object. <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param ht The hashtable where to add the variables.
     */
    public void updateMixedHashtable(Hashtable ht)
    {
        if (_listValues.size() > 1)
        {
            ht.put(_sName, new Vector(_listValues));
        }
        else
        {
            ht.put(_sName, _listValues.get(0));
        }
    }

    /**
     * Add a single value to this pair. <br>
     * <br>
     * <b>Description:</b> <br>
     * Add a single value to this pair. <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param value
     */
    public void addValue(String value)
    {
        _listValues.add(value);

    }

    /**
     * Add one or more values to this name value pair. <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @param v
     *            A vector of strings to add.
     */
    public void addValues(Vector v)
    {
        _listValues.addAll(v);
    }

    /**
     * Encodes the name multivalue pair into a url-encoded pair. <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @return The url-encoded string.
     */
    public String encodeMV()
    {
        String sMethod = "encodeMV()";
        StringBuffer sbReturn = new StringBuffer();

        String encodedName = "";
        try
        {
            encodedName = URLEncoder.encode(_sName, "UTF-8");
        }
        catch (UnsupportedEncodingException e)
        {
            _systemLogger.log(Level.FINE, MODULE, sMethod, 
                "Could not url encode with UTF-8", e);
            return sbReturn.toString();
        }

        Iterator i = _listValues.iterator();
        while (i.hasNext())
        {
            String tmpS = (String)i.next();
            try
            {
                String encodedValue = URLEncoder.encode(tmpS, "UTF-8");

                sbReturn.append(encodedName);
                sbReturn.append('=');
                sbReturn.append(encodedValue);
                if (i.hasNext())
                {
                    sbReturn.append('&');
                }
            }

            catch (UnsupportedEncodingException e)
            {
                _systemLogger.log(Level.FINE, MODULE, sMethod, 
                    "Could not url encode with UTF-8", e);
            }
        }

        return sbReturn.toString();
    }

    /**
     * Retrieve the name of this NVPair. <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @return The name of this NVPair.
     */
    public String getName()
    {
        return _sName;
    }

    /**
     * Retrieve the value of this NVPair. <br>
     * <br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     * 
     * @return The value of this NVPair.
     */
    public List getValues()
    {
        return _listValues;
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString()
    {
        StringBuffer sbString = new StringBuffer("N: ");
        sbString.append(_sName);
        sbString.append(" V: ");
        sbString.append(_listValues);
        return sbString.toString();

    }
}
