/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */
package org.aselect.server.request.handler.attributeprocessor;

import java.util.Hashtable;
import java.util.logging.Level;

import javax.servlet.ServletConfig;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.aselect.server.log.ASelectAuthenticationLogger;
import org.aselect.server.request.RequestState;
import org.aselect.server.request.handler.AbstractRequestHandler;
import org.aselect.server.session.SessionManager;
import org.aselect.server.tgt.TGTIssuer;
import org.aselect.server.tgt.TGTManager;
import org.aselect.system.error.Errors;
import org.aselect.system.exception.ASelectConfigException;
import org.aselect.system.exception.ASelectException;
import org.aselect.system.utils.Utils;

/**
 * Request handler for finishing the cross authentication process for the 
 * AttributeProcessor.
 * <br><br>
 * <b>Description:</b><br>
 * Issues a cross TGT if this handler is called by the AttributeProcessor
 * component to finish the authentication process after user interaction
 * with an external application.
 * <br><br>
 * <b>Concurrency issues:</b> <br> - <br>
 * @author Alfa & Ariss
 */
public class AttributeProcessorProceeder extends AbstractRequestHandler
{
    private static final String MODULE = "AttributeProcessorProceeder";
    
    private ASelectAuthenticationLogger _authenticationLogger;
    private TGTManager _tgtManager;
    private SessionManager _sessionManager;
    private String _sMyServerId;
        
    /**
     * Constructor.
     * <br><br>
     * <b>Description:</b>
     * <br>
     * Retrieves the singletons:
     * <ul>
     * <li>ASelectAuthenticationLogger</li>
     * <li>TGTManager</li>
     * <li>SessionManager</li>
     * </ul>
     * <br><br>
     * <b>Concurrency issues:</b> <br> - <br>
     * <br>
     * <b>Preconditions:</b> <br> - <br>
     * <br>
     * <b>Postconditions:</b> <br> - <br>
     */
    public AttributeProcessorProceeder()
    {
        _authenticationLogger = ASelectAuthenticationLogger.getHandle();
        _tgtManager = TGTManager.getHandle();
        _sessionManager = SessionManager.getHandle();
    }
    
    /**
     * Initializes the request handler.
     * <br><br>
     * @see org.aselect.server.request.handler.AbstractRequestHandler#init(javax.servlet.ServletConfig, java.lang.Object)
     */
    public void init(ServletConfig oServletConfig, Object oConfig) throws ASelectException
    {
        String sMethod = "init()";
        
        try
        {
            super.init(oServletConfig, oConfig);
            
            Object oASelect = null;
            try
            {
                oASelect = _configManager.getSection(null, "aselect");
            }
            catch(ASelectConfigException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "Could not find 'aselect' config section in config file", e);
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
            
            try
            {
                _sMyServerId = _configManager.getParam(oASelect, "server_id");
            }
            catch(ASelectConfigException e)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "Could not retrieve 'server_id' config parameter in 'aselect' config section",e);
                throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
            }
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, "Internal error during initialize", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
    }
    
    /**
     * Removes class variables from memory.
     * <br><br>
     * @see org.aselect.server.request.handler.IRequestHandler#destroy()
     */
    public void destroy()
    {
        //nothing to do
    }

    /**
     * Issues an cross TGT.
     * <br>
     * Issues a cross TGT if this handler is called by the AttributeProcessor 
     * component to finish the authentication process after user interaction 
     * with an external application.
     * <br><br>
     * @see org.aselect.server.request.handler.IRequestHandler#process(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    public RequestState process(HttpServletRequest request,
        HttpServletResponse response) throws ASelectException
    {
        String sMethod = "process()";
        try
        {
            String sRid = request.getParameter("rid");
            if (sRid == null)
            {
                _systemLogger.log(Level.FINER, MODULE, sMethod, "No rid in request");
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
            }
            
            if (!validateRid(sRid))
            {
                _systemLogger.log(Level.FINER, MODULE, sMethod, "Invalid rid in request: " + sRid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
            }
            
            Hashtable htSession = _sessionManager.getSessionContext(sRid);
            if (htSession == null)
            {
                _systemLogger.log(Level.WARNING, MODULE, sMethod, 
                    "Session expired -> SessionContext not available for rid: " + sRid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_SESSION_EXPIRED);
            }
            
            String sRemoteOrganization = (String)htSession.get("remote_organization");
            if (sRemoteOrganization == null)
            {
                _systemLogger.log(Level.FINER, MODULE, sMethod, 
                    "Invalid session: No 'remote_organization' in session with rid: " + sRid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
            }
            
            String sAppId = (String)htSession.get("app_id");
            if (sAppId == null)
            {
                _systemLogger.log(Level.FINER, MODULE, sMethod, 
                    "Invalid session: No 'app_id' in session with rid: " + sRid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
            }  
            
            Hashtable htRemoteAttributes = (Hashtable)htSession.get("cross_attributes");
            if (htRemoteAttributes == null)
            {
                _systemLogger.log(Level.FINER, MODULE, sMethod, 
                    "Invalid session: No 'cross_attributes' in session with rid: " + sRid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
            }
            
            String sUID = (String)htRemoteAttributes.get("uid");
            if (sUID == null)
            {
                _systemLogger.log(Level.FINER, MODULE, sMethod, 
                    "Invalid session: No 'uid' in 'cross_attributes' in session with rid: " + sRid);
                throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
            }
            
            //Log successful authentication
            _authenticationLogger.log(new Object[] {
                    "Cross", 
                    sUID,
                    request.getRemoteAddr(), 
                    sRemoteOrganization,
                    sAppId,
                    "granted"});
    
            TGTIssuer oTGTIssuer = new TGTIssuer(_sMyServerId);
            
            // Issue a cross TGT since we do not know the AuthSP
            // and we might have received remote attributes.
            oTGTIssuer.issueCrossTGT(sRid, null, htRemoteAttributes, response, getTGTFromCredentials(request), request);
        }
        catch (ASelectException e)
        {
            throw e;
        }
        catch (Exception e)
        {
            _systemLogger.log(Level.SEVERE, MODULE, sMethod, 
                "Internal error during process", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
        
        return new RequestState(null);
    }

    /*
     * validates the syntax of the rid parameter
     */
    private boolean validateRid(String rid)
    {
        return rid.matches("[A-Za-z0-9]{16}");
    }
    
    /*
     * Retrieves the  A-Select credentials from the cookie.
     */ 
    private String getTGTFromCredentials(HttpServletRequest request)
    {
        String sMethod = "getTGTFromCredentials()";
        // check for credentials that might be present
        Cookie[] caCookies = request.getCookies();
        if (caCookies == null)
            return null;
        
        String sCredentialsCookie = null;
        for (int i = 0; i < caCookies.length; i++)
        {
            if (caCookies[i].getName().equals("aselect_credentials"))
            {
                sCredentialsCookie = caCookies[i].getValue();
                
                _systemLogger.log(Level.FINER, MODULE, sMethod, 
                    "Cookie 'aselect_credentials' has value: " + sCredentialsCookie);
                                
                //remove '"' surrounding cookie if applicable
                int iLength = sCredentialsCookie.length();
                if(sCredentialsCookie.charAt(0) == '"' &&
                    sCredentialsCookie.charAt(iLength-1) == '"')
                {
                    sCredentialsCookie = sCredentialsCookie.substring(
                        1, iLength-1);
                }
            }
        }
        if (sCredentialsCookie == null)
            return null;
        
        Hashtable sCredentialsParams = Utils.convertCGIMessage(sCredentialsCookie);
        if (sCredentialsParams == null)
            return null;
        
        String sTgt = (String)sCredentialsParams.get("tgt");
        if (sTgt == null)
            return null;
        
        String sUserId = (String)sCredentialsParams.get("uid");
        if (sUserId == null)
            return null;
        
        String sServerId = (String)sCredentialsParams.get("a-select-server");
        if (sServerId == null)
            return null;
        
        if (!sServerId.equals(_sMyServerId))
            return null;
        
        Hashtable htTGTContext = _tgtManager.getTGT(sTgt);
        if (htTGTContext == null)
            return null;
        
        if (!sUserId.equals(htTGTContext.get("uid")))
            return null;
        
        return sTgt;
    }
}
