/*
 * Copyright (c) Stichting SURF. All rights reserved.
 * 
 * A-Select is a trademark registered by SURFnet bv.
 * 
 * This program is distributed under the A-Select license.
 * See the included LICENSE file for details.
 * 
 * If you did not receive a copy of the LICENSE 
 * please contact SURFnet bv. (http://www.surfnet.nl)
 */
package org.aselect.server.request.handler.entree.sso.cookiemonster;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Hashtable;
import java.util.logging.Level;

import javax.servlet.ServletConfig;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.aselect.server.elo.ELO;
import org.aselect.server.elo.ELOFactory;
import org.aselect.server.elo.IELOStorage;
import org.aselect.server.request.RequestState;
import org.aselect.server.request.handler.AbstractRequestHandler;
import org.aselect.server.session.SessionManager;
import org.aselect.system.error.Errors;
import org.aselect.system.exception.ASelectConfigException;
import org.aselect.system.exception.ASelectException;

/**
 * This handler receives requests sent by the ELO to indicate that a certain user
 * has authenticated. 
 * <br><br>
 * <b>Description:</b><br>
 * The ELO sends the following parameters with its request to the Cookie Monster:
 * <ul>
 * <li><b><code>rid</code></b> - the request ID, associated with the current
 * authentication session. The rid was initially sent to the ELO by the WAYF in the authentication
 * initialization request.</li>
 * <li><b><code>uid</code></b> - The user ID, as used by the ELO.</li>
 * <li><b><code>shared_secret</code></b> - The secret code that was established on the registration
 * of the ELO at the A-Select server.</li>
 * <li><b><code>attributes</code></b> (optional) - A string of URL-encoded attributes.</li>
 * </ul>
 * <br><br>
 * <b>Concurrency issues:</b> <br> - <br>
 * @author Alfa & Ariss
 */
public class CookieMonsterHandler extends AbstractRequestHandler
{
    private static final String MODULE = "CookieMonsterHandler";
    private String _sSSOserviceURL = null;
    private IELOStorage _oStore = null;
    private SessionManager _oSessionManager = null;
    
    private static final String REPLY_PARAM_NAME = "redirect_url";

    /**
     * Initialization of the handler.
     * <br><br>
     * @see org.aselect.server.request.handler.AbstractRequestHandler#init(javax.servlet.ServletConfig, java.lang.Object)
     */
    public void init(ServletConfig oServletConfig, Object oConfig) throws ASelectException
    {
        super.init(oServletConfig, oConfig);
        String sMethod = "init()";
        
        _oStore = ELOFactory.getHandle().getEloStore();
        _oSessionManager = SessionManager.getHandle();
        
        Object oSSOService = null;
        try
        {
            oSSOService = _configManager.getSection(oConfig, "ssoservice");
        }
        catch (ASelectConfigException e)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "No config section 'ssoservice' found");
            throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
        }
        
        try
        {
            _sSSOserviceURL = _configManager.getParam(oSSOService, "url");
        }
        catch (ASelectConfigException e)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "No config item 'url' in section 'ssoservice' found");
            throw new ASelectException(Errors.ERROR_ASELECT_INIT_ERROR);
        }
    }
    
    /**
     * Does not really do anything.
     * <br><br>
     * @see org.aselect.server.request.handler.IRequestHandler#destroy()
     */
    public void destroy()
    {
        //does not have to do anything
    }

    /**
     * Processes the Cookie Monster request.
     * <br><br>
     * @see org.aselect.server.request.handler.IRequestHandler#process(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    public RequestState process(HttpServletRequest request,
        HttpServletResponse response) throws ASelectException
    {
        
        String sMethod = "process()";
        
        if (_oStore == null || _oSessionManager == null)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "Cookie monster not initialized");
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
        String sRid = request.getParameter("rid");
        String sUID = request.getParameter("uid");
        String sSharedSecret = request.getParameter("shared_secret");
        String sAttributes = request.getParameter("attributes");
        
        if (sRid == null || "".equals(sRid))
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "Elo Cookie Monster request lacks 'rid' parameter, and response could not be sent");
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
        }
        
        if (sUID == null || "".equals(sUID))
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "Elo Cookie Monster request lacks 'uid' parameter, and response could not be sent");
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
        }
        
        if (sSharedSecret == null || "".equals(sSharedSecret))
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "Elo Cookie Monster request lacks 'shared_secret' parameter, and response could not be sent");
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
        }
        
        Hashtable htSession = _oSessionManager.getSessionContext(sRid);
        if (htSession == null)
        {
            _systemLogger.log(Level.INFO, MODULE, sMethod, "Could not load session");
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
        }
        
        String sEloID = (String)htSession.get("elo_id");
        if (sEloID == null)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "Could not retrieve 'elo_id' from session, ELO ID: " + sEloID);
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
        }
        
        ELO oElo = _oStore.getEloByID(sEloID);
        if (oElo == null)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "Could not load Elo information from elo store, ELO ID: " + sEloID);
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_SESSION);
        }
        
        if (! sSharedSecret.equals(oElo.getSharedSecret()))
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "ELO shared secret mismatch, ELO ID: " + sEloID);
            throw new ASelectException(Errors.ERROR_ASELECT_SERVER_INVALID_REQUEST);
        }
        
        //ok, continue updating session
        htSession.put("uid", sUID);
        
        if (sAttributes != null && !"".equals(sAttributes))
        {
            htSession.put("elo_attributes", sAttributes);
        }
        
        htSession.put("remote_organization", oElo.getID());
        
        _oSessionManager.updateSession(sRid, htSession);
        
        //all is OK. reply to user.
        StringBuffer sbValue = new StringBuffer();
        sbValue.append(_sSSOserviceURL);
        if (_sSSOserviceURL.indexOf('?') == -1)
            sbValue.append("?");
        else
            sbValue.append("&");
        sbValue.append("rid=");
        sbValue.append(sRid);
        
        PrintWriter pWriter = null;
        try
        {
            StringBuffer sbReply = new StringBuffer(REPLY_PARAM_NAME);
            sbReply.append("=");
            sbReply.append(URLEncoder.encode(sbValue.toString(), "UTF-8"));
            
            pWriter = response.getWriter();
            
            _systemLogger.log(Level.FINER, MODULE, sMethod,
                "Sending response: " + sbReply.toString());
            
            pWriter.write(sbReply.toString());
        }
        catch (UnsupportedEncodingException e)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "Could not URLEncode with UTF-8", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
        catch (IOException e)
        {
            _systemLogger.log(Level.WARNING, MODULE, sMethod, "Cookie monster could not reply", e);
            throw new ASelectException(Errors.ERROR_ASELECT_INTERNAL_ERROR);
        }
        finally
        {
            if (pWriter != null)
                pWriter.close();
        }

        return new RequestState(null);
    }
}
