use utf8;
use strict;
use warnings FATAL => 'all';

package UvT::Kiki::CAS::Request;

use Xyzzy::Util qw(uri_escape_auto);
use LWP;

use UvT::Kiki::UI::Request -self;

field crypto => sub { shift->cfg->crypto };
field login_url => sub { shift->cfg->cas_login_url };
field validate_url => sub { shift->cfg->cas_validate_url };

param ticket => sub { undef $_ unless defined && /^ST-[a-z0-9-]+$/i };

sub build_url() {
	my $url = shift;
	$url =~ s/(\#.*)//;
	my $frag = $1 // '';
	my $sep = index($url, '?') == -1 ? '?' : '&';
	my @ret = ($url);
	my $q = $_[0];
	@_ = %$q if ref $q;
	while(@_) {
		my $key = shift;
		my $val = shift;
		push @ret, $sep, uri_escape_auto($key), '=', uri_escape_auto($val);
		$sep = '&';
	}
	return join('', @ret, $frag);
}

sub fetch_url {
	my $url = build_url(@_);
	our $ua //= new LWP::UserAgent;
	my $res = $ua->get($url);
	die "got ".$res->status_line." while fetching '$url'\n" unless $res->is_success;
	return $res->decoded_content;
}

sub cas_clean_url {
	my $url = $self->self_url;
	$url =~ s{\?(.*)}{} or return $url;
	my $query = '';
	foreach my $q (split(qr{[;&]+}, $1)) {
		next if $q =~ /^ticket(?:=|$)/;
		$query .= "&$q";
	}
	$query =~ s{^&}{?};
	return $url.$query;
}

sub create_session {
	my $uid = shift;
	my $noise = 's'.$self->remote_addr;
	return scalar $self->crypto->create_token($noise, $uid);
}

field session => sub {
	my $self = shift;
	my $session = $self->cookie('session');
	return undef unless defined $session;
	my $noise = 's'.$self->remote_addr;
	return scalar eval { $self->crypto->check_token($noise, $session, $self->cfg->session_timeout) };
};

sub redirect {
	my $url = shift;
	my $status = new Xyzzy::Status(302);
	my $res = $status->response;
	$res->addheader(Location => $url);
	die $res;
}

sub login {
	my $uid = $self->session;
	return $uid if defined $uid;
	my $url = $self->cas_clean_url;
	if(my $ticket = $self->ticket) {
		my $res = $self->fetch_url($self->validate_url, ticket => $ticket, service => $url);
		if($res =~ /^yes\n([A-Za-z0-9_-]+)\n$/) {
			my $uid = $1;
			my $session = $self->create_session($uid);
			$self->session($session);
			my $doc = new UvT::Kiki::CAS::Document(req => $self, url => $url, session => $session);
			die $doc->response;
		}
	}
	$self->redirect(build_url($self->login_url, service => $url));
}

param nonce => sub {
	my $self = shift;
	my $session = $self->session;
	$_ = defined $session && eval { $self->crypto->check_token('n'.$session, $_, $self->cfg->request_timeout); 1 };
};

sub create_nonce {
	my $session = $self->session;
	return undef unless defined $session;
	return scalar $self->crypto->create_token('n'.$session);
}
