use utf8;
use strict;
use warnings FATAL => 'all';

package KUB::Manage::Request;

use Xyzzy::Util qw(parse_bool);
use Unicode::Normalize qw(decompose);
use Crypt::OpenSSL::Random;
use KUB::URL qw(normalize_url valid_http_url);

use KUB::Request -self;

field private => sub {
	my ($private) = shift->param('private')
		or return undef;
	return eval { parse_bool($private) } // 1;
};

param remove;

my @randomchars = qw(b c d f g h j k l m n p q r s t v w x z B C D F G H J K L M N P Q R S T V W X Z);

param submit;

field cur_name => sub { shift->param('name') };
field new_name => sub {
	my $self = shift;
	my $name = $self->cur_name;
	if(defined $name) {
		$name = decompose($name);
		$name =~ tr{ ._:,!@#$%^&*-}{-}s;
		$name =~ tr{A-Za-z0-9/-}{}cd;
		$name =~ s{--+}{-}g;
		$name =~ s{^-|-$}{}g;
		$name =~ s{//+}{/}g;
		$name =~ s{^/|/$}{}g;
	}
	return $name;
};

field random_name => sub {
	my $self = shift;
	for(my $i = 0; $i < 10; $i++) {
		my $name = join('', map { $randomchars[$_ % @randomchars] }
			unpack('S*', Crypt::OpenSSL::Random::random_bytes(2 * 5)));
		return $name unless $self->name_info($name);
	}
};

field valid_name => sub {
	my $name = shift->new_name;
	return defined $name && $name ne '';
};

field cur_url => sub { shift->param('url') };
field new_url => sub { normalize_url(shift->cur_url) };
field valid_url => sub { valid_http_url(shift->new_url) };

field name_owner => sub {
	my $self = shift;
	my ($num, $owner, $destination) = $self->name_info($self->new_name)
		if $self->valid_name;
	$self->name_num($num);
	$self->name_destination($destination);
	return $owner;
};

field name_num => sub { shift->name_owner; return };
field name_destination => sub { shift->name_owner; return };

sub name_info {
	my $name = shift;
	my $db = $self->db;
	my $q = $db->prepare_cached('SELECT url, owner, destination FROM urls WHERE name = ?');
	$q->execute($name);
	my $res = $q->fetchrow_arrayref;
	$q->finish;
	return @{$res // []};
};

field cur_edit => sub { shift->param('edit') };
field new_edit => sub {
	my $self = shift;
	my ($edit, $name, $owner, $destination) = $self->edit_info($self->cur_edit);
	$self->edit_name($name);
	$self->edit_owner($owner);
	$self->edit_destination($destination);
	$self->edit_valid(defined $owner);
	return $owner && $edit;
};

field edit_owner => sub { shift->new_edit; return };
field edit_name => sub { shift->new_edit; return };
field edit_destination => sub { shift->new_edit; return };
field edit_valid => sub { shift->new_edit; return };

sub edit_info {
	my $edit = shift;

	return unless defined $edit;
	return unless $edit =~ /^\d+$/;
	my $int = eval { int($edit) };
	return unless defined $int;
	return unless $int eq $edit;

	my $db = $self->db;
	my $q = $db->prepare_cached('SELECT name, owner, destination FROM urls WHERE url = ?');
	$q->execute($int);
	my $res = $q->fetchrow_arrayref;
	$q->finish;

	return $int, @{$res // []};
}

field urls => sub {
	my $self = shift;
	my $anr = $self->anr;
	my $db = $self->db;
	my $q = $db->prepare_cached('SELECT url, name, destination, private, visits, EXTRACT(EPOCH FROM ctime)::BIGINT, EXTRACT(EPOCH FROM mtime)::BIGINT, EXTRACT(EPOCH FROM atime)::BIGINT FROM urls WHERE owner = ?');
	$q->execute($anr);
	my $res = $q->fetchall_arrayref;
	$q->finish;
	return [map {
		my %h; @h{qw(num name destination private visits ctime mtime atime)} = @$_;
		new Clarity(%h);
	} @$res];
};

field anr => sub {
	my $self = shift;
	my $uid = $self->aselect_force_login;
	return $self->cfg->dir->anr($uid);
};

const feedback => [];

sub upsert {
	my $db = $self->db;
	if(my $edit = $self->new_edit) {
		my $q = $db->prepare_cached('UPDATE urls SET name = ?, destination = ?, private = ?, mtime = CURRENT_TIMESTAMP WHERE url = ?');
		$q->execute($self->new_name, $self->new_url, $self->private ? 't' : 'f', $edit);
		$q->finish;
	} else {
		my $q = $db->prepare_cached('INSERT INTO urls (name, destination, owner, private) VALUES (?, ?, ?, ?)');
		$q->execute($self->new_name, $self->new_url, $self->anr, $self->private ? 't' : 'f');
		$q->finish;
	}
}

sub delentry {
	my $db = $self->db;
	my $num = shift;
	my $q = $db->prepare_cached('DELETE FROM urls WHERE url = ?');
	$q->execute($num);
	$q->finish;
}

field success => sub {
	my $self = shift;

	my $feedback = $self->feedback;
	my $anr = $self->anr;

	if(defined $self->remove) {
		my $remove = $self->remove;
		my ($num, $owner) = $self->name_info($remove);
		if($owner) {
			if($owner != $anr) {
				push @$feedback, ['delete-notyours', $self->cfg->dir->displayname($owner)];
			}
		} else {
			push @$feedback, ['delete-nonexistent', {name => $self->remove}];
		}

		return undef if @$feedback;

		return undef unless $self->nonce;

		$self->delentry($num);
	} else {
		if($self->cur_edit) {
			if(my $owner = $self->edit_owner) {
				if($owner != $anr) {
					push @$feedback, ['edit-notyours', $self->cfg->dir->displayname($owner)];
				}
			} else {
				push @$feedback, ['edit-deleted'];
			}
		}

		if($self->valid_name) {
			if(defined $self->cur_name) {
				if($self->cur_name ne $self->new_name) {
					push @$feedback, ['name-tweaked', {name => $self->cur_name}];
				}
				if(my $owner = $self->name_owner) {
					if($owner == $anr) {
						if(my $edit = $self->new_edit) {
							if($self->name_num != $edit) {
								push @$feedback, ['edit-overwrite', {num => $self->name_num, url => $self->name_destination}];
							}
						} else {
							push @$feedback, ['name-exists', {num => $self->name_num, url => $self->name_destination}];
						}
					} else {
						push @$feedback, ['name-notyours', $self->cfg->dir->displayname($owner)];
					}
				}
			}
		} else {
			if(($self->cur_name // '') ne '') {
				push @$feedback, ['name-invalid', {name => $self->cur_name}];
			} else {
				push @$feedback, ['name-missing'];
			}
		}

		if($self->valid_url) {
			if($self->cur_url ne $self->new_url) {
				push @$feedback, ['url-tweaked', {url => $self->cur_url}];
			}
		} else {
			if(($self->cur_url // '') ne '') {
				push @$feedback, ['url-invalid', {url => $self->cur_url}];
			} else {
				push @$feedback, ['url-missing'];
			}
		}

		return undef if @$feedback;

		return undef unless $self->submit && $self->nonce;

		$self->upsert;
	}

	return 1;
}
