package Baseobject;

use strict ;
use Cwd;
use Carp qw (cluck carp);
our @CARP_NOT;
use Data::Dumper;
require      Exporter;
our @ISA = qw (Exporter);
our @EXPORT = qw (baseconfigure verbosity freezeverbosity dumper);

our $VERSION = "1.27";

$Baseobject::directfeedback=0; # for downward compatibility
$Baseobject::quickdeath=0;
$Baseobject::verbosity=1;


sub new {
	my $pkg = shift;
	my $param = shift;
	my $self = bless {},$pkg;

	$self->{LOCALSTATEDIR}= '';
	$self->{SYSCONFDIR}   = '';
	$self->{DATADIR}      = '';
	$self->{PACKAGE}      = '';

	$self->{LOCALSTATEDIR} = $main::LOCALSTATEDIR if defined ($main::LOCALSTATEDIR);
	$self->{SYSCONFDIR} = $main::SYSCONFDIR if defined ($main::SYSCONFDIR);
	$self->{DATADIR} = $main::DATADIR if defined ($main::DATADIR);
	$self->{PACKAGE} = $main::PACKAGE if defined ($main::PACKAGE);

	unless ($self->{DATADIR}) {
		$self->{DATADIR} = $main::datadir if defined ($main::datadir);
	}

	unless ($self->{SYSCONFDIR})  {
		$self->{SYSCONFDIR} = $main::sysconfdir if defined($main::sysconfdir);
	}

	unless($self->{LOCALSTATEDIR}) {
		$self->{LOCALSTATEDIR} = $main::localstatedir if defined($main::localstatedir);
	}

	$self->clearerror();
	$self->clearfriendlyerror();

	$self->addfeatures ( {'allows'=>1, 'needs'=>1} );
	$self->allows ({'name'=>'unnamed'});

	#compatibility
	$self->{public}=$self->{fields};
	return $self;
}

sub features{
	my $self = shift;
	my $feature = shift;
	my $ptr = shift;

	if ( ref($ptr) eq 'HASH') {
		foreach my $var (keys %$ptr) {
			#set the default
			$self->{$feature}{$var} = $ptr->{$var};
			$self->field($var, $ptr->{$var});
		}
	} elsif( ref($ptr) eq 'ARRAY') {
		foreach my $var (@$ptr) {
			$self->{$feature}{$var} = '*';
		}
	} else {
		# scalar
		$self->{$feature}{$ptr}='*';
	}
}

sub allows {
	my $self = shift;
	$self->features("allows",@_);
}

sub needs {
	my $self = shift;
	$self->features("needs", @_);
}

sub addfeatures {
	my $self=shift;
	my $pFeatures=shift;

	foreach my $feature (keys %$pFeatures) {
		$self->{features}->{$feature} = $pFeatures->{$feature};
	}
}

##########################################
sub stderror {
	my $self = shift;
	$self->error(@_);
	carp @_, "\n";
}

sub error {
	my $self = shift;
	if (@_ && $_[0]) {
		confess (@_) if $self->quickdeath;
		$self->{error} .= join ('',@_);
		return 0;
	} else {
		my ($package, $file, $line) = caller;
		return "$package:$file:$line " . $self->{error} if $self->{error};
		return $self->{error};
	}
}

sub clearerror {
	my $self = shift;
	$self->{error} = "";
}

sub friendlyerror
{
	my $self = shift;

	if (@_ && $_[0]) {
		$self->{friendlyerror} .= join ('',@_);
		return 0;
	} else {
		if ($self->{friendlyerror}) {
			return $self->{friendlyerror};
		} else {
			return $self->{error};
		}
	}
}

sub clearfriendlyerror {
	my $self = shift;
	$self->{friendlyerror} = "";
}


###########################################


sub field {
	my $self = shift;
	my $var = shift;
	my $value = shift;

	$var='value' unless $var;
	return $self->fields($var) if ref ($var) eq 'HASH';

	unless (exists ($self->{allowallfields}))	{
		unless ( (exists  $self->{allows}->{$var}) ||
				 (exists  $self->{needs}->{$var})
			) { 
			$self->stderror("field: unknown variable: \"$var\", accepted vars are:\n\t",
							$self->showallows());
			return "";
		}
	}

	return defined($value) ? $self->{fields}{$var} = $value
		: $self->{fields}{$var};
}


sub fields {
	my $self = shift;
	my $pp = shift;

	unless ($pp) {
		return $self->{fields};
	}

	my $type = ref($pp);

	if ($type eq 'HASH') {
		my %cp=%$pp;
		my @keylist;

		foreach my $thing (qw (name options default )) {
			if (exists ($pp->{$thing})) {
				push (@keylist,$thing) ;
				delete $cp{$thing};
			}
		}
		push (@keylist, (keys %cp));

		foreach my $params (@keylist) {
			$self->field($params,$pp->{$params});
		}
		return; #$self->checkneeds();

	} elsif ($type eq 'ARRAY') {
		my @retlist=();
		foreach my $el (@$pp) {
			push (@retlist, $self->field($el));
		}
		return @retlist;
	} elsif (!$type && $pp) {
		#scalar
		return $self->field($pp);
	}
}

sub checkneeds {
	my $self = shift;
	my @missing;

	if (exists ($self->{needs})) {
		foreach my $need (keys %{$self->{needs}}) {
			push (@missing, $need) if !defined ($self->field($need));
		}
	}
	
	$self->error("missing requirement(s): ", join(', ', @missing)) if @missing;

	return $self;
}

sub inspect {
	my $string = Dumper(@_);
	my ($package, $file, $line) = caller();
	$string .= " Inspect called at: $file line $line\n";
}

sub showallows {
	my $self = shift;
	return join (' ',(  (keys %{$self->{needs}}), keys %{$self->{allows}})),"\n";
}

sub dumpfields {
	my $self = shift;
	return $self->inspect($self->{fields});
}


sub dumpall {
#
#call in despair only
#
	my $self = shift;
	use Dumpvalue;
	my $dumper = new Dumpvalue;
	$dumper->set(globPrint => 1);
	$dumper->dumpValue(\*::);
	$dumper->dumpvars('main');
}

sub iscgi {
	return $ENV{'SERVER_ADDR'};
}


sub basefilename {
	# returns basefilename
	my $self = shift;
	my $basename = $0;

	if ($basename =~ /\//) {
		#chop off path, note: it might contain nasty dots, get rid of them now
		$basename=~s/(^.*\/)//; #$path=$1;
	}
#  $basename=~s/\..*?$//;
	chomp $basename;
	return $basename;
}

sub baseconfigure {
	my $ptr=shift;
	if (ref ($ptr) ne 'HASH' ) {
		carp "Baseobject%ERROR baseconfigure needs ref to HASH, not: $ptr";
		return;
	}
	for  (keys (%$ptr))	{
		if (/^quickdeath$/) {
			$Baseobject::quickdeath = $ptr->{$_};
			next;
		} elsif (/^verbosity$/) {
			$Baseobject::verbosity = $ptr->{$_};
			next;
		} elsif (/^directfeedback$/) {
			$Baseobject::directfeedback=$ptr->{$_};
			next;
		}

		carp "Baseobject%ERROR baseconfigure accepts only: directfeedback, verbosity and quickdeath\n";
		carp "NOT: $_\n";

		next;
	}
}

sub quickdeath {
	my $self=shift;
	$Baseobject::quickdeath = 1 if !defined ($Baseobject::quickdeath) && (!@_);

	@_ ? $Baseobject::quickdeath = shift
		: $Baseobject::quickdeath;
}


sub freezeverbosity {
	$Baseobject::freezeverbosity = 0 if !defined ($Baseobject::freezeverbosity) && (!@_);
	@_ ? $Baseobject::freezeverbosity=shift: $Baseobject::freezeverbosity;
}

sub verbosity {
	if (@_) 	{
		my $ref = ref($_[0]);
		if ($ref)		{
			warn "WRONG PARAMETER FOR Baseobject::verbosity: should be 0, 1, 2, 3 or 4, not: type \"$_[0]\"\n";
			warn "Did you by any change call \$object->verbosity(int) ? in stead of Baseobject::verbosity(int)\n";
			warn "verbosity setting IGNORED\n";
			return;
		}
	}

	$Baseobject::verbose = 1 if !defined($Baseobject::verbose) && (!@_);

	(@_ && (!freezeverbosity) ) ? $Baseobject::verbosity = shift
		: $Baseobject::verbosity;
}

# 
sub verbose {
	my $self = shift;
	my ($package, $file, $line) = caller;
	my $string = join('', @_);
	chomp($string);
	print STDERR "$string at $file line $line (verbose)" if verbosity >= 1;
}

sub debug {
	my $self = shift;
	my ($package, $file, $line) = caller;
	my $string = join('', @_);
	chomp($string);
	print STDERR "$string at $file line $line (debug)" if verbosity >= 2;

	# carp heeft een off by one error in callstack als debug vanuit een eval() wordt aangeroepen:
	# carp @_ if verbosity >= 2;
}

sub expandvars {
	my $self = shift;
	my $ptrstring = shift;
	my $h = shift;

	unless (ref ($ptrstring) eq 'SCALAR') {
		carp "EXPANDVARS: first parameter to should be a ref to a scalar!";
		return undef;
	}

	unless (ref ($h) eq 'HASH') {
		carp "EXPANDVARS: second parameter should be a ref to a hash";
		return undef;
	}

	unless (defined ($$ptrstring)) {
		carp "EXPANDVARS: ptr to string undefined";
		return $$ptrstring;
	}

  fail:
	# recursive expand $vars from the fields
	# my $varexpr='\$([\w\{\}]+)';

	while ($$ptrstring =~ /\$([\w\{\}]+)/s) {
		my @varlist = $$ptrstring =~ /\$([\w\{\}]+)/sg;
		# carp "varlist: @varlist";
		foreach my $var (@varlist) {
			# carp "initial var: $var";
			# save curly brackets and slash them
			my $cpvar = $var; $cpvar =~ s/(\W)/\\$1/g;
			# remove curly brackets if they are there
			my ($secvar) = $var =~ /\{([^\}]+)\}/;
			$secvar = $var unless ($secvar);

			if (exists ($h->{$secvar})) {
				#emacs indentation realy screws up here
				my @x;
				if ($var =~ /\}/) {
					@x=$$ptrstring =~ s/\$\{$secvar\}/$h->{$secvar}/sg;
				} else {
					@x=$$ptrstring =~ s/\$$cpvar([\W\b]|$)/$h->{$secvar}$1/sg;
				}
				unless (@x) {
					cluck "Substitute failed: for \"\$$var\" \nQUITING expandvars";
					carp "$$ptrstring";
					return 0;
				} else {
					#show this only at level 2 or higher
					$self->debug("Expandvars: result: $$ptrstring");
				}
			} else {
				cluck "Could not substitute: \"\$$var\" \nQUITING expandvars";
				return 0;
			}
		}
	}
	return $$ptrstring;
}

package Readconfig;
use Carp;
use FileHandle;

our @ISA=('Baseobject' );
our @EXPORT = qw (verbosity);

sub new {
	#
	# configfile should contain lines with format:
	# key=value[ ,value1, value2, ... ]
	#
	# note: leading and trailing spaces are removed,
	#       empty and '#' commented lines are ignored
	#       if a line occurs more than once in a configfile they are concattenated
	#
	# parameters:
	#          FULLPATH to configfile
	#          ref to hash with keys: 'allows' and 'needs'
	#
	# par. ex.:
	# readsysconfig( $someconfigfile, { 'needs'=>[ qw ( key1 key2 key3 ) ], 'allows'=>[qw ( key4 key5)]});
	#
	# returns
	# a ref to a hash is containing key as key, and a ref to a list as value
	#
	my $pkg = shift;
	my $self = bless (new Baseobject(@_),$pkg);
	my $configfile = "$self->{SYSCONFDIR}\/". $self->basefilename(). '.cf';

	$self->allows( [ qw (needs allows cf )] );
	$self->allows (
		{
			configfile => $configfile,
			language => 'en',
			secret => 0,
		} );


	$self->fields(@_);

	return $self if $self->error();
	$configfile = $self->{fields}->{configfile};
	$configfile = "$self->{SYSCONFDIR}/$configfile" unless ($configfile =~ /\//);
	$self->debug("Readconfig: using configfile: $self->{fields}->{configfile}");

	#sorry english or dutch only
	$self->{fields}->{language} = 'en' if ($self->{fields}->{language} ne 'nl');

	my $cf = {};
	$self->field('cf', $cf);

	if ($self->{fields}->{secret}) {
		my $mode = (stat $configfile)[2];
		unless ($mode) {
			$self->error( "Could not stat file: \"$configfile\", $!" );
			return $self;
		}

		my $realmode = sprintf "%04o",($mode & 07777);
		my $gomode = sprintf "%04o",($mode & 00037);
		$gomode =~ s/0//g;
		if ($gomode) {
			$self->error( "ERROR filepermissions too permissive for file $configfile current permissions: $realmode\n");
			return $self;
		}
	}

	my $conf = new IO::File($configfile, '<:utf8');
	unless ($conf) {
		$self->error("Could not read config file: $configfile $!\n");
		unless ( -r $configfile ) {
			my $filename= $configfile;
			$filename =~ s/^.*\///;
			if ( ($self->{DATADIR}) and  ($self->{PACKAGE})) {
				$self->error(" $self->{PACKAGE}: ERROR mandatory CONFIGURATIONFILE NOT FOUND!\n");
				if (-r "$self->{DATADIR}/doc/$self->{PACKAGE}/$filename") {
					$self->error(" an EXAMPLEFILE may be found at location: $self->{DATADIR}/doc/$self->{PACKAGE}/$filename\n");
				}
			}
		}
		return $self;
	}
	my $cline = '';

	my $key;
	my $value;

	my $language = "";
	my $currentlanguage = '';

	if ($self->{fields}->{language}){
		$language="_(en|nl)" ;
		$currentlanguage="_$self->{fields}->{language}";
	}

	while(<$conf>) {
		chomp;
		# skip commented lines and empty lines
		next if (/^\s*\#/); next if /^\s*$/;

		#remove commented part of the line
#    s/\#.*//;

		if (/^\S/) {
			# something on the first column MUST BE A KEY
			#if values contains a '=', don't split it further, by limit of 2
			($key,$value) = split '=',$_,2 ;
			#remove leading/trailing spaces from the key
			$key =~ s/^\s*//; $key =~ s/\s*$//;
			#do wo expect this one?
			my $expected=0;
			foreach my $el (@{$self->{fields}->{needs}},@{$self->{fields}->{allows}} ) {
				$expected++ if ( $key=~/^$el$/);
				if ($language) {
					$expected++ if $key=~/^$el$language$/;
				}
			} unless ($expected) {
				carp "WARNING: Illegal key: \"$key\" in configfile: $configfile, line $.\n$_\n";
				next;
			}
		} else {
			$value = $_;
		}

		if (defined ($value)) {
#      warn "value: \"$value\"";
			# remove leading trailing spaces
			$value =~ s/^\s*//; $value =~ s/\s*$//;
			push (@{$cf->{$key}}, $value) ;
		}
	}
	$conf->close;

	my @missing;
	foreach my $key (@{$self->{fields}->{'needs'}}) {
		unless ($language) {
			push (@missing, $key) unless exists ($cf->{$key});
		} else {
			push (@missing, $key) unless exists ($cf->{$key.$currentlanguage}) or
										  exists ($cf->{$key});
		}

	}
	if (@missing) {
#    warn $self->inspect($cf);
		$self->error("ERROR: Missing required key(s) in configfile: $configfile\n\"", join ('" "',@missing), "\"\n");
	}
	return $self;
}

sub value {
	my $self = shift;
	my $cf = $self->{fields}->{cf};
	my $language = '';

	$language=$self->{fields}->{language} if defined($self->{fields}->{language});

	carp ("Baseobject warning: cf field undefined") unless defined ($cf);

	my $key = shift;
	unless ($key) {
		carp ("Baseobject warning: requesting empty key");
	}

	# if there is a key with language extension default to that
	if ($language) {
		my $lankey = "${key}_$language";
		if (exists ($cf->{$lankey})) {
			return wantarray ? @{$cf->{$lankey}} : "@{$cf->{$lankey}}";
		}
	}

	# no language or no language key take the default
	if (exists ($cf->{$key})) {
		return wantarray ? @{$cf->{$key}} : "@{$cf->{$key}}";
	} else {
		$self->error("Readconfig: key \"$key\" not found\n");
		return undef;
	}
}

sub hash {
	my $self = shift;
	my $cf = $self->{fields}->{cf};
	my $h;
	my $language = '';

	$language = $self->{fields}->{language} if defined($self->{fields}->{language});

	foreach my $key (keys (%$cf)){
		$key =~ s/_$language$//;
		$h->{$key} = $self->value($key) unless exists ($h->{$key});
	}

	return $h;
}


sub mergedata {
	my $self = shift;
	foreach my $cf (@_) {
		unless (ref($cf) eq 'Readconfig') {
			carp "Not a readconfig object" ;
			return;
		} 
		foreach my $key (keys (%{$cf->{fields}->{cf}})) {
			push(@{$self->{fields}->{cf}->{$key}}, (@{$cf->{fields}->{cf}->{$key}}));
		}

		push(@{$self->{fields}->{needs}},@{$cf->{fields}->{needs}});
		push(@{$self->{fields}->{allows}},@{$cf->{fields}->{allows}});
	}
}


42;
__END__

=head1 Baseobject.pm

=head1 SYNOPSIS

Baseobect provides basic variable (fields) checking, errormanaging and self inspecting features.

Baseobject is primarily ment to be used by other objects that inherit from it.


EXAMPLE:

 file: Forcast.pm

 package Forcast;
 use Baseobject;
 our @ISA=('Baseobject');
 sub new
 {
  my $pkg=shift;
  my $name=shift;

  my $self= bless (new Baseobject($name),$pkg);

  $self->allows( {file=>"$ENV{'HOME'}.climatefile"});
  $self->allows( [ qw (rainfall sunshine windspeed)] );

  # read parameters that were given in the $obj =new ({'param'=>'value'}) statement
  $self->fields(@_);

  return $self;
 }


 file: example.pl

 use Forcast;
 $obj=new forcast (
		  {name=>'weatherforcast',
		   rainfall=>'heavy',
		   sunshine=>'hardly'});

 #note: temperature has not been declared
 $obj->field('temperature','modest');
 warn $obj->error() if ($obj->error());

 print $obj->field('rainfall'),"\n\n";

 results in:

 Baseobject_example.pl  Error: unknown variable: "temperature", accepted vars are:
        name windspeed rainfall sunshine file
 heavy


=head1 DEPENDENCIES

  Trailer.pm

=head1 DESCRIPTION

Baseobject provides basic functionality any object should have in
the eyes of the author: Handling of: errorconditions, parameters,
and self inspection tools.

=head1 METHODS

=head2 allows ()

=over 2

SYNOPSIS:

As of now field() and fields() accept but do not require new field(s).

PARAMETERS:
Either a scalar, a ref to a hash or a ref to a list.
See the above example for an illustration.

Note: be sure not to provide al list, but a REF to a list!.

SEE ALSO:
    needs()

=back 2

=head2 checkneeds ()

=over 2

SYNOPSIS:
Checks wether the requisted needs(), where all fulfilled.

SEE ALSO:
    needs()

=back 2

=head2 clearerror ()

=over 2

SYNOPSIS:
    Clears the objecs errormessage buffer.

SEE ALSO:
    error() stderror()


=back 2

=head2 error ()

=over 2

SYNOPSIS:
    parameter: errormessage
        Set errormessage for a Baseobject.

    parameter: none
        Returns and/or prints errormessages for a Baseobject.

DESCRIPTION:
Errrormessages accumulate, and can be read at any time multiple times. This is the general way to check the status of an object. An empty string or 0 indicates success, otherwise the string itself should explain the problem.

The verbosity() level can be changed to your liking from 0 (silent) to 3 (very verbose).

The errormessages accumulate until the status is cleared with clearerror().


=back 2

=head2 field ()

=over 2

SYNOPSIS:
Typical 'getsetter' for objectfeatures, set by allows() and needs()

DESCRIPTION:
field('myvar','myvalue') saves 'myvalue' in the variable 'myvar', provided, that the variable 'myparameter' was introduced by a preceeding allows(myvar) or needs(myvar).

    If the variables has one or more of the special fields:

    options, default or command (see Sep)

    field(myvar) returns the current content of 'myvar'.

    EXAMPLE:

 examplefile: field_example.pl

 use Baseobject;
 my $obj=new Baseobject('weather');
 $obj->verbosity(3);
 $obj->allows([ qw (options default name) ]);
 $obj->field('name','anton');
 $obj->field('options', [ qw (rain fog clouds sunshine) ]);
 $obj->field('default','rain');

 #print some fields:
 print join (': ', $obj->fields( [ qw (name default)] )),"\n";

 #print all current fields by getting ref to internal fields hash:
 my $hashref=$obj->fields();
 foreach my $key (keys %$hashref)
 {
  print "key: $key, content: $hashref->{$key}\n";
 }

 print "options are: @{$obj->field(options)}\n";

 #change our luck
 #change several fields in one call
 $obj->fields( { 'name'=>'WeatherGod',
                'default'=>'sunshine' });

 print $obj->field('name'),": ";
 print $obj->field('default'),"\n";

    RESULTS in:

 anton: rain
 key: options, content: ARRAY(0x81038bc)
 key: default, content: rain
 key: name, content: anton
 options are: rain fog clouds sunshine
 WeatherGod: sunshine

SEE ALSO:
    fields()

=back

=head2 fields ()

=over

SYNOPSIS:
    Wrapper for field().

    parameter: ref to hash
        Name value pairs of this hash are copied internally, after being checked against allows() and needs().

    parameter: ref to array containing fieldnames
        returns list of values corresponding to list of fields.

    parameter: none
        returns ref to internal hash of fields.

Expects a ref to hash containing field,value pairs.

EXAMPLE:
    $obj->fields( { 'wind'=>'howling', 'winter'=>'cold' });

=back

=head2 inspect ()

=over 2

SYNOPSIS:
    Gives a datatree representation of the parameter.

iscgi()

SYNOPSIS:
Is the current program a CGI script?

 sub iscgi
 {
   return $ENV{'SERVER_ADDR'};
 }

Quit disgusting I know. I hope to improve it one day..

=back 2

=head2 needs ()

=over 2

SYNOPSIS:
    As of now field() and fields() accept and require the new field.
    This new field is mandatory.
    The parameters are identical to allows();

SEE ALSO:
    allows() checkneeds()

=back 2

=head2 new ()

=over 2

SYNOPSIS:
    Constructor, which set default needs() and allows()

DESCRIPTION:
    Note: the constructor already contains the following lines:

     $self->needs ([qw(name)]);
     $self->allows([qw (options default value)]);

    Your object should typically call this constructor in it's own constructor:

EXAMPLE:

 @newclass::ISA=('Baseobject');  # or: our @ISA=('Baseobject'), NOT my @ISA=..;
 sub new
 {
  my $pkg=shift;
  my $self= bless (new Baseobject($pkg),$pkg);

  $self->needs(['version']);
  $self->allows( [qw(file user)]);
  return $self;
 }

PARAMETERS:
    objectname

    An objectname is required. This name is used in errormessages as identification.

RETURNS:
    $self

=back 2

=head2 quickdeath()

=over 2

SYNOPSIS:
    Die at the first time an error() call is made with an errormessage. . Default = 1

SEE ALSO:
    error() verbosity()

=back 2

=head2 verbose ()

=over 2

SYNOPSIS:
    Writes a string to STDERR if verbosity() > 0 .

=back

=head2 verbosity ()

=over

SYNOPSIS:
    Sets or read verbositylevel for the entire object. Values rang from 0 to 4.

    level 0
        silent

    level 1
        activate warnings (default)

    level 2
        show caller in error()

    level 3
        show entire callstack in error() when error() called with parameters

    level 4
        show entire callstack in error() also when error() called without parameters. Produces far more output then you normally would want.

=back

=head2 stderror ()

=over

    Calls error() internally, but also writes toe STDERR directory.

=back

=head1 BUGS

Documentation is incomplete, obsolete, and sometimes wrrong, etc.
One day all things will be better.

=head1 AUTHOR

Anton Sluijtman, E<lt>anton@uvt.nlE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2004 by anton sluijtman

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.4 or,
at your option, any later version of Perl 5 you may have available.


=cut

# Local Variables:
# mode: perl
# mode: font-lock
# End:


