package Errorlog;
use Carp qw (cluck carp confess);
our @CARP_NOT;
use strict;
use Time::HiRes qw( gettimeofday tv_interval);

use Baseobject;
use Exporter;
our @ISA=('Baseobject' );
our @EXPORT = qw (baseconfigure verbosity);
use FileHandle;
use Carp ;
use Sys::Hostname;
use Sys::Syslog;
use Errorlog::TeeOutput;

#
# Can't locate object method "BINMODE" via package "Errorlog::TeeOutput::NewTee" at /home/anton/bin/keymaster line 10.
#

my $version = '$Id: Errorlog.pm 40273 2013-09-23 10:05:25Z anton $ ';
our $VERSION = "1.32";
#global variable, needed for TeeOutput
our $SELF;

sub import {
	$SIG{__DIE__} = sub { 
		die @_ if ref $_[0];
		if ($SELF->{fields}->{autoconfess}) {
			local $_ = join('', @_);
			if (s/^.*\K at \S+ line \d+\.?\n\z//) {
				local $Carp::CarpLevel = 1;
				confess($_);
			} else {
				die $_;
			}
		}
	}
}

sub end {
	carp  "Errorlog:end; Exitcode is: $?"  if $Baseobject::verbosity >=2;
	if ($SELF->{fields}->{mailonfailure}){
		my $logbuffer = $SELF->{fields}->{logbuffer} if $SELF->{fields}->{logbuffer};
		$SELF->mail("Errorlog v$VERSION failure: Errorstatus: $?, \n$logbuffer") ;
	}
}

END {
	my $status = $?;
	return if $SELF->{fields}->{skipEND};
	if ($status) {
		# call an extra 'end' method to be able to call this method from an eternally living
		# perl interpretor such as modperl.
		end();
	}
	$? = $status;
}


sub new {
	my $pkg = shift;
	my $self = bless (new Baseobject(@_),$pkg);
	$self->{tstart} = [gettimeofday];
	$self->{tstored} = $self->{tstart};

	$self->field('name','Log');

	my $fh;
	my $logfile = $self->makelogfilename();

	my $logdir = $self->{LOCALSTATEDIR};
	my $configdir = $self->{SYSCONFDIR};

	my $hostname = $self->{hostname} = hostname();
	my $user = $ENV{'REMOTE_USER'};
	$user = $ENV{'USER'} unless (defined ($user));

	$self->allows(
		{
			'configdir', $configdir,
			'facility' => 'local0',
			'fh' => $fh,
			'header' => '',
			'ipaddress' => $ENV{'REMOTE_ADDR'},
			'logbuffer' => "",
			'logdir' => $logdir,
			'logfile' => $logfile,
			'mailerrorsto' => "host_\@$self->{hostname}",
			'mailfrom' => "$self->{hostname}",
			'mailprogram' => '/usr/sbin/sendmail',
			'mailsubject' => "Errorlog: $0\@$self->{hostname}",
			'mailto' => '',
			'mailonfailure' => 0,
			'marknewrun' => 1,
			'newrunmarker' => '-'x60,
			'output' => [qw (stderr logbuffer)],
			'printinfo' => 1,
			'sysloglevel' => 'info',
			'syslogmaxbuf' => 512+255, # just guessing, 1024 is too big, things get truncated
			'hirestimer' => 0,
			'user' => $user,
			'version' => $VERSION,
			'autoconfess' => 1,
			'skipEND' => '',
		}
		);

	#note: modified version of field is called
	$self->fields(@_);

	if ($self->{fields}->{configdir}) {
		unless ( -d $self->{fields}->{configdir} ) {
			carp "Warning; Errorlog (version $VERSION):  configdir: \"$self->{fields}->{configdir}\" does not exist";
		} 
	} else {
		carp "No configdir specified";
	}
	
	$self->overrule();
	if ($self->error()) {
		carp "Errorlog failure";
		return $self ;
	}

	my @params = ();
	my %params;

	if ($self->{outputmodes}->{logfile}) {
		$logdir = $self->field('logdir');
		$logfile = $self->field('logfile');

		my $file = "$logdir/$logfile";
		unless (-w $file) {
			#test if we can write here before tie fails on it
			my $fh = new FileHandle();
			unless (open ($fh,">utf8", $file)) {
				carp "WARNING: Could not open file: $file, $!\nWARNING  No logging to logfile will occur. Output to stderr forced";
				# make sure stderr is not lost
				$self->{outputmodes}->{stderr} = 1;
				$self->{outputmodes}->{logfile} = 0;
			} else {
				close($fh);
			}
		}
	}

	push(@params,*STDERR) if $self->{outputmodes}->{stderr};
	push(@params,*STDOUT) if $self->{outputmodes}->{stdout};

	if ($self->{outputmodes}->{logfile}) {
		$logfile = $self->fields('logfile');
		$logdir = $self->fields('logdir');

		#if logfile does not contain a path, prepend the current logdir
		$logfile = "$logdir/$logfile"  unless $logfile =~ m{/};
		push (@params, ">>$logfile");
	}

	if ($self->{outputmodes}->{syslog}) {
		if (openlog($0, 'cons,pid',$self->{fields}->{facility}) ) {
			$params{'syslog'} = 1; #1 is dummy value
		} else {
			$self->error("Could not open syslog");
		}
	}

	#do this for now, must think about it some more
	$self->{outputmodes}->{logbuffer} = 1 if $self->{fields}->{mailonfailure};

	if ($self->{outputmodes}->{logbuffer})	{
		$params{'logbuffer'} = 1; #1 is dummy value
	}


#  unless (@params)
#  {
#    $self->stderror("WARNING: No output defined, stderr will be lost!");
#  }
#  else
#  {
    $params{'callback'} = \&teecallback;
#  }
	$SELF=$self;

	openTee (*STDERR, @params, \%params);
	carp "Errorlog (Version: $VERSION) Entering logmode for: $0 " if $Baseobject::verbosity >=2;
	carp ("configdir: $self->{fields}->{configdir}") if $Baseobject::verbosity >=2 ;

	return $self;
}


#
# field watcher for => SYSLOGLEVEL and OUTPUT
#
sub field {
	my $self = shift;

	#writing?
	if ($_[1]) {
		#sysloglevel
		if ($_[0]eq 'sysloglevel') {
			my @sysloglevels=qw (emerg alert crit err warning notice info debug);
			my $sysloglevel=$_[1];

			unless (grep ( /^$sysloglevel$/, @sysloglevels)) {
				$self->error("invalid sysloglevel: \"$sysloglevel\", accepted levels are: @sysloglevels" );
				return "";
			}
		} elsif ($_[0] eq 'output' ) {
			my @outputmodes = qw(stderr syslog stdout logfile logbuffer);
			map {$self->{outputmodes}->{$_} = 0} @outputmodes;

			foreach my $mode (@{$_[1]}) {
				if (grep (/^$mode$/, @outputmodes))
				{
					$self->{outputmodes}->{$mode} = 1;
				} else {
					#in case of error leave stderr intact!
					map {$self->{outputmodes}->{$_} = 0} @outputmodes;
					$self->{outputmodes}->{stderr} = 1;
					$self->error("$mode is an invalid outputmode: allowed modes are: @outputmodes");
					return "";
				}
			}
		}
	}
	return $self->SUPER::field(@_);
}


#NOTE NO self

sub teecallback {
	my $hiRes='';
	$hiRes = $SELF->hiResTime() if $SELF->{fields}->{hirestimer};

	#NOTE: MUST RETURN TRUE, otherwise nothing is printed!
	my ($fh, $stderr, @string) = @_;
	my $longmess = "";
	my $time = scalar localtime; $time =~ s/\s(\d{4})$//;

	#assignement should not be necessary, but it is..
	(@string) = map { s/\x0/\<NUL\>/g ;s/\n$//g; $_.="\n" } grep m{.}, @string;

	if ($fh eq 'syslog') {
		my $head;
		$head = $SELF->banner($fh);
		if ($head) {
			$head = "[$SELF->{hostname}:${0}] --> $head"  if ($stderr);
			syslog($SELF->{fields}->{sysloglevel}, ($head)) ;
		}
		foreach my $string (@string) {
			foreach my $substr (split (/\n/,$string)) {
				next unless ($substr);

#       syslog $priority, $format, @args
#           If $priority permits, logs ($format, @args) printed as by "printf(3V)",
#           with the addition that %m is replaced with "$!" (the latest error message).
#
				my $blocksize = $SELF->{fields}->{syslogmaxbuf};

				if (length ($substr) > $blocksize) {
					my $length = length($substr);
					for (my $progress = 0; $progress < $length; $progress += $blocksize) {
						my $chopped = 'chop cont';
						$chopped='chopstart' unless ($progress);

						my $chopstring = substr($substr,$progress,$blocksize);
						syslog($SELF->{fields}->{sysloglevel},"[Errorlog: $chopped]%s",$chopstring);
					}
				} else {
					syslog($SELF->{fields}->{sysloglevel}, "%s","${hiRes}$substr");
				}
			}
		}

		if ($Baseobject::verbosity >= 3) {
			$longmess = Errorlog::presentstackframe(' -  ') ;
      #      #syslog ignores \n, so call it for each line
			foreach my $mess (split("\n",$longmess)) {
				#remove longmessage 's tab character
				$mess =~ s/\t//g;
				syslog($SELF->{fields}->{sysloglevel},"$mess");
			}
		}

    # $fh ne syslog
	} elsif ($fh eq 'logbuffer') {
		$longmess = Errorlog::presentstackframe() if $Baseobject::verbosity >= 3;
		$SELF->{fields}->{logbuffer} .= $SELF->banner($fh) . "@string$longmess";
	} else {
		foreach my $longline (@string) {
			foreach my $line (split (/\n/, $longline)) {
				my @line;
				push @line, "$time - $line\n";
				# Leading string last!
				# add pid to stderror to distinguish this run

				my $leading = '';
				$leading="[$$] [$SELF->{hostname}:$0]  -" if $SELF->fields->{printinfo};
				$longmess = Errorlog::presentstackframe($leading) if $Baseobject::verbosity >= 3;
				if ($SELF->{fields}->{printinfo}) {
					unshift (@line, "[$SELF->{hostname}:${0}$hiRes] --> " ) if $stderr;
					unshift (@line, "[$$] " . $SELF->banner($fh));
				}

				if ($longmess) {
					$longmess =~ s/\t/ /g;
					push (@line, $longmess);
				}
				print $fh  @line ;
			}
		}
	}

	1;
}


sub hiResTime
{
	my $self=shift;
	my $tprev=$self->{tstored};
	$self->{tstored} = [gettimeofday];

	my $tdif=tv_interval ($tprev, $self->{tstored});
	my $ttotal=tv_interval ($self->{tstart}, $self->{tstored});
	my $slow='';
	$slow=" TIME POOL" if ($tdif >= $self->{fields}->{hirestimer});

	return "dt: $tdif tt: ${ttotal}$slow - ";
}

sub makelogfilename {
	my $self = shift;
	my $name = $0;

	$name =~ s/^.*\///;
	($name) = $name =~ /(^.*)\./ if $name =~ /\./;
	return "$name.log";
}


sub close {
	my $self = shift;
	close $self->field('fh') if defined(($self->field('fh')));
	$self->field('fh', undef);
}

sub DESTROY {
	my $self = shift;
	$self->close();
}


sub banner {
	my $self = shift;
	my $fh = shift;
	return "" unless $self->field('marknewrun');

	my $delim = ' -- ';
	my $pid = "";
	# NB, FIXME nog testen!
	my $time = time;
	$pid = "\n[$time - $$] " unless $fh eq 'syslog';

	my $myipaddress = $self->field('ipaddress') ;
	my $ipaddress = "";  $ipaddress="$delim$myipaddress" if $myipaddress;

	my $myuser = $self->field('user');
	my $user = "";
	$user = "$delim$myuser" if $myuser;

	my $myheader = $self->field('header') || "";
	my $header = "";
	$header = "$delim$myheader" if $myheader;

	my $string = "";
	# print a newrunmarker once (was: for each filehandle)
	unless (defined($self->{marked}->{$fh})) {
		my $timestamp = scalar(localtime(time()));
		my $marker = $self->field('newrunmarker');
		my $script = "";
		$script = $0; $script =~ s/^.*\///; $script .= $delim ;

		$string= "-- $script$timestamp $ipaddress$user$header $marker$pid";
		$self->{marked}->{$fh} = 1;
	}
	return $string;
}

sub mail {
	my $self=shift;
	my $string=join ("\n",@_);
	carp "Sending mail to: $self->{fields}->{mailto}" ;
	carp "using mailprog: $self->{fields}->{mailprogram}" if ($Baseobject::verbosity >=2 );

	unless ($self->{fields}->{mailto}) {
		carp "ERROR: No mail send, 'mailto' not set!";
		return;
	}

	my $mailhandle=new FileHandle();

	if (open($mailhandle , "|-", "$self->{fields}->{mailprogram}",
			   "-f $self->{fields}->{mailerrorsto}",
			   "$self->{fields}->{mailto}" )) {
		binmode $mailhandle, ':utf8';

		my $mail = join ("\n",
						 (
						  "To: <$self->{fields}->{mailto}>",
						  "X-mailer: perl $]",
						  "Subject: $self->{fields}->{mailsubject}($self->{fields}->{facility})",
						  "Precedence: bulk",
						  "From: $self->{fields}->{mailfrom}",
						  "Errors-To: $self->{fields}->{mailerrorsto}",
						  "Mime-Version: 1.0",
						  "Content-Type: text/plain; charset=utf-8",
						  "Content-Disposition: inline",
						  "Content-Transfer-Encoding: 8bit",
						  "",
						  "Report: (from script:$0)\n",
						  $string,
						  "",
						 ));
		print $mailhandle $mail;
		CORE::close $mailhandle ;
#    carp ("Errorlog mailed to $self->{fields}->{mailto}\n$string\n");

	} else {
		carp( "Could not open mail: $self->{fields}->{mailprogram}, $!");
		carp( "trying to mail: $string\n");
	}
}


#NOTE NO SELF
sub presentstackframe {
	my $longmess = Carp::longmess.Carp::shortmess;
	my $lead = " -";
	$lead = shift if $_[0];

	#skip all on the stack that matches the following:
	my $skip = join('|^\s*', qw ( '' at Carp:: Local:: Baseobject::stderror Errorlog:: Baseobject::debug Baseobject::verbose));
	my @stack;

	#remove our lines unless
	if ($Baseobject::verbosity<4) {
		@stack = grep(!/$skip/, split("\n",$longmess));
	} else {
		@stack=split("\n",$longmess)
	}

	return "" unless @stack;
	$longmess = $lead . join("\n$lead", @stack);
	$longmess .= "\n" unless $longmess =~ /\n$/;
}


sub overrule {
	#
	# sets verbositylevel and all the fields used by Errorlog.pm
	# the usage of an '=' assignement character will overwrite the
	# original value(s) and will prevent the verbositylevel to
	# be changed by the application, enforced by a freezeverbosity() call;
	#
	# The '|' assigment character will lead to the merging of lists if
	# the name contains a ref to a list.
	# The freezeverbosity call is not issued, so the application still can
	# change the verbositylevel
	#
	# etc/defaults_errorlog.cf   is read first
	# etc/$basename_errorlog.cf  is read second
	# if 'etc/overrule_errorlog.cf' is found this file is read last
	# and thereby taking precedence
	#
	# usage of this configfile is only indicated with verbositylevel > 1
	# or under errorconditions
	#
	my $self = shift;
	my ($basepart, $configfile);
	my $found = 0;
	my $fh = new FileHandle();
	my $configdir = $self->field('configdir');

	return unless $configdir;
	if (-d $configdir) {
		foreach $basepart ('defaults', $self->basefilename(),'overrule') {
			$configfile = "$configdir/${basepart}_errorlog.cf";
			next unless -r $configfile;
			carp "Found configfile: $configfile" if $Baseobject::verbosity >=2 ;
			unless (open ($fh,$configfile)) {
				carp "Found but could not open configfile: $configfile, $!";
				return;
			}

			local $_;
			while (<$fh>) {
				next if /^\#/;
				next if /^\s*$/;
				chomp;

				my ($name, $split, $value) = split /\s*([=\|])\s*/;
				my @values = split (/,/, $value);
#      carp $self->inspect($name, $value, $split,\@values);

				if ($name eq 'verbosity') {
					verbosity($value);
					freezeverbosity(1) if $split eq '=';
					next;
				}

				my $ref = ref($self->field($name));

				#the target expects an array
				if ($ref eq 'ARRAY') {
					if ($split eq '=') {
						# carp "overwriting with \"@values\"";
						# overwrite with config values
						$self->field($name, [@values]);
					} else {
						#merge with config values
						$self->field($name, uniquelist(@values,@{$self->field($name)}));
					}
				}

				elsif  ($ref eq '') {
					$self->field($name,$value);
				} else {
					carp "ignoring variable: $name: type \"$ref\" not (yet:) supported";
				}
			}
			warn "Errorlog (version: $VERSION): Using configfile: $configfile" 
				if $Baseobject::verbosity > 1 or $self->error();
		}
	} else {
		carp "Warning; Errorlog (version: $VERSION) configdir: \"$configdir\" does not exist";
	}
}

sub uniquelist {
	return undef unless @_;
	my %hash;
	map {$hash{$_} = 1} @_;
	return [keys (%hash)];
}

42;


=head1 NAME

Errorlog - catch stderr

=head1 DESCRIPTION


Redirects and/or duplicates stderr to several outputstreams.  Allows errors to be mailed
to the maintainer.  Also, optionally, a copy of stderr is mailed if the program has an exit
status of none-zero.

A Carp stack frame may be presented for each stderr event or at exit.


=head1 SYNOPSIS

All stderr output then is redirected or duplicated through a perl
tied filehandle to one or more of the following outputstreams:

stderr, stdout, syslog, a logfile, a logbuffer

=head1

The module is typically activated within a BEGIN block of a perl
program. This way errors at the perl interpretor level will also be
redirected, which is probably what you want.

Depending on context each flow of a program's output is optionally
preceeded by a header, showing the name of the program, a timestamp,
and environment variables: $REMOTE_USER, $REMOTE_ADDR.

Depending on context each original output line is preceeded by a PID
hostname and fullpath programname.

The module attempts to read certain configurationfiles which modify
verbosity level and various other parameters for each program or
programgroup.

Depending on the verbosity level each outputline is presented with a
stackframe for which the extern B<Carp> module is used.

Default locations of configuration and logfiles are taken from
$main::SYSCONFDIR
$main::LOCALSTATEDIR

or if that fails from
$main::sysconfdir
$main::localstatedir

if these variables exist.

The alternative is to present the 'logdir' and 'configdir' fields
in the new() call.

=head1 DEPENDENCIES

Baseobject.pm Carp.pm TeeOutput.pm

=item EXAMPLE

NOTE in the next example @prefix@ and @localstatedir@ are considered
to be modified by autotools.

#!/usr/local/bin/perl -w
# file: errorlog.pl
# Time-stamp: <2004-04-27 14:38:47 anton>

use strict;
use Errorlog;
our $prefix;
our $localstatedir;

my $log;

BEGIN {
	$prefix = "@prefix@";
	$localstatedir = "@localstatedir@";

	$log=new Errorlog({output => [qw (logfile stderr)]});
    warn "problem: ", $log->error if $log->error;
}

warn 'hi there';
$log->mail('spamming maintainer')

output on the screen (via stderr)

 [4809] -- errorlog.pl -- Tue Apr 27 14:57:52 2004  -- anton ------------------------------------------------------------
 [4809] [timewind:/home/anton/swources/test/errorlog.pl] --> hi there at /home/anton/swources/test/errorlog.pl line 19.

The logfile:

 cat /home/anton/log/test/errorlog.log

 [4809] -- errorlog.pl -- Tue Apr 27 14:57:52 2004  -- anton ------------------------------------------------------------
 [4809] hi there at /home/anton/swources/test/errorlog.pl line 19.

The output I<directed to stderr> contains the hostname and $0 (the perl
programname) for each output line. This is not the case for all other
outputstreams, as can be observed in the above example.

=head1 new()

=head1 SYNOPSIS

The redirection takes place at the moment of creation of the object, this is the moment to configure.
Another location to set these variables are the L<configuration-files>, described further on.

=over 1

=item PARAMETERS

Parameters are passed as a ref to a hash. The following name value pairs have been defined as default:

		{                    
                  'configdir',$configdir,                  # $main::SYSCONFDIR
                  'facility'=>'local0'
                  'header'=>'',                            # add your own header
                  'ipaddress'=>$ENV{'REMOTE_ADDR'},        #
                  'logbuffer'=>"",
                  'logdir'=>$logdir,                       # $main::localstatedir
                  'logfile'=>$logfile,                     # basefilename.log (foo.pl -> foo.log)
                  'mailerrorsto'=>"host_\@$self->{hostname}",
                  'mailfrom'=>"host_$self->{hostname}",
                  'mailprogram'=>'/usr/sbin/sendmail',     # please check this
                  'mailsubject'=>"ErrorLog: $0\@$self->{hostname}",  #
                  'mailto'=>'anton@uvt.nl',                # SET YOUR OWN!
                  'marknewrun'=>1,                         # toggle writing of newrunmarker
                  'newrunmarker'=>'-'x60,                  # seperator contains 60 dashes
                  'output'=>[qw (stderr)],
				  'printinfo'=>1,                          # toggle info preceeding each outputline
                  'sysloglevel'=>'info',
                  'mailonfailure'=>1,                      # if the program ends with exit status none-zero, the error buffer is mailed
                  'syslogmaxbuf'=>512+255,                 # just guessing, 1024 is too big, things get truncated
                  'user'=>$user,                           # default $ENV{'REMOTE_USER'}, or $ENV{'USER'}
		}

Some of the parameters explained in more detail:

=item logbuffer

Is typically used as a logbuffer, so a program can read it's own stderr.

print $log->field('logbuffer');

print $log->{fields}->{logbuffer};

=item output

Should contain a ref to list containing one or more of the following elements:

stderr syslog stdout logfile logbuffer

After activation stderr will be send to one ore more of the above targets.

=item mailonfailure=1

If this is specified in the configuration file, or passed with new() the program will (if still able to) mail the entire STDERR on exit status
none-zero.

Internally the 'logbuffer' mechanisme, as mentioned earlier, is used for this.


=back 

=head1 mail()

=head1 SYNOPSIS

Call the method with a string, or list of strings. The list of strings is joined with a "\n". The joined string will be sent by the L<"mailprogram"> to user L<"mailto">.

=head1 configuration-files

Configaration files are searched in the directory:
"$main::SYSCONFDIR".

If your want to make use of this features make sure you provide the
above variable in an extreme early stage like the example below.

To benefit the most from this construction it is advisable
to configure (using .configure of your package) for your
project 'foo' as something like 'prefix/etc/foo', whereas the
default typically is 'prefix/etc'.

par. ex.:
./configure --prefix=/home/anton/prefix --sysconfdir='$prefix/etc/foo'

(Without further configuration in configfiles the example will duplicate
 stderr to: stderr and syslog.)

	our $prefix;
our $sysconfdir ;

use Baseobject;
use Errorlog;

my $log;
BEGIN
{
	$prefix="@prefix@";
	$sysconfdir = "@sysconfdir@";

	$log=new Errorlog({
		output=>[qw ( stderr  syslog )],
   });
   warn "Errorlog problem:",$log->error() if ($log->error());
 }


Given a project: 'foo' and a two programs 'foo.pl' and 'bar.pl'
residing in this project, the following search order will be used for
'foo.pl'.

 etc/foo/defaults_errorlog.cf
 etc/foo/foo_errorlog.cf
 etc/foo/overrule_errorlog.cf

and similar for 'bar_errorlog.cf'

 etc/foo/defaults_errorlog.cf
 etc/foo/bar_errorlog.cf
 etc/foo/overrule_errorlog.cf

This creates the possibility to present default values for all
the files in the project, to deviate from this on an individual program base,
and to overwrite again on a projects base.

A configuration might look like this:

 etc/foo/defaults_errorlog.cf
  mailto=anton@uvt.nl
  mailonfailure=1

 etc/foo/foo_errorlog.cf
  verbosity=2

 etc/foo/bar_errorlog.cf
  verbosity|1
  output|syslog

 etc/foo/overrule_errorlog.cf
  mailprogram=/usr/bin/mail
  output|logfile,stderr
  logfile=all.log

The defaults_errorlog.cf set the default mailto variable. A program can deviate
from this by overwriting it. On failure anton@uvt.nl will be mailed automatically.

The individual foo_errorlog.cf and bar_errorlog.cf allow for different configurations
for the individual programs.

The overrule_errorlog.cf is used here to specify the values certain to be used
by all programs. If you want a group of programs to write to a single
logfile, this is the place to indicate so.

=over 1

=item = vs |

Normally the assignement character '=' will just result in a simple
assignment as you would expect.
In case of the variables 'verbosity' and 'output' the characters '='
and '|' have special meaning.

=item verbosity

If the '=' assignement is used, the verbosity is not only set to the
specified value, but it is I<froozen> to this setting. The program
itself is not allowed to changed the value, and attempts to do so will
be ignored.

In contrast, the '|' assignment will just set the value as
specified. The program is free to modify this.

=item output

In the case of output, the '=' assignement leads to a configuration with
just and only the elements specified here. Any elements the added before,
internally, or by defaults_errorlog.cf are overwritten.

The '|' assignment leads to a merge, or a logical 'or' of the current
lists of the program and the list specified here.

=back 

=head1 verbosity

Verbosity is a property of B<Baseobject>. This module uses its
settings to determine how the redirected lines are treated.

=over 1

=item verbosity = 0

Silence

=item verbosity = 1

Normal output.

=item verbosiy = 2

More verbose, considered debug level.

=item verbosity = 3

Output is presented with a B<Carp> stackframe dump. The calls made to
Errorlog.pm and its underlying mechanisms are not included in this stackframe.

=item verbosity = 4

The stackframe dump is presented including the Errorlog.pm stack. This
is seldom what you would want.


=back 

=head1 BUGS

=over 1

=item use before new

Messages like this may occur:
 Can't locate Encode/ConfigLocal.pm in @INC (@INC contains: /etc/perl /usr/local/lib/perl/5.8.4 /usr/lo
cal/share/perl/5.8.4 /usr/lib/perl5 /usr/share/perl5 /usr/lib/perl/5.8 /usr/share/perl/5.8 /usr/local/lib/site_perl .) at /usr/lib/perl/5.8/Encode.pm line 57.

To prevent this always place your use statements I<before> the BEGIN block in which 'new Errorlog ' resides:

 use strict;
 use Package_a;
 use Package_b;
 use Errorlog;
 use Package_c;

 my $log;
 BEGIN
 {
    $log=new Errorlog();
 }

 #BETTER DON'T:
 use Package_d;

=back

=head1 SEE ALSO

Baseobject Sys::Syslog Carp



