package Longoptx;
use strict;

use Getopt::Long;
use Carp;
use Baseobject;
use Exporter;

our $VERSION="1.1";

#does it need a baseobject?
our @ISA=('Baseobject' );
our @EXPORT = qw (baseconfigure);

sub new{
  my $self = bless (new Baseobject(@_),shift);
  return $self;
}

sub getoptions {
  my $self = shift;
  my $pre = shift;
  my $commands = shift;
  if ($commands) {
    $self->{refcommand} = shift (@$commands);
    $self->{commands} = shift (@$commands);
  }

  my $res;
  $self->{pre} = $pre;
  my $opt = $self->tolongopt($pre);
  $res = GetOptions(%$opt);
  if ($res) {
    $res = $self->commands() if $commands;
  }

  return $res;
}


sub tolongopt {
  my $self = shift;
  my $pre = shift;
  my $opt;

  foreach my $key (keys (%$pre))  {
    my $pvar = (@{$pre->{$key}}[0]);
    my $default = (@{$pre->{$key}}[1]);

    # if the default contains a list, the var ref is filled with ref to a sub
    if (ref $default eq 'ARRAY')    {
      $opt->{$key} =  sub { $self->selectone(@_,$pvar,$default) };
      $$pvar = @{$default}[0] if (ref $pvar eq 'SCALAR');
    } else {
      $opt->{$key} = $pvar;
      $$pvar=$default if ref $pvar eq 'SCALAR';
    }
  }
  return $opt;
}

sub commandsusage {
  my $self = shift;
  my $maxlen = 0;
  my @list;
  my $default = shift;
  my ($prog) = $0 =~ /\/([^\/]+)$/;

  print "$prog [command] [--option(s)] [files]\n\n Commands:\n";

  foreach my $key (sort keys (%{$self->{commands}}))  {
    push (@list, $key);
    $maxlen=length($key) if length($key) > $maxlen;
  }


  foreach my $key (sort keys (%{$self->{commands}})) {
    my $str = shift(@list);
    print "  $str",' 'x($maxlen - length($str)+4), "$self->{commands}->{$key}\n";
  }
  print "\n Options:\n";
}


sub usage {
  my $self = shift;
  $self->commandsusage() if exists ($self->{refcommand});
  my $pre = $self->{pre};
  my @list;
  my $maxlength = 0;
  my $string = '';

  #establish longest key length
  my $maxkeylen = 0;
  my $STRING = '[string]';

  my $space;

  foreach my $key (sort  keys (%$pre)) {
	  my ($var, $delim) = split (/([=\!+\|].*)/, $key);
	  
      my $default ='';
      unless (ref ((@{$pre->{$key}}[1]))) {
		  $default='['. @{$pre->{$key}}[1].']' if (@{$pre->{$key}}[1]);
      } else {
		  my $ref = @{$pre->{$key}}[1];
		  my $def = shift (@$ref);
		  $def ='' unless defined($def);
		  $default = "[$def] @$ref";
      }

      $delim ='' unless defined($delim);
      #remove | sign, possibly duplicate this line for each alternative
      $var =~ s/\|.*//;
      if ($delim eq '!' && ($var !~ /^help/) && ($default)) {
		  $var = "--no$var" ; $default='';
      } else {
		  $var = "--$var";
      }
      if ($delim eq '=s') {
		  $default = $STRING;
      }
      if ($delim =~/\|(\w)+/) {
		  $default = "(-$1) " .$default;
      }

      my $name = "$var $default";
      $maxlength = length($name) unless $maxlength > length($name);
      push @list, {$name => @{$pre->{$key}}[2]};
  }

  foreach my $h (@list) {
	  foreach my $key (keys %$h) {
		  $string .= " $key" . ' 'x($maxlength - length($key)+1) . "$h->{$key}\n";
	  }
  }
  print $string;
}

sub dumpoptions {
	my $self = shift;
	my $string = "";
	my $pre = shift;
	foreach my $key (keys %$pre) {
		my $pvar = (@{$pre->{$key}}[0]);
		if (ref ($pvar) eq 'SCALAR') {
			$string .= "$key value:\"$$pvar\"\n" ;
		}
	}
	$string;
}

sub selectone {
	my $self = shift;
	my ($option, $value, $pvar, $list) = @_;
	shift (@$list) unless defined(@$list[0]);
	my @match = grep /^$value.*/, @$list ;

	if (@match == 1) {
		$$pvar=$value;
	} elsif (@match >1) {
		carp "value: $value is inconclusive for list: @$list\n";
		return 0;
	} elsif (@match == 0) {
		carp  "option: $option, value: \"$value\" is incorrect: should be one of: @$list\n";
		return 0;
	}
}

sub commands {
	my $self = shift;
	my $arg = $ARGV[0];
	my @commandlist = (sort keys %{$self->{commands}});

	if ($arg) {
		my @possiblecommands = grep /^$arg.*/,@commandlist;
		unless (@possiblecommands) {
			#no carp, it insists on printing the linenumber
			print STDERR "Unrecognized command: \"$arg\", valid commands are: @commandlist\n";
			return 0;
		} else {
			if (@possiblecommands > 1) {
				print STDERR "Command not unique: \"$arg\", cannot decide between: @possiblecommands\n";
				return 0;
			} else {
				${$self->{refcommand}} = $possiblecommands[0];
				shift (@ARGV);
			}
		}
	} else {
		#no carp, dont want no line number
		print STDERR  "No command found, please choose from: @commandlist\n";
		return 0;
	}
	return 1;
}
42;
=head1 Longoptx.pm

=head1 SYNOPSIS

Provide on the fly 'usage' for your commandline parameter, fully depending on Getopt::Long.

=head1 Description

Wrapper around Getopt::Long, providing auto documentation and introduces 'commands'
in addition to the Getop::Long 'options'.

=head1 Example

   my $opt = new Longoptx();
   Getopt::Long::Configure ("bundling");
  
   my $commands = [\$command, {
      'diff' => 'show difference between the files on remote vs the current location',
      'list' => 'show the files currently in the database',
      'info' => 'show info on the files currently in the database for the specified directory [pwd]',
    }];
  
   my $options = {
    'mkdir!' => [ \$forcecreatepath,
     0,
    'force the creation of the path at the remote location if it does not exist' ] ,
    'help!'   => [ sub {$opt->usage (); exit},
  		         undef,
  		        'provides usage()' ],


=head1 DEPENDENCIES

Getopt::Long Carp Baseobject

=head1 BUGS

Documentation is incomplete, obsolete, and sometimes wrrong, etc.

=head1 AUTHOR

Anton Sluijtman, E<lt>anton@uvt.nlE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2004 by anton sluijtman

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.4 or,
at your option, any later version of Perl 5 you may have available.

=cut

# Local Variables:
# mode: perl
# mode: font-lock
# End:
