# $Id: FileTransfer.pm 8092 2013-07-16 14:28:57Z joostvb $
# $URL: https://pong.uvt.nl/its-nettel/Netwerken/Procedures/libuvt-noc-perl/lib/UvT/NOC/FileTransfer.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package UvT::NOC::FileTransfer;

use parent qw( Exporter );
our @EXPORT_OK = qw (downloadFileUsingFTP downloadFileUsingFTPS downloadFileUsingHTTPS downloadUserTrackingUsingHTTPS);

use UvT::NOC::Util::FileUtil;
use UvT::NOC::Config::Resource;
use UvT::NOC::Net::FTPSSL;
use Net::Lite::FTP;
use Net::FTP;
use LWP::UserAgent;
use XML::LibXML;

our $debug;


# Functienaam : downloadFileUsingFTP
#
# Parameters : (4x)
#	1) een resource-object met de volgende ftp client-gegevens :
#		- host
#		- username
#		- password
#		- passive (1 of 0)
#		- filedir (locatie van de file op de ftp server), indien niet meegegeven wordt de standaard dir gebruikt
#	2) naam van het op te halen bestand
#	3) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory)	 
#	4) de naam van het bestand op het lokale filesysteem	 
#
# Return : -
#
# Omschrijving :
#       Functie om een file te downloaden met behulp van ftp

sub downloadFileUsingFTP
{
	my $resource = shift;
	my $filename = shift;
	my $destinationDir = shift;
	my $destinationFilename = shift;

	#chdir($destination) or die("Cannot change to $destination : $!");

	my $host = $resource->getProperty('host')->value;
	if ($debug) { warn "Retreiving $filename from $host ...\n"; }

	eval
	{
		my $username = $resource->getProperty('username')->value;
		my $password = $resource->getProperty('password')->value;
		my $filedir = $resource->getProperty('filedir') ? $resource->getProperty('filedir')->value : undef;
		my $passive = $resource->getProperty('passive')->value;

		if ($debug) { warn "Connecting to host : $host\n"; }
		my $ftp = Net::FTP->new($host, Debug => $debug, Passive => $passive)
			or die "Cannot connect to $host";

		if ($debug) { warn "Login\n"; }
		$ftp->login($username, $password)
			or die "Cannot login : ", $ftp->message;

		if ($filedir)
		{
			if ($debug) { warn "Change to directory : $filedir\n"; }
			$ftp->cwd($filedir)
				or die "Cannot change to working directory : ", $ftp->message;
		}

		if ($debug) { warn "Get file : $filename\n"; }
		$ftp->get($filename, "$destinationDir/$destinationFilename")
			or die "Cannot get file : ", $ftp->message;

		if ($debug) { warn "Closing connection\n"; }
		$ftp->quit()
			or die "Cannot close connection";
	};
	die "Error retreiving file $filename : $@" if $@;
}


# Functienaam : downloadFileUsingFTPS
#
# Parameters : (3x)
#	1) een resource-object met de volgende ftps client-gegevens :
#		- host
#		- port (indien meegegeven wordt er standaard 21 gebruikt)
#		- username
#		- password
#		- filedir (locatie van de file op de ftp server), indien niet meegegeven wordt de standaard dir gebruikt
#	2) naam van het op te halen bestand
#	3) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory)
#	4) de naam van het bestand op het lokale filesysteem	  
#
# Return : -
#
# Omschrijving :
#       Functie om een file te downloaden met behulp van ftps

sub downloadFileUsingFTPS
{
    my $resource = shift;
    my $filename = shift;
    my $destinationDir = shift;
    my $destinationFilename = shift;

	my $host = $resource->getProperty('host')->value;
    if ($debug) { warn "Retreiving $filename from $host ...\n"; }

    eval
    {
		my $username = $resource->getProperty('username')->value;
		my $password = $resource->getProperty('password')->value;
		my $port = $resource->getProperty('port') ? $resource->getProperty('port')->value : '21';
		my $encryption = $resource->getProperty('encryption')->value;
		my $filedir = $resource->getProperty('filedir') ? $resource->getProperty('filedir')->value : undef;

		if ($debug) { warn "Connecting to host : $host\n"; }
        my $ftps = UvT::NOC::Net::FTPSSL->new($host, Port => $port, Encryption => $encryption, Debug => $debug)
        	or die "Cannot connect to $host";

        if ($debug) { warn "Login\n"; }
        $ftps->login($username, $password)
			or die "Cannot login : ", $ftps->last_message;

        if ($filedir)
        {
            if ($debug) { warn "Change to directory : $filedir\n"; }
            $ftps->cwd($filedir)
            	or die "Cannot change to working directory : ", $ftps->last_message;
        }

        if ($debug) { warn "Get file : $filename\n"; }
        $ftps->get($filename, "$destinationDir/$destinationFilename")
        	or die "Cannot get file : ", $ftps->last_message;

        if ($debug) { warn "Closing connection\n"; }
        $ftps->quit()
        	or die "Cannot close connection";
	};
	die "Error retreiving file $filename : $@" if $@;
}


# Functienaam : downloadFileUsingHTTPS
#
# Parameters : (3x)
#       1) een hash met de volgende https client-gegevens :
#		- host
#		- port
#		- realm
#		- username
#		- password
#	2) de url van het op te halen bestand
#	3) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory inclusief bestandsnaam) 
#
# Return : -
#
# Omschrijving :
#       Functie om een file te downloaden met behulp van ftp

sub downloadFileUsingHTTPS
{
	my $resource = shift;
	my $url = shift;
	my $destination = shift;

	eval
	{
		my $host = $resource->getProperty('host')->value;
		my $port = $resource->getProperty('port')->value;
		my $realm = $resource->getProperty('realm')->value;
		my $username = $resource->getProperty('username')->value;
		my $password = $resource->getProperty('password')->value;

		$ENV{HTTPS_DEBUG} = $debug;

		my $useragent = LWP::UserAgent->new();
		# enforce SSLv3 voor brakke Oracle-SSL op uvtapp.uvt.nl
		$useragent->ssl_opts( SSL_version => "SSLv3");
		$useragent->credentials($host . ":" . $port, $realm, $username, $password);
		my $request = HTTP::Request->new(GET => "$url");
		my $response = $useragent->request($request);
		
		unless ($response->is_success()) { die "Cannot get url $url : ", $response->status_line; }

		UvT::NOC::Util::FileUtil::stringToFile($destination, $response->content);
	};
        if($@)
	{
                warn "Error retreiving $url : $@\n";
	};
}


# Functienaam : downloadUserTrackingUsingHTTPS
#
# Parameters : (2x)
#       1) een resource-object met de volgende gegevens om te verbinden met de user tracking data extraction engine van ciscoworks :
#		- username
#		- password
#		- url
#		- command
#	2) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory inclusief bestandsnaam) 
#
# Return : -
#
# Omschrijving :
#       Functie om de user tracking gegevens van ciscoworks te downloaden

sub downloadUserTrackingUsingHTTPS
{
	my $resource = shift;
	my $destination = shift;

	eval
	{
		my $payload = constructUserTrackingPayload($resource);

		if ($debug) { warn "Retreive user-tracking data with payload : " . $payload->toString(); }

		my $useragent = LWP::UserAgent->new();
		my $request = HTTP::Request->new(POST => $resource->getProperty('url')->value);
		$request->content_type('text/html');
		$request->content($payload->toString());
		my $response = $useragent->request($request);

		unless ($response->is_success()) { die "Cannot get url $resource->getProperty('url')->value : ", $response->status_line; }
	
		my $output = $response->content;

    	# Strip out embedded newlines and returns if line doesn't end in tag.
    	$output =~ s/\r//g;
    	$output =~ s/([^>])\n/$1/g;

    	# Take out leading and trailing spaces
    	$output =~ s/\>\  /\>/g;
    	$output =~ s/\  \</\</g;

		UvT::NOC::Util::FileUtil::stringToFile($destination, $output);
        };
        if($@)
        {
                warn "Error retreiving usertracking data from ciscoprime : $@\n";
        };
}


# Functienaam : constructUserTrackingPayload
#
# Parameters : (1x)
#       1) een hash met de volgende gegevens om te verbinden met de user tracking data extraction engine van ciscoworks :
#		- username
#		- password
#		- url
#		- command
#
# Return : (1x)
#
#	1) payload in xml-formaat
#
# Omschrijving :
#       Functie om de payload in xml-formaat te genereren voor de data extraction engine van ciscoworks

sub constructUserTrackingPayload
{
	my $resource = shift;
	
	my $payload = XML::LibXML::Document->new();
	my $root = $payload->createElement('payload');
	$payload->setDocumentElement($root);

	my $username = $payload->createElement('username');
	$username->appendText($resource->getProperty('username')->value);
	$root->appendChild($username);

	my $password = $payload->createElement('password');
    $password->appendText($resource->getProperty('password')->value);
    $root->appendChild($password);	

	my $command = $payload->createElement('command');
    $command->appendText($resource->getProperty('command')->value);
    $root->appendChild($command);

	return $payload;
}


# Functienaam : debug
#
# Parameters : (1x)
#       1) debugwaarde, 1 = aan, 0 = uit
#
# Return : -
#
# Omschrijving :
#       Functie om debugging aan te zetten

sub debug
{
	$debug = shift;
}

1;
