# $Id: FileTransfer.pm 10340 2015-11-12 08:40:54Z hurko $
# $URL: https://pong.uvt.nl/its-nettel/Netwerken/Procedures/libuvt-noc-perl/lib/UvT/NOC/FileTransfer.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package UvT::NOC::FileTransfer;

use parent qw( Exporter );
our @EXPORT_OK = qw (downloadFileUsingFTP downloadFileUsingFTPS downloadFileUsingHTTPS downloadUserTrackingUsingHTTPS downloadPrimeClientDetailsUsingRestAPI uploadFileUsingSSH uploadFileUsingSSH);

use UvT::NOC::Util::FileUtil;
use UvT::NOC::Config::Resource;
use UvT::NOC::Net::FTPSSL;
use Net::Lite::FTP;
use Net::FTP;
use Net::SCP;
use LWP::UserAgent;
use XML::LibXML;

our $debug;


# Functienaam : downloadFileUsingFTP
#
# Parameters : (3x)
#       1) een hash de resource properties
#	2) een hash met de resourceFile properties
#	3) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory)	  
#
# Return : -
#
# Omschrijving :
#       Functie om een file te downloaden met behulp van ftp

sub downloadFileUsingFTP
{
	my $resource = shift;
	my $resourceFile = shift;
	my $destinationDir = shift;

	my $filename = $resourceFile->getProperty('sourceFileName')->value;
	my $destinationFilename = $resourceFile->getProperty('destinationFileName')->value;

	#chdir($destination) or die("Cannot change to $destination : $!");

	my $host = $resource->getProperty('host')->value;
	if ($debug) { warn "Retreiving $filename from $host ...\n"; }

	eval
	{
		my $username = $resource->getProperty('username')->value;
		my $password = $resource->getProperty('password')->value;
		my $filedir = $resource->getProperty('filedir') ? $resource->getProperty('filedir')->value : undef;
		my $passive = $resource->getProperty('passive')->value;

		if ($debug) { warn "Connecting to host : $host\n"; }
		my $ftp = Net::FTP->new($host, Debug => $debug, Passive => $passive)
			or die "Cannot connect to $host";

		if ($debug) { warn "Login with user : $username\n"; }
		$ftp->login($username, $password)
			or die "Cannot login : ", $ftp->message;

		if ($filedir)
		{
			if ($debug) { warn "Change to directory : $filedir\n"; }
			$ftp->cwd($filedir)
				or die "Cannot change to working directory : ", $ftp->message;
		}

		if ($debug) { warn "Get file : $filename\n"; }
		$ftp->get($filename, "$destinationDir/$destinationFilename")
			or die "Cannot get file : ", $ftp->message;

		if ($debug) { warn "Closing connection\n"; }
		$ftp->quit()
			or die "Cannot close connection";
	};
	die "Error retreiving file $filename : $@" if $@;
}


# Functienaam : downloadFileUsingFTPS
#
# Parameters : (3x)
#       1) een hash de resource properties
#	2) een hash met de resourceFile properties
#	3) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory)	  
#
# Return : -
#
# Omschrijving :
#       Functie om een file te downloaden met behulp van ftps

sub downloadFileUsingFTPS
{
	my $resource = shift;
	my $resourceFile = shift;
	my $destinationDir = shift;

	my $filename = $resourceFile->getProperty('sourceFileName')->value;
	my $destinationFilename = $resourceFile->getProperty('destinationFileName')->value;

	my $host = $resource->getProperty('host')->value;
	if ($debug) { warn "Retreiving $filename from $host ...\n"; }

	eval
	{
		my $username = $resource->getProperty('username')->value;
		my $password = $resource->getProperty('password')->value;
		my $port = $resource->getProperty('port') ? $resource->getProperty('port')->value : '21';
		my $encryption = $resource->getProperty('encryption')->value;
		my $filedir = $resource->getProperty('filedir') ? $resource->getProperty('filedir')->value : undef;

		if ($debug) { warn "Connecting to host : $host\n"; }
		my $ftps = UvT::NOC::Net::FTPSSL->new($host, Port => $port, Encryption => $encryption, Debug => $debug)
        	or die "Cannot connect to $host";

        	if ($debug) { warn "Login with user : $username\n"; }
        	$ftps->login($username, $password)
			or die "Cannot login : ", $ftps->last_message;

        	if ($filedir)
        	{
            		if ($debug) { warn "Change to directory : $filedir\n"; }
            			$ftps->cwd($filedir)
            		or die "Cannot change to working directory : ", $ftps->last_message;
        	}

        	if ($debug) { warn "Get file : $filename\n"; }
        		$ftps->get($filename, "$destinationDir/$destinationFilename")
        	or die "Cannot get file : ", $ftps->last_message;

        	if ($debug) { warn "Closing connection\n"; }
        		$ftps->quit()
        	or die "Cannot close connection";
	};
	die "Error retreiving file $filename : $@" if $@;
}


# Functienaam : downloadFileUsingHTTPS
#
# Parameters : (3x)
#       1) een hash de resource properties
#	2) een hash met de resourceFile properties
#	3) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory inclusief bestandsnaam) 
#
# Return : -
#
# Omschrijving :
#       Functie om een file te downloaden met behulp van https

sub downloadFileUsingHTTPS
{
	my $resource = shift;
	my $resourceFile = shift;
	my $destinationDir = shift;

	my $url = $resourceFile->getProperty('sourceFileName')->value;
	my $destinationFilename = $resourceFile->getProperty('destinationFileName')->value;

	eval
	{
		my $host = $resource->getProperty('host')->value;
		my $port = $resource->getProperty('port')->value;
		my $realm = $resource->getProperty('realm')->value;
		my $username = $resource->getProperty('username')->value;
		my $password = $resource->getProperty('password')->value;
		my $ssl_opts = $resource->getProperty('ssl_opts') ? $resource->getProperty('ssl_opts')->value : undef;

		$ENV{HTTPS_DEBUG} = $debug;

		my $useragent = LWP::UserAgent->new();

		# enforce ssl options (bv voor SSLv3 voor brakke Oracle-SSL op uvtapp.uvt.nl)
		if ($ssl_opts)
		{
			if ($debug) { warn "Set SSL opts : $ssl_opts\n"; }
			$useragent->ssl_opts( SSL_version => $ssl_opts);
		}

		$useragent->credentials($host . ":" . $port, $realm, $username, $password);
		my $request = HTTP::Request->new(GET => "$url");
		my $response = $useragent->request($request);
		
		unless ($response->is_success()) { die "Cannot get url $url : ", $response->status_line; }

		UvT::NOC::Util::FileUtil::stringToFile("$destinationDir$destinationFilename", $response->content);
	};
        if($@)
	{
                warn "Error retreiving $url : $@\n";
	};
}


# Functienaam : downloadUserTrackingUsingHTTPS
#
# Parameters : (3x)
#       1) een hash de resource properties
#	2) een hash met de resourceFile properties
#	3) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory inclusief bestandsnaam) 
#
# Return : -
#
# Omschrijving :
#       Functie om de user tracking gegevens van ciscoworks te downloaden

sub downloadUserTrackingUsingHTTPS
{
	my $resource = shift;
	my $resourceFile = shift;
	my $destinationDir = shift;

	my $url = $resourceFile->getProperty('sourceFileName')->value;
	my $destinationFilename = $resourceFile->getProperty('destinationFileName')->value;

	eval
	{
		my $payload = constructUserTrackingPayload($resource);

		if ($debug) { warn "Retreive user-tracking data with payload : " . $payload->toString(); }

		my $useragent = LWP::UserAgent->new();
		my $request = HTTP::Request->new(POST => $resource->getProperty('url')->value);
		$request->content_type('text/html');
		$request->content($payload->toString());
		my $response = $useragent->request($request);

		unless ($response->is_success()) { die "Cannot get url $resource->getProperty('url')->value : ", $response->status_line; }
	
		my $output = $response->content;

    	# Strip out embedded newlines and returns if line doesn't end in tag.
    	$output =~ s/\r//g;
    	$output =~ s/([^>])\n/$1/g;

    	# Take out leading and trailing spaces
    	$output =~ s/\>\  /\>/g;
    	$output =~ s/\  \</\</g;

		UvT::NOC::Util::FileUtil::stringToFile("$destinationDir$destinationFilename", $output);
        };
        if($@)
        {
                warn "Error retreiving usertracking data from ciscoprime : $@\n";
        };
}


# Functienaam : downloadPrimeClientDetailsUsingRestAPI
#
# Parameters : (2x)
#       1) een hash de resource properties
#       2) een hash met de resourceFile properties
#	3) de lokatie waar het bestand heen moet op het lokale filesysteem (de directory inclusief bestandsnaam) 
#
# Return : -
#
# Omschrijving :
#       Functie om de Client Details van Prime Infrastructure (in XML vorm) te downloaden

sub downloadPrimeClientDetailsUsingRestAPI
{
	my $resource = shift;
	my $resourceFile = shift;
	my $destinationDir = shift;

	my $url = $resource->getProperty('url')->value;
	my $username = $resource->getProperty('username')->value;
	my $password = $resource->getProperty('password')->value;
	my $params = $resourceFile->getProperty('command')->value;
	my $maxResults = $resourceFile->getProperty('maxPageResults')->value;

	my $destinationFilename = $resourceFile->getProperty('destinationFileName')->value;

	eval
	{
		my $destination = "results.xml";

		my $firstPage = getPrimeClientDetailsPage($url, $username, $password, $params, $maxResults, 0);
		my $total = $firstPage->findnodes('/queryResponse/@count')->string_value;
		my $pageLast = $firstPage->findnodes('/queryResponse/@last')->string_value;

		my $totalResults = $firstPage;

		while ($pageLast < $total - 1)
		{
			my $page = getPrimeClientDetailsPage($url, $username, $password, $params, $maxResults, $pageLast + 1);
			$pageLast = $page->findnodes('/queryResponse/@last')->string_value;

			my $totalResultsRoot = $totalResults->getDocumentElement();
			foreach my $entity ($page->findnodes('queryResponse/entity'))
			{
				$totalResultsRoot->appendChild($entity);
			}
		}

		UvT::NOC::Util::FileUtil::stringToFile("$destinationDir$destinationFilename", $totalResults->toString(1));
        };
        if($@)
        {
                warn "Error retreiving data from ciscoprime rest api : $@\n";
        };
}


# Functienaam : getPrimeClientDetailsPage
#
# Parameters : (2x)
#       1) url van de rest-api
#       2) username voor communicatie met de rest-api
#		3) password voor communicatie met de rest-api
#		4) html parameters om de aanroep naar de rest-api te construeren
#		5) max results per pagina
#		6) first results geeft aan welke pagina
#
# Return : client details output can prime infrastructure in xml formaat
#
# Omschrijving :
#       Functie om de Client Details van Prime Infrastructure (in XML vorm) gepagineert te downloaden

sub getPrimeClientDetailsPage
{
        my $url = shift;
        my $username = shift;
        my $password = shift;
        my $params = shift;
        my $max_results = shift;
        my $first_result = shift;

        $url = $url . $params . "&.firstResult=$first_result" . "&.maxResults=$max_results";

        if ($debug) { warn "Retreiving data from $url\n"; }

        my $useragent = LWP::UserAgent->new();
        $useragent->ssl_opts( verify_hostname => 0);

        my $request = HTTP::Request->new(GET => $url);
        $request->authorization_basic($username, $password);

        my $response = $useragent->request($request);
        unless ($response->is_success()) { die "Cannot get url $url : ", $response->status_line; }

        my $parser = XML::LibXML->new();
        my $result = $parser->load_xml(string => $response->content, no_blanks => 1);

        return $result;
}


# Functienaam : constructUserTrackingPayload
#
# Parameters : (1x)
#       1) een hash met de volgende gegevens om te verbinden met de user tracking data extraction engine van ciscoworks :
#		- username
#		- password
#		- url
#		- command
#
# Return : (1x)
#
#	1) payload in xml-formaat
#
# Omschrijving :
#       Functie om de payload in xml-formaat te genereren voor de data extraction engine van ciscoworks

sub constructUserTrackingPayload
{
	my $resource = shift;
	
	my $payload = XML::LibXML::Document->new();
	my $root = $payload->createElement('payload');
	$payload->setDocumentElement($root);

	my $username = $payload->createElement('username');
	$username->appendText($resource->getProperty('username')->value);
	$root->appendChild($username);

	my $password = $payload->createElement('password');
	$password->appendText($resource->getProperty('password')->value);
	$root->appendChild($password);	

	my $command = $payload->createElement('command');
	$command->appendText($resource->getProperty('command')->value);
	$root->appendChild($command);

	return $payload;
}


# Functienaam : uploadFileUsingSSH
#
# Parameters : (4x)
#	1) een resource-object met de volgende ssh client-gegevens :
#		- host
#		- username
#		- filedir (locatie van de file op de ssh server), indien niet meegegeven wordt de standaard dir gebruikt
#	2) de naam en directory van het te pushen bestand op het lokale filesysteem	 
#
# Return : -
#
# Omschrijving :
#       Functie om een file te uploaden met behulp van SSH (op basis van SSH-KEYS)

sub uploadFileUsingSSH
{
	my $resource = shift;
	my $sourceFile = shift;

	my $host = $resource->getProperty('host')->value;
	if ($debug) { warn "Pushing $sourceFile to $host ...\n"; }

	eval
	{
		my $username = $resource->getProperty('username')->value;
		my $destinationDir = $resource->getProperty('destinationDir') ? $resource->getProperty('destinationDir')->value : undef;

		if ($debug) { warn "Connecting to host : $host\n"; }
		my $scp = Net::SCP->new($host)
			or die "Cannot connect to $host";

		if ($debug) { warn "Login with user : $username\n"; }
		$scp->login($username)
			or die "Cannot login : ", $scp->{errstr};

		if ($destinationDir)
		{
			if ($debug) { warn "Change to destination directory : $destinationDir\n"; }
			$scp->cwd($destinationDir)
				or die "Cannot change to working directory : ", $scp->{errstr};
		}

		if ($debug) { warn "Put file : $sourceFile\n"; }
		$scp->put($sourceFile)
			or die "Cannot get file : ", $scp->{errstr};

		if ($debug) { warn "Closing connection\n"; }
		$scp->quit()
			or die "Cannot close connection";
	};
	die "Error pushing file " . $sourceFile . ": $@" if $@;
}


# Functienaam : debug
#
# Parameters : (1x)
#       1) debugwaarde, 1 = aan, 0 = uit
#
# Return : -
#
# Omschrijving :
#       Functie om debugging aan te zetten

sub debug
{
	$debug = shift;
}

1;
