#!/usr/bin/perl

# $Id: $
# $URL: $

use strict;
use warnings FATAL => 'all';

use UvT::NOC::Net::SNMP;
use UvT::NOC::Util::StringUtil;
use Data::Dumper;

my $debug = 0;

unless ( defined($ARGV[0]) && defined($ARGV[1]) )
{
    print "Usage : $0 <mgmt ip asa> <ip l2l peer> [--debug]\n\n";
    exit 0;
}

my $asa_ip = $ARGV[0];
my $givenL2LPeer = $ARGV[1];

if ( defined($ARGV[2]) )
{
    if ($ARGV[2] eq '--debug')
    {
        $debug = 1;
        warn "-- DEBUGGING MODE ---\n\n";
    }
}

# TODO : snmp community gegevens uit global config halen
my $snmp_community = "hkbu4kma";
my $snmp_options   = "-Oq";

my $cikeTunRemoteValue      = "1.3.6.1.4.1.9.9.171.1.2.3.1.7";
my $cipSecTunIkeTunnelIndex = "1.3.6.1.4.1.9.9.171.1.3.2.1.2";
my $cipSecTunInOctets       = "1.3.6.1.4.1.9.9.171.1.3.2.1.26";
my $cipSecTunOutOctets      = "1.3.6.1.4.1.9.9.171.1.3.2.1.39";

my @identities = split(/\n/, walkInfoBySNMPv2($asa_ip, $snmp_community, $cikeTunRemoteValue, $snmp_options));
my @sessionIndexes = split(/\n/, walkInfoBySNMPv2($asa_ip, $snmp_community, $cipSecTunIkeTunnelIndex, $snmp_options));
my @sessionInOctets = split(/\n/, walkInfoBySNMPv2($asa_ip, $snmp_community, $cipSecTunInOctets, $snmp_options));
my @sessionOutOctets = split(/\n/, walkInfoBySNMPv2($asa_ip, $snmp_community, $cipSecTunOutOctets, $snmp_options));


# loop through the peer list and store it's ip address in a hash as the key with the peer index as the value

my %peerIndexes = ();
foreach (@identities)
{
    my ($peerOid, $peerIp) = split(/ /, $_);

    $peerIp = removeQuotes($peerIp);
    my ($peerBaseOid, $peerIndex) = split(/\.([^.]+$)/, $peerOid);
    $peerIndexes{$peerIp} = $peerIndex;

    if ($debug) { warn "Peer $peerIp found with peer index $peerIndex\n"; }
}


# a peer can have multiple sessions
# finde the session indexes of the peers and store them in a hash with peer index as the key
# and an array of sessions indexes as the value

my %sessionsPerPeerIndex = ();
foreach (@sessionIndexes)
{
    my ($sessionOid, $peerIndex) = split(/ /, $_);
    my ($sessionBaseOid, $sessionIndex) = split(/\.([^.]+$)/, $sessionOid);

    if ($debug) { warn "Session index $sessionIndex for peer index $peerIndex\n"; }

    # initialize an empty array for the indexes
    my $peerSessions = [];

    # if peerIndex exists -> load sessions
    if (exists $sessionsPerPeerIndex{$peerIndex})
    {
        $peerSessions = $sessionsPerPeerIndex{$peerIndex};
    }

    # add session index to session array
    push(@$peerSessions, $sessionIndex);

    # store session array into hash
    $sessionsPerPeerIndex{$peerIndex} = $peerSessions;
}


# store the rx bytes per session index into a hash with the session index as the key and the rx bytes as the value

my %sessionRxBytes = ();
foreach (@sessionInOctets)
{
    my ($sessionOid, $bytes) = split(/ /, $_);
    my ($sessionBaseOid, $sessionIndex) = split(/\.([^.]+$)/, $sessionOid);

    if ($debug) { warn "Session index $sessionIndex found with $bytes received bytes\n"; }

    $sessionRxBytes{$sessionIndex} = $bytes;
}


# store the tx bytes per session index into a hash with the session index as the key and the tx bytes as the value

my %sessionTxBytes = ();
foreach (@sessionOutOctets)
{
    my ($sessionOid, $bytes) = split(/ /, $_);
    my ($sessionBaseOid, $sessionIndex) = split(/\.([^.]+$)/, $sessionOid);

    if ($debug) { warn "Session index $sessionIndex found with $bytes transmitted bytes\n"; }

    $sessionTxBytes{$sessionIndex} = $bytes;
}


# loop through the peer list and sum the total received and transmitted bytes
# store it in a hash with the ip address of the peer as the key
# and a hash with with the rx/tx bytes as the value

my %peerIOBytes = ();
foreach my $peer ( keys %peerIndexes )
{
    my $peerIndex = $peerIndexes{$peer};
    if ($debug) { warn "Totalling rx/tx bytes for $peer with peer index $peerIndex\n"; }

    my @peerSessions = @{$sessionsPerPeerIndex{$peerIndex}};
    my ($totalRxBytes, $totalTxBytes) = 0;
    foreach my $peerSession (@peerSessions)
    {
        if ($debug) { warn "\tSession index $peerSession found\n"; }
        my $sessionRxBytes = $sessionRxBytes{$peerSession};
        my $sessionTxBytes = $sessionTxBytes{$peerSession};

        if ($debug) { warn "\t\t rx = $sessionRxBytes\n\t\t tx = $sessionTxBytes\n"; }
        $totalRxBytes += $sessionRxBytes{$peerSession};
        $totalTxBytes += $sessionTxBytes{$peerSession};
    }

    if ($debug) { warn "Total rx = $totalRxBytes\n"; }
    if ($debug) { warn "Total tx = $totalTxBytes\n\n"; }

    my %ioBytes = ( rx => $totalRxBytes, tx => $totalTxBytes );
    $peerIOBytes{$peer} = \%ioBytes;
}

# print "--- peer IO Bytes ---\n";
# print Dumper(%peerIOBytes);

# loop through peerIOBytes hash and print rx/tx info per peer (only in debug mode)

if ($debug)
{
    foreach my $peer ( keys %peerIOBytes )
    {
        my $ioBytes = $peerIOBytes{$peer};

        my $rxBytes = $ioBytes->{rx};
        my $txBytes = $ioBytes->{tx};

        warn "IO statictics for peer $peer :\n\t rx=$rxBytes \n\t tx=$txBytes \n\n";
    }
}


# return the rx and tx bytes for the given peer ip in a cacti friendly format

if (exists $peerIOBytes{$givenL2LPeer})
{
    if ($debug) { warn "I/O bytes for peer $givenL2LPeer found :\n"; }
    my $ioBytes = $peerIOBytes{$givenL2LPeer};

    my $rxBytes = $ioBytes->{rx};
    my $txBytes = $ioBytes->{tx};

    if ($debug) { warn "\t rx=$rxBytes\n\t tx=$txBytes\n\n"; }

    if ($debug) { print "<SCRIPT OUTPUT>\n" };
    print "rx:$rxBytes tx:$txBytes";
    if ($debug) { print "\n</SCRIPT OUTPUT>\n" };
}


if ($debug) { warn "End of script reached\n"; }



__END__

=head1 NAME

asa_ipsec_peer_traffic - Retrieve rx/tx counters of an ipsec peer

=head1 SYNOPSIS

asa_ipsec_peer_traffic ASA PEER [OPTION...]

=head1 DESCRIPTION

Reads rx and tx counters of an ipsec peer on the asa.
The output is formatted for cacti.

=head1 OPTIONS

--debug			debug mode

=head1 AUTHOR

Jurgen van den Hurk <jvdhurk@uvt.nl>

=head1 COPYRIGHT AND LICENSE

Copyright 2016, Tilburg University

This program is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
