<?php

/**
 * Authentication module which acts as an A-Select client
 *
 * @author Wessel Dankers, Tilburg University
 */
class sspmod_aselectdirect_Auth_Source_ASelectDirect extends SimpleSAML_Auth_Source {
	private $app_id = 'simplesamlphp';
	private $server_id;
	private $server_url;
	private $private_key;

	/**
	 * Constructor for this authentication source.
	 *
	 * @param array $info  Information about this authentication source.
	 * @param array $config  Configuration.
	 */
	public function __construct($info, $config) {
		/* Call the parent constructor first, as required by the interface. */
		parent::__construct($info, $config);

		$cfg = SimpleSAML_Configuration::loadFromArray($config,
			'Authentication source ' . var_export($this->authId, true));

		$this->app_id = $cfg->getString('app_id');
		$this->server_id = $cfg->getString('server_id');
		$this->server_url = $cfg->getString('server_url');
		$this->private_key = $cfg->getString('private_key', null);
	}

	/**
	 * Initiate authentication.
	 *
	 * @param array &$state  Information about the current authentication.
	 */
	public function authenticate(&$state) {
		$state['aselectdirect::authid'] = $this->authId;
		$id = SimpleSAML_Auth_State::saveState($state, 'aselectdirect:login', true);

		try {
			$app_url = SimpleSAML_Module::getModuleURL('aselectdirect/credentials.php', array('ssp_state' => $id));
			$as_url = $this->request_authentication($app_url);

			SimpleSAML_Utilities::redirect($as_url);
		} catch(Exception $e) {
			// attach the exception to the state
			SimpleSAML_Auth_State::throwException($state, $e);
		}
	}

	private function base64_signature($str) {
		$key = openssl_pkey_get_private($this->private_key);
		if($key === false)
			throw new SimpleSAML_Error_Exception("Unable to load private key: ".openssl_error_string());
		if(!openssl_sign($str, $sig, $key))
			throw new SimpleSAML_Error_Exception("Unable to create signature: ".openssl_error_string());
		openssl_pkey_free($key);
		return base64_encode($sig);
	}

	private static function split_query($in) {
		$pairs = explode('&', $in);
		$ret = array();
		foreach($pairs as $pair) {
			$keyval = explode('=', $pair, 2);
			if(count($keyval) < 2)
				return null;
			$ret[urldecode($keyval[0])] = urldecode($keyval[1]);
		}
		return $ret;
	}

	private static function decode_attributes($base64) {
		$blob = base64_decode($base64, true);
		if($blob === false)
			return null;
		$pairs = explode('&', $blob);
		$ret = array();
		foreach($pairs as $pair) {
			$keyval = explode('=', $pair, 2);
			if(count($keyval) < 2)
				return null;
			$key = urldecode($keyval[0]);
			$key = preg_replace('/\[\]$/', '', $key);
			$val = urldecode($keyval[1]);
			$ret[$key][] = $val;
		}
		return $ret;
	}

	private static $curl_options = array(
		CURLOPT_BINARYTRANSFER => true,
		CURLOPT_FAILONERROR => true,
		CURLOPT_RETURNTRANSFER => true,
		CURLOPT_CONNECTTIMEOUT => 1,
		CURLOPT_TIMEOUT => 5,
		CURLOPT_USERAGENT => "simpleSAMLphp",
	);

	private function create_aselect_url($request, $parameters) {
		$parameters['request'] = $request;
		$parameters['a-select-server'] = $this->server_id;
		if(!is_null($this->private_key)) {
			$signable = '';
			foreach(array('a-select-server', 'app_id', 'app_url', 'aselect_credentials', 'rid') as $p)
				if(array_key_exists($p, $parameters))
					$signable .= $parameters[$p];
			$parameters['signature'] = $this->base64_signature($signable);
		}
		return SimpleSAML_Utilities::addURLparameter($this->server_url, $parameters);
	}

	private function call_aselect($request, $parameters) {
		$url = $this->create_aselect_url($request, $parameters);

		$curl = curl_init($url);
		if($curl === false)
			throw new SimpleSAML_Error_Exception("Unable to create CURL handle");

		if(!curl_setopt_array($curl, self::$curl_options))
			throw new SimpleSAML_Error_Exception("Unable to set CURL options: ".curl_error($curl));

		$res = curl_exec($curl);
		$err = curl_error($curl);

		curl_close($curl);

		if($res === false)
			throw new SimpleSAML_Error_Exception("Unable to retrieve URL: $error");

		$res = self::split_query($res);

		// message is only available with some A-Select server implementations
		if($res['result_code'] != '0000')
			if(array_key_exists('message', $res))
				throw new SimpleSAML_Error_Exception("Unable to contact SSO service: result_code=".$res['result_code']." message=".$res['message']);
			else
				throw new SimpleSAML_Error_Exception("Unable to contact SSO service: result_code=".$res['result_code']);
		unset($res['result_code']);

		return $res;
	}

	public function request_authentication($app_url) {
		$res = $this->call_aselect('authenticate',
			array('app_id' => $this->app_id, 'app_url' => $app_url));

		$as_url = $res['as_url'];
		unset($res['as_url']);

		return SimpleSAML_Utilities::addURLparameter($as_url, $res);
	}

	public function verify_credentials($server_id, $credentials, $rid) {
		$server_id = $this->server_id;

		if($server_id != $this->server_id)
			throw new SimpleSAML_Error_Exception("Acquired server ID ($server_id) does not match configured server ID (".$this->server_id.")");

		$res = $this->call_aselect('verify_credentials',
			array('aselect_credentials' => $credentials, 'rid' => $rid));

		if(array_key_exists('attributes', $res))
			$res['attributes'] = self::decode_attributes($res['attributes']);

		return $res;
	}
}
