<?php

/**
 * Authentication module which interacts with the iPort AuthBroker
 *
 * @author Wessel Dankers, Tilburg University
 */
class sspmod_authbroker_Auth_Source_AuthBroker extends sspmod_core_Auth_UserPassBase {
	private $authbroker_host;
	private $authbroker_port;
	private $uid_attribute;

	/**
	 * Constructor for this authentication source.
	 *
	 * @param array $info  Information about this authentication source.
	 * @param array $config  Configuration.
	 */
	public function __construct($info, $config) {
		/* Call the parent constructor first, as required by the interface. */
		parent::__construct($info, $config);

		$cfg = SimpleSAML_Configuration::loadFromArray($config,
			'Authentication source ' . var_export($this->authId, true));

		$this->authbroker_host = $cfg->getString('authbroker_host');
		$this->authbroker_port = $cfg->getString('authbroker_port', 4033);
		$this->uid_attribute = $cfg->getString('uid_attribute', 'urn:mace:dir:attribute-def:uid');
	}

	/**
	 * Check credentials
	 *
	 * @param string $uid  The username the user wrote
	 * @param string $pwd  The password the user wrote
	 * @param array $sasl  Associative array of SASL options
	 * @return array       Associative array with the users attributes
	 */
	public function login($uid, $pwd, array $sasl = NULL) {
		assert('is_string($uid)');
		assert('is_string($pwd)');

		$doc = new DOMDocument('1.0', 'UTF-8');
		$xrep = $doc->createElement('xrep');
		$doc->appendChild($xrep);
		$logonRequest = $doc->createElement('logonRequest');
		$xrep->appendChild($logonRequest);

		$logonRequest->appendChild($doc->createElement('userID', $uid));
		$logonRequest->appendChild($doc->createElement('password', $pwd));
		$logonRequest->appendChild($doc->createElement('requestSource', 'user'));

		$msg = $doc->saveXML();

		$socket = socket_create(AF_INET, SOCK_STREAM, SOL_TCP);
		if($socket === false)
			throw new Exception("Unable to create socket for authbroker: ".strerror(socket_last_error()));

		$host = $this->authbroker_host;
		$port = $this->authbroker_port;
		if(!socket_connect($socket, $host, $port))
			throw new Exception("Unable to connect to authbroker host $host port $port: ".socket_strerror(socket_last_error($socket)));

		while($msg != '') {
			$r = socket_write($socket, $msg);
			if($r === false)
				throw new Exception("Unable to write to authbroker socket to host $host port $port: ".socket_strerror(socket_last_error($socket)));
			$msg = substr_replace($msg, '', 0, $r);
		}
		socket_shutdown($socket, 1);

		for(;;) {
			$ret = socket_read($socket, 65536);
			if($ret === false)
				throw new Exception("Unable to read from authbroker socket to host $host port $port: ".socket_strerror(socket_last_error($socket)));
			if($ret == '')
				break;
			$msg .= $ret;
		}
		socket_close($socket);

		$doc = new DOMDocument();
		$doc->loadXML($msg);

		$xpath = new DOMXPath($doc);
		if($xpath->query('/xrep/logonResponse/group[.="extusr"]')->length)
			return array($this->uid_attribute => array(strtolower($uid)));

		throw new SimpleSAML_Error_Error('WRONGUSERPASS');
	}
}
