# $Id: Construct.pm 39349 2013-05-03 13:39:28Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/xml-construct/lib/XML/LibXML/Construct.pm $

package XML::LibXML::Construct;

use Carp qw(confess);

sub construct {
	confess("Don't call XML::LibXML::Construct::construct; use either XML::LibXML->construct or \$doc->construct");
}

package XML::LibXML;

require XML::LibXML;

sub construct {
	my $self = shift;
	my $doc = new XML::LibXML::Document('1.0', 'UTF-8');
	my $root = $doc->construct(@_);
	$doc->setDocumentElement($root);
	return $doc;
}

package XML::LibXML::Document;

use Scalar::Util qw(reftype);

sub construct {
	my $self = shift;
	my $name = shift;
	utf8::upgrade($name);
	my $node = $self->createElement($name);
	foreach my $child (@_) {
		if(ref $child) {
			my $type = reftype($child);
			if($type eq 'ARRAY') {
				$node->appendChild($self->construct(@$child));
			} elsif($type eq 'HASH') {
				while(my ($key, $val) = each(%$child)) {
					utf8::upgrade($key);
					utf8::upgrade($val);
					$node->setAttribute($key, $val);
				}
			} else {
				$node->appendChild($child);
			}
		} else {
			utf8::upgrade($child);
			$node->appendText($child);
		}
	}
	return $node;
}

1;

__END__

=pod

=encoding utf8

=head1 NAME

XML::LibXML::Construct - quickly construct XML from perl datastructures

=head1 SYNOPSIS

 use XML::LibXML::Construct;
 my $doc = XML::LibXML->construct(foo => [bar => "baz"]);
 my $node = $doc->construct(quux => {attr => "value"}, "xyzzy");

=head1 INTRODUCTION

This package injects a method "construct" in both XML::LibXML as well as
XML::LibXML::Document. This method recursively creates a DOM XML element
based on the given arguments: the first argument is the name of the root
element, subsequent arguments are child nodes.

For the child nodes, a plain scalar will create an XML text node. A hash
reference will create attributes to the element. An array reference will
create child elements, calling construct recursively. A reference to an
XML::LibXML::Node is added as-is. You can pass as many child nodes of any
type as you like, in any order.

If the construct method is invoked on an XML::LibXML::Document, the created
element will be returned. If it is invoked on XML::LibXML, a
XML::LibXML::Document will be created, filled and returned.

Great care is taken to utf8::upgrade() every string before passing it to
XML::LibXML to guarantee predictable behavior for any text string.

=head1 BUGS

Namespaces are respected but otherwise unsupported; set up your document
with namespaces beforehand if necessary. All created nodes are in the
default (empty) namespace.

If the official XML::LibXML package ever starts shipping a construct method
of its own, it will collide rather fatally.

=head1 EXAMPLES

This:

 my $doc = XML::LibXML->construct(foo => [bar => "baz"]);

will create a document that looks like this when you call $doc->toString:

 <?xml version="1.0" encoding="UTF-8"?>
 <foo><bar>baz</bar></foo>

This:

 my $doc = XML::LibXML->construct(foo => {a => 1, b => 2}, [bar => "baz"]);

will create a document that looks like this when you call $doc->toString:

 <?xml version="1.0" encoding="UTF-8"?>
 <foo a="1" b="2"><bar>baz</bar></foo>

This:

 my $doc = XML::LibXML->construct(foo => [bar => "baz", {a => 1, b => 2}]);

or:

 my $doc = XML::LibXML->construct(foo => [bar => {a => 1}, "baz", {b => 2}]);

will create a document that looks like this when you call $doc->toString:

 <?xml version="1.0" encoding="UTF-8"?>
 <foo><bar a="1" b="2">baz</bar></foo>

=head1 AUTHOR

Wessel Dankers <wsl@uvt.nl>

=head1 COPYRIGHT

Copyright © 2013 Tilburg University.

Licensed under the GPLv3 license or (at your option) any later version.
