# $Id: Construct.pm 40904 2014-01-28 13:13:16Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/xml-construct/lib/XML/LibXML/Construct.pm $

use strict;
use warnings FATAL => 'all';

package XML::LibXML::Construct;

use Carp qw(confess);

sub construct {
	confess("Don't call XML::LibXML::Construct::construct; use either XML::LibXML->construct or \$doc->construct");
}

sub deconstruct {
	confess("Don't call XML::LibXML::Construct::deconstruct; use either \$node->deconstruct or \$doc->deconstruct");
}

package XML::LibXML;

require XML::LibXML;

sub construct {
	my $self = shift;
	my $doc = new XML::LibXML::Document('1.0', 'UTF-8');
	my $root = $doc->construct(@_);
	$doc->setDocumentElement($root);
	return $doc;
}

package XML::LibXML::Document;

use Carp qw(confess);
use Scalar::Util qw(reftype blessed);

sub construct {
	my $self = shift;
	my $name = shift;
	utf8::upgrade($name);
	my $node = $self->createElement($name);
	foreach my $child (@_) {
		my $type = reftype($child);
		if(defined $type) {
			if(my $blessing = blessed($child)) {
				confess("Objects of type '$blessing' not supported")
					unless $child->isa('XML::LibXML::Node');
				$node->appendChild($child);
			} elsif($type eq 'ARRAY') {
				$node->appendChild($self->construct(@$child));
			} elsif($type eq 'HASH') {
				while(my ($key, $val) = each(%$child)) {
					utf8::upgrade($key);
					utf8::upgrade($val);
					$node->setAttribute($key, $val);
				}
			} else {
				confess("Reference type '$type' not supported");
			}
		} else {
			my $c = $child;
			utf8::upgrade($c);
			$node->appendText($c);
		}
	}
	return $node;
}

sub deconstruct { shift->documentElement->deconstruct }

package XML::LibXML::DocumentFragment;

sub deconstruct { map { $_->deconstruct } shift->childNodes }

package XML::LibXML::Node;

sub deconstruct { return }

package XML::LibXML::Comment;

# For some reason, XML::LibXML::Comment's parent class is XML::LibXML::Text,
# even though the manpage seems to suggest it's XML::LibXML::Node.

sub deconstruct { return }

package XML::LibXML::Namespace;

# For some reason, XML::LibXML::Namespace does not have a parent class,
# even though the manpage seems to suggest it's XML::LibXML::Node.

sub deconstruct { return }

package XML::LibXML::Text;

sub deconstruct { shift->nodeValue }

package XML::LibXML::Attr;

sub deconstruct {
	my $self = shift;
	my $name = $self->nodeName;
	return if $name =~ /^xml/;
	$name =~ s/^.*://;
	return $name, $self->nodeValue;
}

package XML::LibXML::Element;

sub deconstruct {
	my $self = shift;
	my $name = $self->nodeName;
#	$name =~ s/^.*://;
	my @element = ($name);
	my %attrs = map { $_->deconstruct } $self->attributes;
	push @element, \%attrs if %attrs;
	push @element, map { $_->deconstruct } $self->childNodes;
	return \@element;
}

1;

__END__

=pod

=encoding utf8

=head1 NAME

XML::LibXML::Construct - quickly construct XML from perl datastructures

=head1 SYNOPSIS

 use XML::LibXML::Construct;

 my $doc = new XML::LibXML::Document("1.0", "UTF-8");
 my $element = $doc->construct(foo => {attr => "value"}, "bar");

 my $doc = XML::LibXML->construct(html => [body => "xyzzy"]);

=head1 INTRODUCTION

=head2 construct($name, ...)

This package injects a method "construct" in both XML::LibXML as well as
XML::LibXML::Document. This method recursively creates a DOM XML element
based on the given arguments: the first argument is the name of the root
element, subsequent arguments are child nodes.

For the child node parameters, four types are accepted:

=over

=item *

A plain scalar will be converted to text and added as an XML text node.

=item *

A hash reference will be converted to a list of attributes and added to the
element that is being created.

=item *

An array reference will be converted to a single element (by calling
construct recursively) and added as a child node.

=item *

A reference to an XML::LibXML::Node is added as-is to the list of child
nodes of the element that is being created.

=back

You can pass as many child nodes of any type as you like, in any order.

If the construct method is invoked on an XML::LibXML::Document, the created
element will be returned. If it is invoked on XML::LibXML, a
XML::LibXML::Document will be created, filled and returned.

Great care is taken to utf8::upgrade() every string before passing it to
XML::LibXML to guarantee predictable behavior for any text string.

=head2 $node->deconstruct

Converts a DOM node back to a tree of perl data elements. Any comments,
processing instructions, doctype declarations and namespace nodes are
discarded.

You can invoke this method on a document or an element. (In fact you can
invoke it on any node, but the result might not be what you expect.)

Note that elements are returned as arrayrefs.

Document fragments are returned as a list of results, so be sure to call
it in a list context.

=head1 BUGS

Namespaces are respected but otherwise unsupported; set up your document
with namespace declarations beforehand if necessary. Any namespace prefixes
are passed through to XML::LibXML unmodified.

If the official XML::LibXML package ever starts shipping a construct or
deconstruct method of its own, it will collide rather fatally.

=head1 EXAMPLES

This:

 my $doc = XML::LibXML->construct(p => "Hello ", [strong => "world"], "!");

will create a document that looks like this when you call $doc->toString:

 <?xml version="1.0" encoding="UTF-8"?>
 <p>Hello <strong>world</strong>!</p>

This:

 my $doc = XML::LibXML->construct(foo => {a => 1, b => 2}, [bar => "baz"]);

will create a document that looks like this when you call $doc->toString:

 <?xml version="1.0" encoding="UTF-8"?>
 <foo a="1" b="2"><bar>baz</bar></foo>

Both:

 my $doc = XML::LibXML->construct(foo => [bar => "baz", {a => 1, b => 2}]);

and:

 my $doc = XML::LibXML->construct(foo => [bar => {a => 1}, "baz", {b => 2}]);

will create a document that looks like this when you call $doc->toString:

 <?xml version="1.0" encoding="UTF-8"?>
 <foo><bar a="1" b="2">baz</bar></foo>

=head1 AUTHOR

Wessel Dankers <wsl@uvt.nl>

=head1 COPYRIGHT

Copyright © 2013 Tilburg University.

Licensed under the GPLv3 license or (at your option) any later version.
