use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::Request::Root;

use URI::Escape;
use Xyzzy::Util qw(uri_escape_auto uri_unescape_plus utf8_testandset utf8_testandset_inplace);
use Xyzzy::MIME;

use Clarity -self;

field in;
field out;
field err;
field env;

sub generic_param() {
	my $params = shift;
	if(@_) {
		my $values = $params->{shift @_}
			or return;
		# possible situations here:
		# ['x'] (most common, a single value exists)
		# [] (no value available)
		# ['x', 'y'] (multi-valued)
		# [undef] (query-string without value: /foo?x)
		return wantarray
			? @$values
			: @$values
				? ($values->[0] // '')
				: undef;
	} else {
		return wantarray
			? keys %$params
			: [keys %$params];
	}
}

sub param { unshift @_, $self->params; &generic_param }
sub url_param { unshift @_, $self->url_params; &generic_param }
sub body_param { unshift @_, $self->body_params; &generic_param }
sub file_param { unshift @_, $self->file_params; &generic_param }
sub cookie { unshift @_, $self->cookies; &generic_param }

field params => sub { shift->parse_params; return };
field url_params => sub { shift->parse_params; return };
field body_params => sub { shift->parse_params; return };
field file_params => sub { shift->parse_params; return };

sub parse_generic_params() {
	my ($str, $encoding) = @_;
	my %params;
	foreach my $pair (split(/[&;]/, $str)) {
		next if $pair eq '';
		my $off = index($pair, '=');

		my ($key, $val);

		if($off == -1) {
			$key = $pair;
		} else {
			$key = substr($pair, 0, $off);
			$val = substr($pair, $off + 1);

			eval { $val = uri_unescape_plus($val) };
			warn $@ if $@;
			if($encoding) {
				eval { $val = $encoding->decode($val, Encode::FB_CROAK) };
				warn $@ if $@;
			} else {
				utf8_testandset_inplace($val);
			}
		}

		eval { $key = uri_unescape_plus($key) };
		warn $@ if $@;
		if($encoding) {
			eval { $key = $encoding->decode($key, Encode::FB_CROAK) };
			warn $@ if $@;
		} else {
			utf8_testandset_inplace($key);
		}

		push @{$params{$key}}, $val;
	}
	return \%params;
}

sub mime_to_encoding() {
	my $part = shift;
	return undef unless defined $part;
	my $charset = $part->head->mime_attr('content-type.charset');
	return undef unless defined $charset;
	my $encoding = Encode::find_encoding($charset);
	return undef unless ref $encoding;
	return $encoding;
}

sub mime_to_file() {
	my $part = shift;
	return undef unless defined $part;
	my $file = $part->bodyhandle->open('r');
	my $encoding = mime_to_encoding($part);
	if($encoding && $encoding->perlio_ok) {
		my $name = $encoding->name;
		binmode($file, ":encoding($name)");
	} else {
		binmode($file);
	}
	return $file;
}

sub upload {
	return map { mime_to_file($_) } $self->file_param(@_)
		if wantarray;
	return mime_to_file(scalar $self->file_param(@_));
}

sub uploadInfo {
	return $self->file_param(@_);
}

sub tmpFileName { undef }

our $body_max = 1048576;

sub join_params() {
	my %params;
	foreach my $p (@_) {
		while(my ($key, $vals) = each(%$p)) {
			push @{$params{$key}}, @$vals;
		}
	}
	return \%params;
}

sub parse_params {
	my $url_params = parse_generic_params($self->query_string);
	$self->url_params($url_params);
	my $body_type = $self->content_type;
	my $body_params = {};
	my $file_params = {};
	if(defined $body_type) {
		my $body = $self->request_body;
		if($body_type eq 'application/x-www-form-urlencoded') {
			die "request body too large for application/x-www-form-urlencoded\n"
				if $self->request_body_large;
			my $encoding = mime_to_encoding($body);
			$body_params = parse_generic_params($body->bodyhandle->as_string, $encoding);
		} elsif($body_type eq 'multipart/form-data') {
			foreach my $part ($body->parts) {
				my $head = $part->head;
				my $name = $head->mime_attr('content-disposition.name');
				next unless defined $name;
				if(defined($head->mime_attr('content-disposition.filename'))) {
					push @{$file_params->{$name}}, $part;
				} else {
					my $bodyhandle = $part->bodyhandle;
					my $size = $bodyhandle->size;
					die "value too large for non-file form-data parameter\n"
						if $size > $body_max;
					my $str = $bodyhandle->as_string;
					if(my $encoding = mime_to_encoding($part)) {
						eval { $str = $encoding->decode($str, Encode::FB_CROAK) };
						warn $@ if $@;
					} else {
						utf8_testandset_inplace($str);
					}
					push @{$body_params->{$name}}, $str;
				}
			}
		}
	}
	$self->body_params($body_params);
	$self->file_params($file_params);
	$self->params(join_params(\%$body_params, \%$url_params));
}

field cookies => sub {
	my $jar = shift->env->{HTTP_COOKIE};
	return {} unless defined $jar;
	my $header = Xyzzy::Header->parse($jar);
	my @cookies = $header->attributes;
	my %cookies;
	for(my $i = 0; $i < @cookies; $i += 2) {
		my ($key, $val) = @cookies[$i, $i+1];
		next unless defined $val;
		eval { $val = uri_unescape($val) };
		warn $@ if $@;
		utf8_testandset_inplace($val);
		push @{$cookies{$key}}, $val;
	}
	return \%cookies;
};

field request_body => sub {
	my $self = shift;
	my $fh = $self->request_body_handle; 
	return undef unless defined $fh;

	my $parser = new Xyzzy::MIME::Parser;
	$parser->use_inner_files(1);
	return $parser->parse($fh);
};

sub DESTROY {
	my $body = $self->{request_body} or return;
	$body->purge;
}

field request_body_large => undef;

field request_body_handle => sub {
	my $self = shift;
	my $content_type = $self->http('Content-Type');
	my $body = '';
	foreach my $hdr ($self->http) {
		if($hdr eq 'Content-Type') {
			$content_type = $self->http($hdr);
		} else {
			$body .= "$hdr: " . $self->http($hdr) . "\n";
		}
	}
	return undef unless defined $content_type;
	$body = "Content-Type: $content_type\n$body\n";

	my $in = $self->in;
	while(length($body) < $body_max) {
		my $r = read($in, $body, $body_max - length($body), length($body));
		die "While reading input from client: $!\n"
			unless defined $r;
		return new IO::File(\$body, '+<') if $r == 0;
	}

	$self->request_body_large(1);

	# create a temp file, write $body into it, and copy the rest of the input
	open(my $fh = new IO::File, '+>', undef)
		or die "Can't open temporary file: $!\n";
	for(;;) {
		$fh->print($body) or die "Writing to tempfile: $!\n";
		my $r = $in->read($body, $body_max);
		die "While reading input from client: $!\n"
			unless defined $r;
		last if $r == 0;
	}
	$fh->seek(0, 0);

	return $fh;
};

field user_agent => sub { utf8_testandset(shift->env->{HTTP_USER_AGENT}) };
field path_info => sub { utf8_testandset(shift->env->{PATH_INFO}) // '' };
field server_name => sub { utf8_testandset(shift->env->{SERVER_NAME}) // 'localhost' };
field server_software => sub { utf8_testandset(shift->env->{SERVER_SOFTWARE}) // 'xyzzy' };
field server_port => sub { int(shift->env->{SERVER_PORT} // 80) };
field server_protocol => sub { shift->env->{SERVER_PROTOCOL} // 'HTTP/1.0' };
field remote_ident => sub { utf8_testandset(shift->env->{REMOTE_IDENT}) };
field remote_user => sub { utf8_testandset(shift->env->{REMOTE_USER}) };
field auth_type => sub { shift->env->{AUTH_TYPE} };
field request_uri => sub { shift->env->{REQUEST_URI} // confess("REQUEST_URI missing from CGI info") };
field request_method => sub { shift->env->{REQUEST_METHOD} // 'GET' };
field query_string => sub { shift->env->{QUERY_STRING} // '' };
field script_name => sub { utf8_testandset(shift->env->{SCRIPT_NAME}) // confess("SCRIPT_NAME missing from CGI info") };
field script_filename => sub { utf8_testandset(shift->env->{SCRIPT_FILENAME}) // confess("SCRIPT_FILENAME missing from CGI info") };

field path_translated => sub {
	my $self = shift;
	return $self->script_filename . $self->path_info;
};

field remote_host => sub {
	my $self = shift;
	my $env = $self->env;
	return $env->{REMOTE_HOST}
		// $env->{REMOTE_ADDR}
		// confess("REMOTE_HOST missing from CGI info");
};

field remote_addr => sub {
	my $self = shift;
	my $env = $self->env;
	return $env->{REMOTE_ADDR}
		// confess("REMOTE_ADDR missing from CGI info");
};

stub Accept;
stub raw_cookie;

field referer => sub {
	return shift->http('Referer');
};

field user_name => sub {
	my $self = shift;
	my $env = $self->env;
	return utf8_testandset($self->http('From')) // $self->remote_ident // $self->remote_user;
};

field virtual_host => sub {
	my $self = shift;
	my $host = $self->http('X-Forwarded-Host')
		// $self->http('Host')
		// $self->server_name;
	$host =~ s/:\d+$//;
	return $host;
};

field virtual_port => sub {
	my $self = shift;
	if(my $host = $self->http('X-Forwarded-Host') // $self->http('Host')) {
		if($host =~ /:(\d+)$/) {
			return int($1);
		} else {
			return $self->protocol eq 'https' ? 443 : 80;
		}
	} else {
		return $self->server_port;
	}
};

field param_string => sub {
	my $self = shift;
	return join('&', map {
		my $key = $_;
		map { uri_escape_auto($key).'='.uri_escape_auto($_) } $self->param($key)
	} $self->param);
};

field protocol => sub {
	my $self = shift;
	return 'https' if $self->https eq 'on' || $self->server_port == 443;
	my ($protocol) = split(qr{/}, $self->server_protocol);
	return lc($protocol);
};

field content_type => sub {
	my $self = shift;
	my $mime = $self->request_body;
	return undef unless defined $mime;
	return $mime->effective_type;
};

field base_url => sub {
	my $self = shift;
	my $protocol = $self->protocol;
	my $host = $self->virtual_host;
	my $port = $self->server_port;
	my $url = "$protocol://$host";
	$url .= ":$port"
		unless ($protocol eq 'http' && $port == 80)
			|| ($protocol eq 'https' && $port == 443);
	return $url;
};

field url_path => sub {
	my $self = shift;
	my $p = $self->request_uri;
	my $i = index($p, '?');
	return $i == -1 ? $p : substr($p, 0, $i);
};
*path_url = \&url_path;

field url => sub {
	my $self = shift;
	return $self->base_url . $self->url_path;
};

field self_path => sub {
	my $self = shift;
	my $query = $self->param_string;
	my $url = $self->url_path;
	$url .= '?'.$query if $query ne '';
	return $url;
};

field self_url => sub {
	my $self = shift;
	return $self->base_url . $self->self_path;
};

field script_path => sub {
	my $self = shift;
	my $p = $self->path_info;
	my $u = $self->url_path;
	return $u if $p eq '';
	my $r = uri_unescape($u);
	$r =~ s{/+}{/}g;
	return undef unless substr($r, - length($p)) eq $p;
	my @u = split(qr{(/+)}, $u);
	my @p = split(qr{(/+)}, $p);
	splice(@u, 1 - @p);
	return join('', @u);
};

field script_url => sub {
	my $self = shift;
	my $path = $self->script_path;
	return undef unless defined $path;
	return $self->base_url . $path;
};

sub http {
	my $env = $self->env;
	if(defined(my $key = shift)) {
		return $env->{$key} if substr($key, 0, 5) eq 'HTTP_';
		$key =~ tr/-/_/;
		$key = uc $key;
		return $env->{"HTTP_$key"} // $env->{$key};
	} else {
		my @fields =
			map { s/^HTTP_//; tr/_/-/; s/(\w+)/\u\L$1/gi; $_ }
			grep { substr($_, 0, 5) eq 'HTTP_' }
			sort keys %$env;
		return wantarray ? @fields : \@fields;
	}
}

sub https {
	my $env = $self->env;
	return lc($env->{HTTPS} // 'off') unless @_;
	if(defined(my $key = shift)) {
		return $env->{$key} if substr($key, 0, 5) eq 'HTTPS_';
		$key =~ tr/-/_/;
		$key = uc $key;
		return $env->{"HTTPS_$key"};
	} else {
		my @fields =
			map { s/^HTTPS_//; tr/_/-/; s/(\w+)/\u\L$1/gi; $_ }
			grep { substr($_, 0, 6) eq 'HTTPS_' }
			sort keys %$env;
		return wantarray ? @fields : \@fields;
	}
}
