use utf8;
use strict;
use warnings FATAL => 'all';

$0 = "xyzzy (initializing)";

$SIG{HUP} = sub {};

use Time::HiRes qw(clock_gettime CLOCK_PROCESS_CPUTIME_ID CLOCK_MONOTONIC);

my $wall_proc_start = clock_gettime(CLOCK_MONOTONIC)
	- clock_gettime(CLOCK_PROCESS_CPUTIME_ID);

my $class;

BEGIN {
	$class = 'Xyzzy';

	use Sys::Syslog qw(openlog closelog syslog);

	openlog('xyzzy', 'nofatal,pid', Sys::Syslog::LOG_DAEMON);

	sub log_warnings {
		my $fmt = $class eq 'Xyzzy' ? '%s' : "($class) \%s";
		if(-t STDERR) {
			STDERR->autoflush(0);
			foreach my $line (split("\n", $_[0])) {
				$line =~ s/\s+$//;
				next unless $line;
				print STDERR '['.localtime()."] [$class] $line\n";
				$line =~ s/\t/    /g;
				syslog(Sys::Syslog::LOG_WARNING, $fmt, $line);
			}
			STDERR->flush;
			STDERR->autoflush(1);
		} else {
			foreach my $line (split("\n", $_[0])) {
				$line =~ s/\s+$//;
				next unless $line;
				$line =~ s/\t/    /g;
				syslog(Sys::Syslog::LOG_WARNING, $fmt, $line);
			}
		}
	}

	$SIG{__WARN__} = \&log_warnings;
}

use FCGI;
use IO::Handle;

use Xyzzy;
use Xyzzy::Header;
use Xyzzy::Status;
use Xyzzy::Request::Root;

sub concat {
	return undef if grep !defined, @_;
	return join('/', @_);
}

my $file = $ARGV[0];

my $benchmark;
my $numreqs;
my $handler;
my $hupped;

sub load_config {
	my $cfg = new Xyzzy;
	$cfg->include($file);
	$class = ref $cfg;
	$benchmark = $cfg->benchmark;
	$handler = $cfg->handler;
}

sub reload_config {
	undef $hupped;
	eval { load_config() };
	if($@) {
		warn "While reloading configuration from $file:\n";
		warn $@;
		warn "Proceeding using previous configuration\n";
		$0 = "$class (reload failed)";
	} else {
		warn "Configuration reloaded\n";
		$0 = "$class (reloaded)";
	}
}

eval {
	die "Usage: xyzzy <configfile>\n"
		unless $file;

	load_config();

	$SIG{HUP} = sub { $hupped = 1 };

	$0 = "$class (new)";

	my $fcgi = FCGI::Request(new IO::Handle, new IO::Handle, new IO::Handle, {}, 0, FCGI::FAIL_ACCEPT_ON_INTR);

	if($benchmark) {
		my $cpu_proc_init = clock_gettime(CLOCK_PROCESS_CPUTIME_ID);
		my $wall_proc_init = clock_gettime(CLOCK_MONOTONIC) - $wall_proc_start;

		printf STDERR "initial: wall=\%.3fs cpu=\%.3fs (\%.2f\%\%)\n", $wall_proc_init, $cpu_proc_init, 100.0 * $cpu_proc_init / $wall_proc_init;
	}

	for(;;) {
		$SIG{__WARN__} = \&log_warnings;
		reload_config() while $hupped;
		$SIG{HUP} = sub { $hupped = 1 };
		if($! = -$fcgi->Accept) {
			next if $!{EINTR};
			die "FCGI: accept(): $!\n";
		}
		$numreqs++;
		$0 = "$class (serving #$numreqs)";

		my $wall_req_start = clock_gettime(CLOCK_MONOTONIC);
		my $cpu_req_start = clock_gettime(CLOCK_PROCESS_CPUTIME_ID);

		my ($in, $out, $err, $env);
		if($fcgi->IsFastCGI) {
			($in, $out, $err) = $fcgi->GetHandles;
			$env = $fcgi->GetEnvironment;
		} else {
			($in, $out, $err) = (*STDIN{IO}, *STDOUT{IO}, *STDERR{IO});
			*STDIN = new IO::File('/dev/null', '<');
			*STDOUT =
			*STDERR = new IO::File('/dev/null', '>');
			$env = {%ENV};
			%ENV = (PATH => delete $env->{PATH} // '/usr/local/bin:/usr/bin:/bin');
		}

		my $output = eval {
			my $ctx = new Xyzzy::Request::Root(in => $in, out => $out, err => $err, env => $env);
			my $res = eval { $handler->handle($ctx) };
			if($res) {
				die "not a Xyzzy::Response object\n"
					unless UNIVERSAL::isa($res, 'Xyzzy::Response');
			} else {
				die "no response from handler\n" unless $@;
				die $@ unless UNIVERSAL::isa($@, 'Xyzzy::Response');
				$res = $@;
				undef $@;
			}

			$res->as_cgi;
		};
		if($@) {
			warn $@;
			my $status = new Xyzzy::Status(500);
			$output = $status->response->as_cgi;
		}

		$0 = "$class (output)";

		foreach(@$output) { $out->write($_) }

		$fcgi->Flush;
		$fcgi->Finish;

		if($benchmark) {
			my $cpu_req_total = clock_gettime(CLOCK_PROCESS_CPUTIME_ID) - $cpu_req_start;
			my $wall_req_total = clock_gettime(CLOCK_MONOTONIC) - $wall_req_start;

			printf STDERR "request: wall=\%.3fs cpu=\%.3fs (\%.2f\%\%)\n", $wall_req_total, $cpu_req_total, 100.0 * $cpu_req_total / $wall_req_total;
		}

		$0 = "$class (idle #$numreqs)";
	}
};

my $err = $@;

if($benchmark) {
	my $cpu_proc_total = clock_gettime(CLOCK_PROCESS_CPUTIME_ID);
	my $wall_proc_total = clock_gettime(CLOCK_MONOTONIC) - $wall_proc_start;

	printf STDERR "process: wall=\%.3fs cpu=\%.3fs (\%.2f\%\%)\n", $wall_proc_total, $cpu_proc_total, 100.0 * $cpu_proc_total / $wall_proc_total;
}

if($err) {
	warn $err;
	exit(1);
}
