use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::Cookie;

use Xyzzy::Util qw(uri_escape_auto parse_time);
use Xyzzy::Header;

use Clarity -self;

field name;
field value;
field parameters => {};

{
my %boolean = (secure => undef, discard => undef, httponly => undef);
my %case = (httponly => 'HttpOnly');

sub headerize() {
	my @headerized;

	while(@_) {
		my $key = lc(shift);
		my $val = shift;
		if(exists $boolean{$key}) {
			next unless $val;
			undef $val;
		}
		$key =~ s{([a-z]+)}{$case{$1} // ucfirst($1)}eg;
		push @headerized, $key, $val;
	}

	return wantarray ? @headerized : \@headerized;
}

sub setparameters {
	my $parameters = $self->parameters;
	while(@_) {
		my $key = lc(shift);
		$parameters->{$key} = shift;
	}
}

*setparameter = *setparameters;

sub unsetparameters {
	delete @{$self->parameters}{map { lc } @_};
}

*unsetparameter = *unsetparameters;
}

sub new {
	my $name = shift;
	my $value = shift;

	$self = super(name => $name, value => $value);
	$self->setparameter(Version => 1, @_);
	return $self;
}

{
my @months = qw(Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
my @days = qw(Sun Mon Tue Wed Thu Fri Sat);

sub expire {
	my $time = shift;

	if(defined $time) {
		$time = parse_time($time);
		my $date;
		if($time) {
			my @date = gmtime(time + $time);
			$date = sprintf('%s, %02d-%s-%04d %02d:%02d:%02d GMT',
				$days[$date[6]], $date[3], $months[$date[4]], 1900+$date[5],
				$date[2], $date[1], $date[0]);
		} else {
			$time = 0;
			$date = 'Thu, 01-Jan-1970 00:00:00 GMT';
		}
		$self->setparameters(Expires => $date, 'Max-Age' => $time);
	} else {
		$self->unsetparameters(qw(expires max-age));
	}
}
}

sub as_header {
	my %values;

	return new Xyzzy::Header(
		uri_escape_auto($self->name) => uri_escape_auto($self->value),
		headerize(%{$self->parameters}));
}

sub toString { return $self->as_header->toString };

use overload
	'""' => \&toString,
	fallback => 1;

__END__

=pod

=encoding utf8

=head1 Xyzzy::Cookie

Cookie management class for the Xyzzy framework.

A cookie object consists of a name, a value, and several attributes (such
as Path, Secure, etc).

Normal attributes will show up in the output as Foo=bar but some attributes
are marked as booleans, which means that their value controls whether they
show up in the output at all (and if they do, it's without a value).

Cookie expiry is handled seperately, see the expire() method.

=head2 new($name, $value, ...attributes...)

Create a new Cookie object with the given name and value and any attributes
you pass. Example:

 # create a https-only cookie called ‘session’ with value ‘foo’
 my $cookie = new Xyzzy::Cookie(session => 'foo', Secure => 1);

Consider using the bakecookie() method in Xyzzy::Document instead, as it
will set up the cookie with some useful defaults.

=head2 expire($interval)

Set the lifetime for this cookie, expressed as a Xyzzy interval string.
The value ‘0’ means ‘delete this cookie’. The value C<undef> means ‘keep
this cookie for the duration of the browser session’. Example:

 # tell the cookie to expire after one hour:
 $cookie->expire('1h');

 # tell the cookie to expire when the browser session ends:
 $cookie->expire(undef);

 # delete the cookie from the browser:
 $cookie->expire(0);
 
=head2 setparameters($key1, $value1, ...$key2, $value2, ...)

Set the named cookie parameter(s) to the specified values.
This method is also available as ‘setparameter’ for convenience.
Example:

 # create a site-wide cookie
 $cookie->setparameter(Path => '/');

 # apply security measures
 $cookie->setparameters(Secure => 1, HttpOnly => 1);

=head2 unsetparameters($key1...)

Remove the specified cookie parameter(s) from the cookie.

This method is also available as ‘unsetparameter’ for convenience.

=head2 as_header()

Return the complete cookie as a Xyzzy::Header object, suitable for use in
HTTP output.

=cut
