use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::Document::Styled;

use Xyzzy::Document::XHTML -self;

sub build {
	my $dtd = $self->createInternalSubset('html',
		'-//W3C//DTD XHTML 1.1//EN',
		'http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd');

	my $html = $self->createElementNS('http://www.w3.org/1999/xhtml', 'html');
	$html->setAttribute('xml:lang', 'en');
	$self->setDocumentElement($html);

	my $head = $self->createElement('head');
	$head->setAttribute('id', 'head');
	$html->appendChild($head);

	my $body = $self->createElement('body');
	$body->setAttribute('id', 'body');
	$html->appendChild($body);

    $self->addTemplates('base');

	$self->hook_next;

	foreach($self->findnodes('/*|/*/*|/*/head//*')) {
		$_->removeAttribute('id')
			if $_->hasAttributes
	}
}

our %structuralnodes;
@structuralnodes{qw(html head body div table tr ul ol dl form select)} = ();

sub killnodes() {
	foreach my $node (@_) {
		my $type = ref $node;
		if($type eq 'XML::LibXML::Text') {
			my $parent = $node->parentNode;
			if(defined $parent && exists $structuralnodes{$parent->localName}) {
				my $text = $node->data;
				$parent->removeChild($node)
					unless $text =~ /\S/;
			}
#		} elsif($type eq 'XML::LibXML::Comment') {
#			$node->parentNode->removeChild($node)
		} else {
			killnodes($node->childNodes)
		}
	}
}

sub getChunks {
	my ($file, $cachethis) = @_;
	my $cfg = $self->cfg;
	my $templatecache = $cfg->templatecache;

	if(exists $templatecache->{$file}) {
		return $templatecache->{$file};
	} else {
#		print STDERR '['.localtime, '] [notice] ', ref $self, " parsing $file\n";
		my $parser = $self->xmlParser;
		my $newdoc = $parser->parse_file($file);

		my @nodes = grep { $_->hasAttributes } $newdoc->documentElement->childNodes;
		if($cachethis) {
			killnodes(@nodes);
			$templatecache->{$file} = \@nodes
		}
		return \@nodes;
	}
}

sub getChunk {
	my ($file, $id) = @_;
	my $cfg = $self->cfg;
	my $fragmentcache = $cfg->fragmentcache;

	my $fragment;
	if(exists $fragmentcache->{$file}) {
		$fragment = $fragmentcache->{$file}{$id};
	} else {
		my $nodes = $self->getChunks($file, 1);
		my %fragments = map {
				my $i = $_->getAttribute('id');
				$i ? ($i, $_) : ()
			} @$nodes;
		$fragmentcache->{$file} = \%fragments
			if $cfg->cache_stylesheets;
		$fragment = $fragments{$id};
	}
	die "Fragment '$id' not in '$file'\n"
		unless defined $fragment;

	$fragment = $fragment->cloneNode(1);
	$self->xml->adoptNode($fragment);

	our $expr ||= new XML::LibXML::XPathExpression('.//*[@id]');
	my %ids;
	foreach my $node ($fragment->findnodes($expr)) {
		$ids{$node->getAttribute('id')} = $node;
		$node->removeAttribute('id');
	}
	return $fragment, \%ids;
}

sub addChunks {
	my $cachethis = shift;

	foreach my $file (@_) {
		my $nodes = $self->getChunks($file, $cachethis);
		
		foreach my $node (@$nodes) {
			my $id = $node->getAttribute('id');
			my $victim = $self->getElementById($id);
			next unless defined $victim;
			my $newnode = $node->cloneNode(1);
			$self->xml->adoptNode($newnode);
			$victim->replaceNode($newnode);
		}
	}
}

sub getFragment {
	my $dir = $self->cfg->fragmentdir;
	my $file = shift;
	return $self->getChunk("$dir/$file.xml", @_);
}

sub addTemplates {
	my $cfg = $self->cfg;
	my $dir = $cfg->templatedir;
	$self->addChunks($cfg->cache_stylesheets, map { "$dir/$_.xml" } @_);
}

sub addContent {
	my $dir = $self->cfg->contentdir;
	$self->addChunks(0, map { "$dir/$_.xml" } @_);
}
