#! /usr/bin/perl

use strict;
use warnings FATAL => 'all';

use Test::More;
use Test::Xyzzy::Eval;

BEGIN { use_ok('Xyzzy::Cookie') or BAIL_OUT('need Xyzzy::Cookie to run') }

new_ok('Xyzzy::Cookie') or BAIL_OUT('need a Xyzzy::Cookie object to run');

foreach my $r (
	[zxnrbl => 'session=zxnrbl; Version=1'],
	["\xEB" => 'session=%EB; Version=1'],
	["\x{3042}" => 'session=%E3%81%82; Version=1'],
) {
	my ($value, $expected, @extra) = @$r;
	my $cookie = new_ok('Xyzzy::Cookie', [session => $value, @extra]);
	carefully { is($cookie->toString, $expected, "string representation of session cookie") };
}

do {
	my $cookie = new Xyzzy::Cookie(session => 'DEADBEEF', BLAH => 'oiNk');
	$cookie->setparameter(foo1 => 'bar1');
	$cookie->setparameters(foo2 => 'bar2');
	$cookie->setparameters(foo3 => 'bar3', foo4 => 'bar4');
	$cookie->setparameter(foo5 => 'bar5', foo6 => 'bar6');
	like($cookie->toString, qr/Foo1=bar1/, "setparameter works for a single value");
	like($cookie->toString, qr/Foo2=bar2/, "setparameters works for a single value");
	like($cookie->toString, qr/Foo3=bar3/, "setparameters works for multiple values");
	like($cookie->toString, qr/Foo4=bar4/, "setparameters works for multiple values");
	like($cookie->toString, qr/Foo5=bar5/, "setparameter works for multiple values");
	like($cookie->toString, qr/Foo6=bar6/, "setparameter works for multiple values");
	$cookie->setparameter(foo1 => 'bar7');
	like($cookie->toString, qr/Foo1=bar7/, "setparameter correctly overwrites values");
	unlike($cookie->toString, qr/Foo1=bar1/, "setparameter correctly overwrites values");

	$cookie->unsetparameter('foo1');
	unlike($cookie->toString, qr/Foo1/, "unsetparameter correctly deletes attributes");
	like($cookie->toString, qr/Foo2/, "unsetparameter correctly deletes attributes");
	$cookie->unsetparameters('foo2');
	unlike($cookie->toString, qr/Foo2/, "unsetparameters correctly deletes attributes");
	$cookie->unsetparameters('foo2', 'foo3');
	unlike($cookie->toString, qr/Foo2/, "unsetparameters correctly deletes multiple attributes");
	unlike($cookie->toString, qr/Foo3/, "unsetparameters correctly deletes multiple attributes");
	$cookie->unsetparameter('foo4', 'foo5');
	unlike($cookie->toString, qr/Foo4/, "unsetparameter correctly deletes multiple attributes");
	unlike($cookie->toString, qr/Foo5/, "unsetparameter correctly deletes multiple attributes");
	$cookie->unsetparameter('nonexistent');
	unlike($cookie->toString, qr/Nonexistent/, "unsetparameter correctly handles nonexistent attributes");

	like($cookie->toString, qr/Blah/, "setparameter works case-insensitively for key names");
	like($cookie->toString, qr/oiNk/, "setparameter works case-sensitively for key values");

	$cookie->unsetparameter('blAh');
	unlike($cookie->toString, qr/blah/i, "unsetparameter works case-insensitively");
	unlike($cookie->toString, qr/oiNk/, "unsetparameter works case-insensitively");
};

do {
	my $cookie = new Xyzzy::Cookie(session => 'DEADBEEF', Secure => 1, HttpOnly => 1);
	like($cookie->toString, qr/\bSecure(?:;|$)/i, "Secure is a boolean parameter");
	like($cookie->toString, qr/\bHttpOnly(?:;|$)/i, "HttpOnly is a boolean parameter");
	like($cookie->toString, qr/\bSecure(?:;|$)/, "Secure is properly capitalized");
	like($cookie->toString, qr/\bHttpOnly(?:;|$)/, "HttpOnly is properly capitalized");
};

do {
	my $cookie = new Xyzzy::Cookie(session => 'DEADBEEF', Secure => undef, HttpOnly => undef);
	unlike($cookie->toString, qr/Secure/i, "Secure is left out when false");
	unlike($cookie->toString, qr/HttpOnly/i, "HttpOnly is left out when false");
};

do {
	my $cookie = new Xyzzy::Cookie(session => 'DEADBEEF');
	unlike($cookie->toString, qr/Expires/i, "cookies are session based by default (Expires)");
	unlike($cookie->toString, qr/Max-Age/i, "cookies are session based by default (Max-Age)");
};

do {
	my $cookie = new Xyzzy::Cookie(session => 'DEADBEEF');
	$cookie->expire('1h');
	like($cookie->toString, qr/Expires=/i, "cookies can be expired (Expires)");
	like($cookie->toString, qr/Max-Age=3600/i, "cookies can be expired (Max-Age)");
	$cookie->expire(undef);
	unlike($cookie->toString, qr/Expires/i, "cookies can be made to expire by session (Expires)");
	unlike($cookie->toString, qr/Max-Age/i, "cookies can be made to expire by session (Max-Age)");
	$cookie->expire(0);
	like($cookie->toString, qr/Expires=Thu, 01-Jan-1970 00:00:00 GMT/i, "cookies can be deleted (Expires)");
	like($cookie->toString, qr/Max-Age=0/i, "cookies can be deleted (Max-Age)");
};

done_testing();
