# $Id: Aselect.pm 44542 2016-01-20 14:36:35Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/aselect-perl/lib/Aselect.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package Aselect;

use Xyzzy::Directory;

use Aselect::Resources;
use Aselect::UI::Attributes;
use Aselect::UI::Cancel;
use Aselect::UI::Combined;
use Aselect::UI::Giveup;
use Aselect::UI::Login;
use Aselect::UI::Logout;
use Aselect::UI::Password;
use Aselect::UI::Session;
use Aselect::UI::Settings;
use Aselect::UI::Static;
use Aselect::UI::Status;
use Aselect::UI::Success;
use Aselect::UI::Windshield;
use Aselect::WS::RequestAuthentication;
use Aselect::WS::Server;
use Aselect::WS::ValidateCASv1;
use Aselect::WS::ValidateCASv2;
use Aselect::WS::VerifyCredentials;

use Xyzzy -self;
use Aselect::Config -mixin;
use Aselect::LDAP::Config -mixin;
use Aselect::GSSAPI::Config -mixin;
use Aselect::Crypto::Config -mixin;
use Aselect::UI::SPNEGO::Config -mixin;

sub handler {
	my $rsc = new Aselect::Resources(cfg => $self);

	my $server = new Aselect::WS::Server(cfg => $rsc, requests => {
		authenticate => new Aselect::WS::RequestAuthentication(cfg => $rsc),
		verify_credentials => new Aselect::WS::VerifyCredentials(cfg => $rsc),
	});

	my $status = new Aselect::UI::Status(cfg => $rsc);
	my $static = new Aselect::UI::Static(cfg => $rsc);
	my $success = new Aselect::UI::Success(cfg => $rsc);
	my $credentials = $self->kerberos_principal
		? new Aselect::UI::Combined(cfg => $rsc, success => $success)
		: new Aselect::UI::Password(cfg => $rsc, success => $success);
	my $validate = new Aselect::WS::ValidateCASv2(cfg => $rsc);

	my $root = new Xyzzy::Directory(cfg => $rsc,
		handler => $static,
		fallback => $static,
		subdirs => {
			login => new Aselect::UI::Login(cfg => $rsc, methods => [
				new Aselect::UI::Cancel(cfg => $rsc),
				new Aselect::UI::Session(cfg => $rsc, success => $success),
				$credentials,
				new Aselect::UI::Giveup(cfg => $rsc),
			]),
			status => $status,
			settings => new Aselect::UI::Settings(cfg => $rsc),
			attributes => new Aselect::UI::Attributes(cfg => $rsc),
			logout => new Aselect::UI::Logout(cfg => $rsc),
			server => $server,
			aselectserver => new Xyzzy::Directory(cfg => $rsc,
				handler => $status,
				subdirs => { server => $server },
			),
			validate => new Aselect::WS::ValidateCASv1(cfg => $rsc),
			serviceValidate => $validate,
			proxyValidate => $validate,
		}
	);
	return new Aselect::UI::Windshield(cfg => $rsc, handler => $root);
}

__END__

=encoding utf8

=head1 NAME

Aselect - Single Sign On for web applications

=head1 DESCRIPTION

A module for use with the Xyzzy FastCGI framework to deliver
a seamless Single Sign On experience connecting LDAP, Kerberos
(including Microsoft's Active Directory) with A-Select and CAS
aware applications.

=head1 CLUSTERS

The A-Select server can be used in a high-available and/or load balancing
clustered setup almost trivially. No state is shared between the nodes, all
that is needed is to make sure that:

=over 4

=item *

The value of CryptoSecret is the same across all nodes in the cluster;

=item *

The clocks are synchronized (using NTP or something similar).

=back

If your load balancing mechanism does not guarantee that the server is
always accessed with the same name (in the HTTP Host: request header) you
should also set AselectServerID and AselectCookieDomain on all nodes (to a
value that is the same on all nodes).

Unless you have perfect confidence in your clocks and your NTP solution,
consider setting CryptoClockJitter as well.

=head1 CONFIGURATION

The Aselect server configuration file uses Xyzzy syntax, that is, key/value
pairs separated by whitespace. Some keys accept multiple values, which can
be specified by subsequent whitespace indented lines.

Time units can be specified using a number followed by an optional unit (ns,
us, ms, s, m, h, d, w, l, q, y). The default is seconds; ‘l’ means months,
‘q’ means 3 months.

Available options:

=over 4

=item AselectServerID I<id>

Identifier used by the A-Select protocol that identifies this server
instance. If you run multiple A-Select servers in a cluster, this
parameter should be the same across all the nodes in the cluster.

=item AselectOrganization I<name>

The A-Select protocol identifies the organization a user belongs to. In
this implementation all users belong to the same organization, which you
can specify using this parameter.

=item AselectCookieDomain I<domain>

The DNS-domain the session/settings HTTP cookies are attached to. Useful if
you have multiple servers with different hostnames but a common domain.

If no cookie domain is set, determining the cookie scope is left to the
browser.

=item AselectAttributes I<name> [I<base>]

Declares an attribute set (also known as a ‘policy’ or ‘attribute release
policy’) that can be attached to requestors (A-Select application IDs). See
AselectRequestor below.

Subsequent lines specify attributes, one per line, as a name/value pair
separated by whitespace.

Attribute values can be any string. If the string contains no dollar signs
it is simply used literally as the value. Any $I<name> or ${I<name>}
substrings are replaced by the LDAP attribute value of that name
(interpolation).

If the user has no LDAP attributes of that name, the entire line is omitted
from the resulting attribute set. If a multi-valued LDAP attribute is
interpolated, the entire line is repeated for each LDAP value.

In case the line contains multiple multi-valued interpolations, the
end-result will be the cartesian product of all values.

Example:

	AselectAttributes foo
		foo bar
		department $ou
		name $sn, $gn

In the above example every user will always have an A-Select Attribute
‘foo’ with value ‘bar’. The department attribute will be set to any
organizational units the user may be a member of. If the user is in
multiple units, he or she will have multiple resulting departement
attribute values. The name attribute will be something like ‘Smith, Bob’.

If you would like to extend an existing attribute set with extra
attributes, you can name the original attribute set as the second argument.

Example:

	AselectAttributes derp foo
		mail $emailAddress

This creates a new attribute set called ‘derp’ that has all the attributes
of ‘foo’ as well as a mail attribute.

=item AselectRequestor I<name> [I<policy>]

Declares an A-Select authentication requestor. A requestor (identified by
its application ID) is an entity, usually an application or site, that is
allowed to initiate an authentication procedure.

An optional attribute policy name may be specified, which must have been
declared first using AselectAttributes (see below).

If an SSL public key or certificate is specified on subsequent indented
lines, all requests for this application ID will be required to be signed
by that key.

=item AselectLanguages I<language> [I<language> ...]

The list of languages that are supported by the templates. Each language
should be specified as a two-letter (ISO 639) code corresponding to the
HTTP Accept-Language header.

If (and only if) this parameter is specified, templates are loaded from the
subdirectory of StylesheetDir that corresponds to the language that is
negotiated using request headers and/or cookies. Similarly, static content
is loaded from a subdirectory of ContentDir.

=item CryptoSecret I<long random string>

The key that is used to sign and verify all tickets and tokens. This should
be a long string (at least 64 characters) of random characters from a
cryptographically sound origin. Leaking this key allows an attacker to
impersonate any user on the system.

If you run multiple A-Select servers in a cluster, this key should
be the same across all nodes.

Changing this configuration setting will cause extant sessions that were
created with a different (old) value to be invalid, effectively logging
all users out.

=item AselectSecret

A deprecated name for CryptoSecret. Please use CryptoSecret instead.

=item CryptoClockJitter I<interval>

If you run multiple A-Select servers in a cluster, clocks in the
cluster MUST be synced using a protocol like NTP. However, even if NTP is
in use, small time differences may still occur.

This parameter configures the maximum clock difference that the server
accepts. A value of 1 second should be more than enough in most cases.

Defaults to 0.

=item ServiceFilter [I<regex>]

A filter to be applied to CAS service URLs and unsigned A-Select requests.
This filter consists of two parts: a generic regular expression and a list
of domains. If both are specified, both are applied. Use the generic regular
expression to apply generic policies like ‘all sites must use https’. Use
the domain list to restrict access to sites under your control.

Specify the domain list as indented lines under the ServiceFilter statement.
Each line can be either a domain name, a domain name with a dot in front
(which causes the server to include subdomains), or an anchored regular
expression (which must start with C<^>). Example:

	ServiceFilter ^https:
		.example.org
		example.com
		www.example.com

This example will cause the server to only accept service URLs that start
with https: and only for example.org (and all its subdomains) as well as
example.com and www.example.com. Requests for ftp.example.com, for example,
are I<not> accepted.

=item KerberosPrincipal I<principal>

The principal name that clients use when they generate kerberos tickets for
signing on. Its value is heavily dependent on your Kerberos environment.

Example:

	KerberosPrincipal HTTP/sso.example.org@DOMAIN.EXAMPLE.ORG

=item SPNEGOUserAgent I<regex>

SPNEGO/Kerberos authentication is not tried by default, as it may lead to
confusing password popups on some platforms if SPNEGO is not available.

However, if the browser user agent matches this regular expression then
SPNEGO is assumed to be configured correctly by the client's administrator
and Kerberos based negotiation will take place.

Any configuration by the user will override this mechanism.

=item HTTPSEverywhere I<yes/no>

Forces all applications to use https by rewriting all service URLs as they
come in. Even if a sign-in is attempted on a http site, the user will be
redirected back to the https version upon completing their sign-in.

Defaults to ‘no’.

=item AselectSessionTimeout I<interval>

How long an SSO session should last. Default is 10 hours.

=item AselectRequestTimeout I<interval>

How long a signon session should last. This is the interval between a user
starting to sign in to an SSO enabled site and completing that sign-in.
Default is 5 minutes, which should normally be more than enough for people
to enter their passwords.

=item AselectCredentialTimeout I<interval>

The time required for a site to verify the credentials that were passed from
the A-Select server to the site. This interval should be as short as
possible since these credentials tend to end up in the browser history and
grant access to the given site for their entire duration.

The default is 30 seconds but most sites respond within seconds, so setting
a shorter interval (say 10 seconds) may improve security without
significant drawbacks.

=item AuthFailTimeout I<interval>

The time to pause when a user enters incorrect credentials (to slow down
dictionary attacks). Default is 1 second.

=item StrictTransportSecurity I<interval>

Set the Strict-Transport-Security HTTP header on all pages, to ensure that
browsers always use https when accessing the A-Select server. See
https://en.wikipedia.org/wiki/HTTP_Strict_Transport_Security for details.

Default is to not issue Strict-Transport-Security headers.

=item AselectLoginURL I<url>

Set the URL that is passed to application when they request authentication.
This should point at a variation on C<https://sso.example.com/login>.

The default is to automatically construct the URL from the request.

=item StylesheetDir I<directory>

A directory containing XSLT templates for the user-facing pages.

=item ContentDir I<directory>

Requests that do not match any of the internal handlers of the A-Select server
are loaded from files under this directory, after having ‘.xml’ appended to the
name. Stylesheets are applied to these XML files.

=item XHTMLStylesheets I<basename> [I<basename> ...]

A list of extra XSLT templates to apply after the per-page templates have been
applied.

=item LDAPServer I<uri>

The URI to the ldap-server. Example:

	ldap://ldap.example.com

=item LDAPBase I<base>

The search base of the LDAP server. Example:

	dc=example,dc=com

=item LDAPUsername I<dn>

Username to use when searching for uids. Omit this setting if your LDAP
server supports anonymous searches.

=item LDAPPassword I<password>

Password to use when searching for uids. Omit this setting if your LDAP
server supports anonymous searches.

=item LDAPSecure I<yes/no>

Require the connection to the LDAP-server to be created securely, using
either LDAPS or STARTTLS (preferred). Defaults to yes.

=item LDAPCApath I<directory>

Directory containing certificates that can be used to connect to the LDAP
server securely. Specify either this or LDAPCAfile.

=item LDAPCAfile I<file>

File containing the certificate of the LDAP server. Specify either this or
LDAPCApath.

=item LDAPAttribute I<uid>

LDAP attribute containing the login principal of the user. Usually ‘uid’,
which is also the default.

=item LDAPFilter I<(...)>

An additional filter to be applied to users logging in. Example:

	LDAPFilter (organizationalStatus=staff)

To allow only staff to log in.

=item LDAPExpiryAttributes

LDAP attributes that, if any are present for a user, cause a warning to be
displayed when the user starts an SSO session. Each attribute must be
specified on a seperate line. These are passed to the stylesheet in an
<expire> element. An optional name for the stylesheet element can be
specified for each attribute, separated from the attribute with whitespace.
If no element name is specified, the attribute name is used as the element
name.

If any LDAP attribute names are given directly after the
LDAPExpiryAttributes keyword, these form the list of attribute names that
indicate whether password expiration reminders need to be given to the
user. The presence of any of these attributes will trigger the warning to
the user. If no attribute names are specified in this way, the server
defaults to the list of attributes that are displayed.

See below for an example.

=item LockClientIP I<yes/no>

Whether to lock sessions to the client's IP address. Provides extra
security but may be inconvenient in environments where the IP address of a
client changes often (for example, due to wireless roaming) or in
dual-stack IPv4/IPv6 setups.

Changing this configuration setting will cause extant sessions that were
created with the inverse (old) value to be considered invalid, effectively
logging all users out.

Defaults to ‘yes’.

=back

=head1 EXAMPLE

Sample configuration file:

	#! /usr/bin/perl /usr/lib/cgi-bin/xyzzy

	Application Aselect

	StylesheetDir /etc/aselect-perl/stylesheets

	LDAPServer ldap://ldap.example.org
	LDAPBase dc=example,dc=org
	LDAPUsername cn=aselect,dc=example,dc=org
	LDAPPassword hunter2
	LDAPCApath /etc/ssl/certs

	# If the passwordExpirationStage LDAP attribute is present,
	# the user will be warned and the passwordLastChanged,
	# passwordExpirationStage and passwordExpirationFinalDay
	# attributes are passed to the stylesheet.
	LDAPExpiryAttributes passwordExpirationStage
		passwordLastChanged last-change
		passwordExpirationStage stage
		passwordExpirationFinalDay final-day

	KerberosPrincipal HTTP/sso.example.org@DOMAIN.EXAMPLE.ORG

	CryptoSecret zxnrbl

	AselectServerID sso.example.org
	AselectCookieDomain sso.example.org

	AselectAttributes basic
		uid $uid
		anr $employeeNumber
		mail $mail

	AselectRequestor foobar basic
	AselectRequestor website basic

	AselectAttributes federated
		urn:mace:dir:attribute-def:uid $uid
		urn:mace:dir:attribute-def:employeeNumber $employeeNumber
		urn:mace:dir:attribute-def:mail $mail
		urn:mace:dir:attribute-def:sn $sn
		urn:mace:dir:attribute-def:cn $sn, $gn
		urn:mace:dir:attribute-def:givenName $gn
		urn:mace:dir:attribute-def:eduPersonEntitlement $eduPersonEntitlement
		urn:mace:dir:attribute-def:eduPersonPrincipalName $eduPersonPrincipalName

	AselectRequestor federation federated
		-----BEGIN PUBLIC KEY-----
		MIIjRL4Y9yXsB06awwd8nVBLSuc9K1zQSaU1DVTSnB47icDeq7Fpp6goiohRDgG4
		PYbnBuTAaSoP4nvBjuD8bH334iPR0Y7XjL8tNaxfGwMAMHFWm0UEBGBo9AxzByuQ
		...
		-----END PUBLIC KEY-----
