# $Id: Kiki.pm 51302 2024-03-27 09:46:03Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/kiki/lib/UvT/Kiki.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package UvT::Kiki;

use Xyzzy::Database;
use Xyzzy::ReST;
use Xyzzy::Method;

use UvT::Kiki::ACL;
use UvT::Kiki::Activator;
use UvT::Kiki::Database;
use UvT::Kiki::UI::Export;
use UvT::Kiki::UI::Main;
use UvT::Kiki::UI::Static;
use UvT::Kiki::WS::Alias::Delete;
use UvT::Kiki::WS::Alias::Get;
use UvT::Kiki::WS::Alias::Post;
use UvT::Kiki::WS::Blacklist;
use UvT::Kiki::WS::Domains;
use UvT::Kiki::WS::Nonce;
use UvT::Kiki::WS::Person::Delete;
use UvT::Kiki::WS::Person::Get;
use UvT::Kiki::WS::Person::Put;
use UvT::Kiki::WS::Persons;
use UvT::Kiki::WS::Search;

use Xyzzy -self;
use Xyzzy::Crypto::Config -mixin;
use Xyzzy::Database::Config -mixin;
use UvT::Kiki::Config -mixin;
use UvT::Kiki::ACL::Config -mixin;

sub handler {
	my $static = new UvT::Kiki::UI::Static(cfg => $self);
	my $postalias = new UvT::Kiki::WS::Alias::Post(cfg => $self);
	my $search = new UvT::Kiki::WS::Search(cfg => $self);

	local *get = sub {
		my $handler = shift;
		return new Xyzzy::Method(cfg => $self, methods => {
			HEAD => $handler,
			GET => $handler,
		});
	};

	my $handler = new Xyzzy::ReST(cfg => $self,
		handler => new UvT::Kiki::UI::Main(cfg => $self),
		generic => {
			blacklist => get(new UvT::Kiki::WS::Blacklist(cfg => $self)),
			domains => get(new UvT::Kiki::WS::Domains(cfg => $self)),
			export => get(new UvT::Kiki::UI::Export(cfg => $self)),
			nonce => get(new UvT::Kiki::WS::Nonce(cfg => $self)),
			persons => get(new UvT::Kiki::WS::Persons(cfg => $self)),
			search => new Xyzzy::Method(cfg => $self, methods => {
				HEAD => $search,
				GET => $search,
				POST => $search,
			}),
			aliases => new Xyzzy::Method(cfg => $self, methods => {
				POST => $postalias,
			}),
		},
		specific => {
			aliases => new Xyzzy::Method(cfg => $self, methods => {
				DELETE => new UvT::Kiki::WS::Alias::Delete(cfg => $self),
				GET => new UvT::Kiki::WS::Alias::Get(cfg => $self),
				POST => $postalias,
			}),
			persons => new Xyzzy::Method(cfg => $self, methods => {
				DELETE => new UvT::Kiki::WS::Person::Delete(cfg => $self),
				GET => new UvT::Kiki::WS::Person::Get(cfg => $self),
				PUT => new UvT::Kiki::WS::Person::Put(cfg => $self),
			}),
		},
		fallback => $static,
	);

	$handler = new UvT::Kiki::Database(cfg => $self, handler => $handler);
	$handler = new Xyzzy::Database(cfg => $self, handler => $handler);
	$handler = new UvT::Kiki::Activator(cfg => $self, handler => $handler);
	$handler = new UvT::Kiki::ACL(cfg => $self, handler => $handler);

	return $handler;
}

__END__

=pod

=encoding utf8

=head1 NAME

UvT::Kiki - Tilburg University specific tool to manage email aliases

=head1 Configuration options

=over

=item Languages

List of languages codes that are supported. Correspond with directories
under F</etc/kiki/stylesheets> and F</etc/kiki/content>.

=item NotificationSender

From-address for notification e-mails. May include a name. Should be valid
RFC2822-syntax.

=item NotificationRecipients

Destination addresses for notification e-mails. May include names.

=item ActivateCommand

Command that is run every time the database is modified. Use this to sync
your LDAP.

=item RequestTimeout

Timeout for CSRF nonces. Use ns/us/ms/s/m/h/d/w suffixes for
nanoseconds/microseconds/milliseconds/seconds/hours/days/weeks.

=item AllowedUsers

List of $REMOTE_USER names that are allowed to log in. Users (as opposed to
robots) are allowed to delete labels for persons but may not add or delete
persons.

=item AllowedRobots

List of $REMOTE_USER names that are allowed to log in. Robots (as opposed
to users) are allowed to add and delete persons but may not remove labels,
only add them.

=back

=cut
