-- $Id: create.sql 43886 2015-09-10 15:35:24Z wsl $
-- $URL: https://svn.uvt.nl/its-id/trunk/sources/kiki/doc/create.sql $

\encoding UTF8

SET client_min_messages TO WARNING;

BEGIN;

DROP SCHEMA IF EXISTS kiki CASCADE;
CREATE SCHEMA kiki;

SET search_path TO kiki;

CREATE SEQUENCE seq MINVALUE 1000000;

-- Eén domein kan meerdere namen hebben, bijvoorbeeld uvt.nl en tilburguniversity.edu

CREATE TABLE domainnames (
	domainname BIGINT
		NOT NULL
		DEFAULT nextval('seq')
		PRIMARY KEY,
	domain BIGINT
		NOT NULL,
	name TEXT
		NOT NULL
		UNIQUE
		DEFERRABLE INITIALLY DEFERRED
		CHECK (name ~ '^(?:xn--)?[a-z0-9]+(?:-[a-z0-9]+)*(?:\.(?:xn--)?[a-z0-9]+(?:-[a-z0-9]+)*)+$')
);

CREATE INDEX domainnames_domain ON domainnames (domain);

CREATE TABLE domains (
	domain BIGINT
		NOT NULL
		DEFAULT nextval('seq')
		PRIMARY KEY,
	-- geeft aan welke van de eventuele meerdere namen de primaire is
	main BIGINT
		NOT NULL
		UNIQUE
		DEFERRABLE INITIALLY DEFERRED
		REFERENCES domainnames (domainname)
			ON UPDATE CASCADE
			ON DELETE CASCADE
			DEFERRABLE INITIALLY DEFERRED
);

ALTER TABLE domainnames ADD CONSTRAINT domainnames_domain_fkey
	FOREIGN KEY (domain) REFERENCES domains (domain)
		ON UPDATE CASCADE
		ON DELETE CASCADE
		DEFERRABLE INITIALLY DEFERRED;
CREATE INDEX domains_main ON domains (main);

-- gegevens die via GLW/FIM binnenkomen en in sync gehouden
-- moeten worden
CREATE TABLE persons (
	-- stiekem gewoon anr:
	person BIGINT
		NOT NULL
		CHECK (person > 0 AND person < 1000000)
		PRIMARY KEY,
	pita BOOL
		NOT NULL
		DEFAULT FALSE,
	mailaddress TEXT
		NOT NULL
		CHECK (mailaddress ~ '^[A-Za-z0-9_'']+(?:[.-][A-Za-z0-9_'']+)*@(?:xn--)?[a-z0-9]+(?:-[a-z0-9]+)*(?:\.(?:xn--)?[a-z0-9]+(?:-[a-z0-9]+)*)+$'),
	-- refereert aan een mail_alias dat het hoofdalias zal zijn
	-- voor deze persoon, en dat gebruikt zal worden voor canonicalisatie
	canonical BIGINT
);

CREATE UNIQUE INDEX persons_mailaddress_uniq ON persons (lower(mailaddress));

CREATE TABLE mail_aliases (
	mail_alias BIGINT
		NOT NULL
		DEFAULT nextval('seq')
		PRIMARY KEY,
	name TEXT
		NOT NULL
		CHECK (name ~ '^[A-Za-z0-9_&`'']+(?:[.-][A-Za-z0-9_&`'']+)*$'),
	mtime TIMESTAMP,
	lastuser TEXT,
	comments TEXT,
	domain BIGINT
		NOT NULL
		REFERENCES domains
			ON UPDATE CASCADE
			ON DELETE CASCADE
			DEFERRABLE INITIALLY DEFERRED,
	addressbook BOOL
		NOT NULL,
	-- voor een persoonlijk mailalias (person NOT NULL) moet gelden:
	-- 1) er is precies één destination voor dit alias, en
	-- 2) dit is een personal destination naar precies deze persoon
	person BIGINT
		REFERENCES persons
			ON UPDATE CASCADE
			ON DELETE CASCADE
			DEFERRABLE INITIALLY DEFERRED
);

ALTER TABLE persons ADD CONSTRAINT persons_canonical_fkey
	FOREIGN KEY (canonical) REFERENCES mail_aliases (mail_alias)
		ON UPDATE CASCADE
		ON DELETE SET NULL
		DEFERRABLE INITIALLY DEFERRED;

CREATE INDEX mail_alias_person ON mail_aliases (person);
CREATE INDEX mail_alias_name ON mail_aliases (lower(name));
CREATE INDEX mail_alias_domain ON mail_aliases (domain);
CREATE UNIQUE INDEX mail_alias_uniq ON mail_aliases (lower(name), domain);
-- UNIQUE constraint op basis van functies kan (nog) niet gedeferd worden:
--ALTER TABLE mail_aliases ADD UNIQUE USING INDEX mail_alias_uniq DEFERRABLE INITIALLY DEFERRED;

-- wijst naar een persoon
CREATE TABLE personal_destinations (
	personal_destination BIGINT
		NOT NULL
		DEFAULT nextval('seq')
		PRIMARY KEY,
	mail_alias BIGINT
		NOT NULL
		REFERENCES mail_aliases
			ON UPDATE CASCADE
			ON DELETE CASCADE
			DEFERRABLE INITIALLY DEFERRED,
	person BIGINT
		NOT NULL
		REFERENCES persons
			ON UPDATE CASCADE
			ON DELETE CASCADE
			DEFERRABLE INITIALLY DEFERRED,
	UNIQUE (mail_alias, person) DEFERRABLE INITIALLY DEFERRED
);

CREATE INDEX personal_destination_mail_alias ON personal_destinations (mail_alias);
CREATE INDEX personal_destination_person ON personal_destinations (person);

-- wijst naar iets interns (een ander alias dus)
CREATE TABLE internal_destinations (
	internal_destination BIGINT
		NOT NULL
		DEFAULT nextval('seq')
		PRIMARY KEY,
	mail_alias BIGINT
		NOT NULL
		REFERENCES mail_aliases
			ON UPDATE CASCADE
			ON DELETE CASCADE
			DEFERRABLE INITIALLY DEFERRED,
	destination BIGINT
		NOT NULL
		REFERENCES mail_aliases
			ON UPDATE CASCADE
			ON DELETE CASCADE
			DEFERRABLE INITIALLY DEFERRED,
	UNIQUE (mail_alias, destination) DEFERRABLE INITIALLY DEFERRED
);

CREATE INDEX internal_destination_mail_alias ON internal_destinations (mail_alias);
CREATE INDEX internal_destination_destination ON internal_destinations (destination);

-- wijst naar iets externs (met een domein dat dus niet in domains staat)
CREATE TABLE external_destinations (
	external_destination BIGINT
		NOT NULL
		DEFAULT nextval('seq')
		PRIMARY KEY,
	mail_alias BIGINT
		NOT NULL
		REFERENCES mail_aliases
			ON UPDATE CASCADE
			ON DELETE CASCADE
			DEFERRABLE INITIALLY DEFERRED,
	mailaddress TEXT
		NOT NULL
		CHECK (mailaddress ~ '@(?:(?:xn--)?[a-z0-9]+(?:-[a-z0-9]+)*(?:\.(?:xn--)?[a-z0-9]+(?:-[a-z0-9]+)*)+|localhost)$')
);

CREATE INDEX external_destination_mail_alias ON external_destinations (mail_alias);
CREATE INDEX external_destination_mailaddress ON external_destinations (lower(mailaddress));
CREATE UNIQUE INDEX external_destination_uniq ON external_destinations (mail_alias, lower(mailaddress));

COMMIT;

--VACUUM FULL ANALYZE;
--REINDEX DATABASE kiki;
