# $Id: Person.pm 47634 2018-06-26 13:03:40Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/kiki/lib/UvT/Kiki/Database/Person.pm $

use re '/aa';

package UvT::Kiki::Database::Person;

use Clarity -self;

use Email::Address;

use constant type => 'person';

field db;
field anr;

field labels => sub {
	my $self = shift;
	return $self->db->labels_for_anr($self->anr);
};

field mailbox => sub {
	my $self = shift;
	return $self->db->mailbox_for_anr($self->anr);
};

field original => sub { shift->clone(original => undef) };

sub clone {
	my $class = ref $self;
	return $class->new(
		anr => $self->anr,
		labels => $self->labels,
		mailbox => $self->mailbox,
		@_
	);
}

sub full {
	return $self->labels->[0]->full;
}

sub update_mailbox {
	my ($mailbox) = @_;

	return [error => 'missing-address']
		unless defined $mailbox;

	my $original = $self->original;

	unless($original && $mailbox eq $original->mailbox) {
		my ($x, $y) = Email::Address->parse($mailbox);

		return [error => 'invalid-address']
			unless defined $x && !defined $y;

		my $host = lc($x->host);
		$mailbox = $x->user.'@'.$host;

		unless($original && lc($mailbox) eq lc($original->mailbox)) {
			my $db = $self->db;

			return [error => 'internal-domain']
				if $db->domain_by_name($host);

			return [error => 'invalid-domain']
				unless $db->hostresolvable($host);

			my $p = $db->person_for_mailbox($mailbox);
			return [error => 'duplicate-mailbox']
				if defined $p && $p->anr != $self->anr;
		}
	}

	$self->mailbox($mailbox);

	return [ok => 'mailbox', $mailbox];
};

sub update_labels {
	my ($labels, $may_delete_labels) = @_;

	my $original = $self->original;
	my @orig = @{$original->labels} if $original;
	my %orig = map { lc($_->full) => $_ } @orig;

	my $db = $self->db;

	my @notes;
	my %uniq;
	my $fail;

	foreach my $label (@$labels) {
		if($label =~ /^\s*\@(?:[^\@]*)?\z/) {
			push @notes, [ok => 'empty', $label =~ s/^\s*//r];
		} elsif($label =~ /^(.*)\@([^\@]*)\z/) {
			my ($local, $domainname) = ($1, $2);
			my $domain = $db->domain_by_name($domainname);
			if($domain) {
				$domainname = $domain->name;
				my $canon = "$local\@$domainname";
				my $canon_lc = lc($canon);
				if(exists $uniq{$canon_lc}) {
					push @notes, [ok => 'duplicate', $canon];
				} elsif(exists $orig{$canon_lc}) {
					my $alias = $uniq{$canon_lc} = $orig{$canon_lc};
					if($alias->localpart eq $local || $local =~ /^[A-Z][0-9]{6,}\z/) {
						push @notes, [ok => 'unchanged', $canon];
					} else {
						$alias->update_localpart($local);
						push @notes, [ok => 'case', $canon];
					}
				} elsif(length($local) > 64) {
					$fail = 1;
					push @notes, [error => 'localpart-too-long'];
				} elsif($db->exists_alias($local, $domain)) {
					$fail = 1;
					push @notes, [error => 'address-exists'];
				} elsif($local =~ /^[a-z0-9_]+(?:[.-][a-z0-9_]+)*\z/i) {
					# s123456, p123456, u1234567 etc should always be lowercase
					$local =~ s/^[A-Z][0-9]{6,}\z/\l$&/;
					$uniq{$canon_lc} = $db->tentative_label($local, $domain);
					push @notes, [ok => 'new', $canon];
				} else {
					$fail = 1;
					push @notes, [error => 'malformed-localpart'];
				}
			} else {
				$fail = 1;
				push @notes, [error => 'unknown-domain'];
			}
		} else {
			$fail = 1;
			push @notes, [error => 'invalid-address'];
		}
	}

	unless($may_delete_labels) {
		my @missing =
			map { $orig{$_}->full }
			grep { !exists $uniq{$_} }
			keys %orig;
		
			return [[error => 'missing-label', \@missing]]
				if @missing;
	}

	unless($fail) {
		if(%uniq) {
			# the list of unique labels in the original order
			my @uniq = map { delete $uniq{lc($_->[2])} // () } @notes;
			$self->labels(\@uniq);
		} else {
			return [[error => 'empty-labels']];
		}
	}

	return \@notes;
}

sub remove {
	$self->db->remove_person($self);
	return ['ok', 'removed'];
}

sub store {
	$self->db->upsert_person($self);
}

sub expand { return }

# A textual representation of how this entry is different from an older version.
# If no older version is available, a pseudo-diff is generated.
sub diff {
	my @labels = map { $_->full } @{$self->labels};
	my $full = shift @labels;
	my $anr = $self->anr;

	my $diff = "$full ($anr):\n";

	if(my $old = $self->original) {
		my @old_labels = map { $_->full } @{$old->labels};
		my $old_full = shift @old_labels;
		my $old_anr = $old->anr;

		my %labels; @labels{@labels} = ();
		my %old_labels; @old_labels{@old_labels} = ();

		my $full_text = $full eq $old_full ? $full : "$old_full => $full";
		my $anr_text = $old_anr == $anr ? $anr : "$old_anr => $anr";

		$diff = "$full_text ($anr_text):\n";

		foreach my $label (sort { lc($a) cmp lc($b) || $a cmp $b } @old_labels) {
			$diff .= "-\t$label\n" unless exists $labels{$label};
		}

		foreach my $label (sort { lc($a) cmp lc($b) || $a cmp $b } @old_labels) {
			$diff .= " \t$label\n" if exists $labels{$label};
		}

		foreach my $label (sort { lc($a) cmp lc($b) || $a cmp $b } @labels) {
			$diff .= "+\t$label\n" unless exists $old_labels{$label};
		}
	} else {
		foreach my $label (sort { lc($a) cmp lc($b) || $a cmp $b } @labels) {
			next if $label eq $full;
			$diff .= " \t$label\n";
		}
	}

	return $diff;
}

sub toString {
	return join(', ', map { $_->full } @{$self->labels}) . ': ' . $self->mailbox;
}

sub TO_JSON {
	return {
#		type => $self->type,
		anr => int($self->anr),
		labels => [map { $_->full } @{$self->labels}],
		mailbox => $self->mailbox,
	};
}
