#! /usr/bin/perl

use strict;
use warnings FATAL => 'all';
use autodie;

use Test::More;
use Errno qw(ENOENT);
use File::Temp;

my $tmp = File::Temp->newdir;

BEGIN { use_ok('File::Hashset') or BAIL_OUT('need File::Hashset package to run') }

my $testfile1 = "$tmp/test1";

eval { unlink $testfile1 };
die $@ if $@ && !(UNIVERSAL::can($@, 'errno') && $@->errno == ENOENT);

open my $fh1, '>', $testfile1;

for(my $i = 0; $i < 256; $i++) {
	print $fh1 chr($i)x32;
}

for(my $i = 0; $i < 256; $i++) {
	print $fh1 chr($i)x32;
}

for(my $i = 0; $i < 256; $i++) {
	print $fh1 ('A'x8 . chr($i)x24);
}

close $fh1;

File::Hashset::sortfile($testfile1, 32);

my $a = File::Hashset->load($testfile1);

for(my $i = 0; $i < 256; $i++) {
	ok($a->exists(chr($i)x32), "chr($i)x32 should exist");
}

for(my $i = 0; $i < 256; $i++) {
	ok($a->exists('A'x8 . chr($i)x24), "'A'x8 . chr($i)x24 should exist");
}

ok(!$a->exists('AB'x16), "ABAB should not exist");

undef $a;

my $testfile2 = "$tmp/test2";

eval { unlink $testfile2 };
die $@ if $@ && !(UNIVERSAL::can($@, 'errno') && $@->errno == ENOENT);

open my $fh2, '>', $testfile2;

for(my $i = 0; $i < 256; $i++) {
	print $fh2 ('Z'x8) . (chr($i)x24);
}

close $fh2;

my $b = File::Hashset->load($testfile2);

for(my $i = 0; $i < 256; $i++) {
	ok($b->exists('Z'x8 . chr($i)x24), "'Z'x8 . chr($i)x24 should exist");
}

ok(!$b->exists('AB'x16), "ABAB should not exist");
ok(!$b->exists(scalar "\x00"x32), "\\x0000 should not exist");
ok(!$b->exists(scalar "\xFF"x32), "\\xFFFF should not exist");

undef $b;

my $testfile3 = "$tmp/test3";

my @sources = map { File::Hashset->load($_) } (($testfile1, $testfile2) x 500);

File::Hashset->merge($testfile3, 32, @sources);

my $c = File::Hashset->load($testfile3);

for(my $i = 0; $i < 256; $i++) {
	ok($c->exists(chr($i)x32), "chr($i)x32 should exist");
}

for(my $i = 0; $i < 256; $i++) {
	ok($c->exists('A'x8 . chr($i)x24), "'A'x8 . chr($i)x24 should exist");
}

for(my $i = 0; $i < 256; $i++) {
	ok($c->exists('Z'x8 . chr($i)x24), "'Z'x8 . chr($i)x24 should exist");
}

ok(!$c->exists('AB'x16), "ABAB should not exist");

undef $c;

done_testing();
