# $Id: Transaction.pm 42192 2014-09-10 11:44:16Z wsl $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/kiki/lib/UvT/Kiki/Database/Transaction.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package UvT::Kiki::Database::Transaction;

use UvT::Kiki::Database::Alias;
use UvT::Kiki::Database::Person;
use UvT::Kiki::Database::External;
use UvT::Kiki::Database::Domain;

use Clarity -self;

field db;
field cfg;

field hostresolvable_cache => {};

sub hostresolvable {
	my $host = shift;
	$host =~ s/.*\@//;
	$host = lc $host;
	return 0 unless $host =~ /^(?:xn--)?[a-z0-9]+(?:-[a-z0-9]+)*(?:\.(?:xn--)?[a-z0-9]+(?:-[a-z0-9]+)*)+$/;
	my $cache = $self->hostresolvable_cache;
	return $cache->{$host}
		if exists $cache->{$host};
	my $resolver = $self->cfg->dns;
	foreach my $type (qw(MX AAAA A)) {
		my $q = $resolver->send($host, $type)
			or next;
		foreach my $rr ($q->answer) {
			return $cache->{$host} = 1
				if $rr->type eq $type;
		}
	}
	return $cache->{$host} = 0;
}

field domains_by_id => sub {
	my $self = shift;
	my %domains;
	my $res = $self->query('SELECT n.domain, n.name
		FROM domainnames n JOIN domains d USING (domain)
		ORDER BY n.domainname = d.main DESC, n.name');
	foreach my $d (@$res) {
		my ($domain, $name) = @$d;
		push @{$domains{$domain}}, $name;
	}
	my %objects;
	my %names;
	while(my ($key, $val) = each %domains) {
		my $domain = new UvT::Kiki::Database::Domain(db => $self, id => $key, names => $val);
		$objects{$key} = $domain;
		foreach my $name (@$val) {
			$names{$name} = $domain;
		}
	}
	$self->domains_by_name(\%names);
	return \%objects;
};

field domains_by_name => sub { shift->domains_by_id; return };

sub domain_by_id {
	my $id = shift;
	return $self->domains_by_id->{$id};
}

sub domain_by_name {
	my $name = shift;
	return $self->domains_by_name->{lc($name)};
}

sub record2alias {
	my $domains = $self->domains_by_id;
	my @res;
	foreach my $record (@_) {
		my ($id, $name, $domain, $addressbook, $anr) = @$record;
		if(defined $domain) {
			$domain = $domains->{$domain} // confess("Unknown domain '$domain'");
			if(defined $anr) {
				push @res, new UvT::Kiki::Database::Person(db => $self, anr => $anr, id => $id, name => $name, domain => $domain);
			} else {
				push @res, new UvT::Kiki::Database::Alias(db => $self, id => $id, name => $name, domain => $domain, addressbook => $addressbook ? 1 : 0);
			}
		} else {
			push @res, new UvT::Kiki::Database::External(db => $self, id => $id, address => $name);
		}
	}
	return \@res;
}

sub new_alias {
	return new UvT::Kiki::Database::Alias(db => $self, id => undef, original => undef);
}

sub perform {
	my $q = $self->db->prepare_cached(shift);
	my $res = $q->execute(@_);
	$q->finish;
	return $res;
}

sub query {
	my $sql = shift;
	my $dst;
	if(ref $sql) {
		$dst = $sql;
		$sql = shift;
	}
	my $q = $self->db->prepare_cached($sql);
	$q->execute(@_);
	my $res = $q->fetchall_arrayref;
	$q->finish;
	if($dst) {
		push @$dst, @$res;
		return;
	} else {
		return $res;
	}
}

sub create_alias {
	my $alias = shift;

	my $q = $self->db->prepare_cached('INSERT INTO mail_aliases (name, domain, addressbook) VALUES (?, ?, ?) RETURNING mail_alias');
	$q->execute($alias->name, $alias->domain->id, $alias->addressbook ? 't' : 'f');
	my ($id) = $q->fetchrow_array;
	$q->finish;
	$alias->id($id);

	$self->add_destinations($alias);
}

sub update_alias {
	my $alias = shift;

	my $id = $alias->id;

	$self->perform('UPDATE mail_aliases SET name = ?, domain = ?, addressbook = ? WHERE mail_alias = ?',
		$alias->name, $alias->domain->id, $alias->addressbook ? 't' : 'f', $id);
	$self->perform('DELETE FROM personal_destinations WHERE mail_alias = ?', $id);
	$self->perform('DELETE FROM internal_destinations WHERE mail_alias = ?', $id);
	$self->perform('DELETE FROM external_destinations WHERE mail_alias = ?', $id);

	$self->add_destinations($alias);
}

sub remove_alias {
	my $alias = shift;

	$self->perform('DELETE FROM mail_aliases WHERE mail_alias = ?', $alias->id);
}

sub add_destinations {
	my $alias = shift;
	my $id = $alias->id;
	my $destinations = $alias->destinations;
	for my $d (@$destinations) {
		my $type = $d->type;
		if($type eq 'alias') {
			$self->perform('INSERT INTO internal_destinations (mail_alias, destination) VALUES (?, ?)', $id, $d->id);
		} elsif($type eq 'person') {
			$self->perform('INSERT INTO personal_destinations (mail_alias, person) VALUES (?, ?)', $id, $d->anr);
		} elsif($type eq 'external') {
			$self->perform('INSERT INTO external_destinations (mail_alias, mailaddress) VALUES (?, ?)', $id, $d->address);
		} else {
			confess("internal error: unknown type '$type'");
		}
	}
}

sub mailbox_for_anr {
	my $anr = shift;

	my $res = $self->query('SELECT p.mailaddress FROM persons p WHERE p.person = ?', $anr);

	return $res->[0][0];
}

sub names_for_anr {
	my $anr = shift;

	my $res = $self->query('SELECT a.mail_alias, a.name, a.domain, a.addressbook FROM mail_aliases a WHERE a.person = ?', $anr);

	return $self->record2alias(@$res);
}

sub person_for_anr {
	my $anr = shift;

	my $res = $self->query('SELECT c.mail_alias, c.name, c.domain, NULL, p.person
		FROM persons p JOIN mail_aliases c ON c.mail_alias = p.canonical
		WHERE p.person = ?', $anr);

	return $self->record2alias(@$res)->[0];
}

sub destinations_for {
	my $id = shift;

	my @res;

	$self->query(\@res, 'SELECT p.canonical, c.name, c.domain, NULL, p.person
		FROM personal_destinations d
			JOIN persons p ON d.person = p.person
			JOIN mail_aliases c ON c.mail_alias = p.canonical
		WHERE d.mail_alias = ?', $id);

	$self->query(\@res, 'SELECT a.mail_alias, a.name, a.domain, a.addressbook
		FROM mail_aliases a
			JOIN internal_destinations i ON i.destination = a.mail_alias
		WHERE i.mail_alias = ?', $id);

	$self->query(\@res, 'SELECT e.external_destination, e.mailaddress
		FROM external_destinations e WHERE e.mail_alias = ?', $id);

	return $self->record2alias(@res);
}

sub referrers_for {
	my $id = shift;

	my $res = $self->query('SELECT a.mail_alias, a.name, a.domain, a.addressbook
		FROM mail_aliases a
		JOIN internal_destinations i USING (mail_alias)
		WHERE i.destination = ?', $id);

	return $self->record2alias(@$res);
}

sub has_referrers {
	my $alias = shift;

	my $res = $self->query('SELECT EXISTS(SELECT * FROM internal_destinations WHERE destination = ?)', $alias->id);

	return $res->[0][0];
}

sub export {
	my $db = $self->db;

	my %res;

	my $q1 = $db->prepare_cached('SELECT s.address, m.address
		FROM internal_destinations d
			JOIN mail_addresses s ON s.mail_alias = d.mail_alias
			JOIN mail_addresses m ON m.mail_alias = d.destination');
	$q1->execute;
	while(my ($from, $to) = $q1->fetchrow_array) {
		push @{$res{$from}}, $to;
	}
	$q1->finish;

	my $q2 = $db->prepare_cached('SELECT s.address, m.address
		FROM personal_destinations d
			JOIN func_mail_addresses s ON s.mail_alias = d.mail_alias
			JOIN persons p ON p.person = d.person
			JOIN mail_addresses m ON m.mail_alias = p.canonical');
	$q2->execute;
	while(my ($from, $to) = $q2->fetchrow_array) {
		push @{$res{$from}}, $to;
	}
	$q2->finish;

	my $q3 = $db->prepare_cached('SELECT s.address, d.mailaddress
		FROM external_destinations d
			JOIN mail_addresses s ON s.mail_alias = d.mail_alias');
	$q3->execute;
	while(my ($from, $to) = $q3->fetchrow_array) {
		push @{$res{$from}}, $to;
	}
	$q3->finish;

	return \%res;
}

sub fetch_alias_by_localname {
	my $query = shift;

	confess("fetch_address(): \$query undefined") unless defined $query;

	confess("\@ sign in localname") if $query =~ /\@/;

	my @res;

	$self->query(\@res, 'SELECT c.mail_alias, c.name, c.domain, NULL, p.person
		FROM mail_aliases a
			JOIN persons p ON p.person = a.person
			JOIN mail_aliases c ON c.mail_alias = p.canonical
		WHERE lower(a.name) = lower(?)', $query);

	$self->query(\@res, 'SELECT a.mail_alias, a.name, a.domain, a.addressbook
		FROM mail_aliases a WHERE lower(a.name) = lower(?) AND a.person IS NULL', $query);

	return $self->record2alias(@res);
}

sub fetch_alias {
	my $query = shift;

	confess("fetch_address(): \$query undefined") unless defined $query;

	my ($local, $domain) = split(/\@/, $query, 2);

	confess("fetch_address(): \$query does not contain an \@ sign") unless defined $domain;

	my $res = $self->query('SELECT c.mail_alias, c.name, c.domain, NULL, p.person
		FROM mail_aliases a
			JOIN domainnames d ON d.domain = a.domain
			JOIN persons p ON p.person = a.person
			JOIN mail_aliases c ON c.mail_alias = p.canonical
		WHERE lower(a.name) = lower(?)
			AND d.name = lower(?)', $local, $domain);

	unless(@$res) {
		$res = $self->query('SELECT c.mail_alias, c.name, c.domain, NULL, p.person
			FROM persons p JOIN mail_aliases c ON c.mail_alias = p.canonical
			WHERE lower(p.mailaddress) = lower(?)', $query);
	}

	unless(@$res) {
		$res = $self->query('SELECT a.mail_alias, a.name, a.domain, a.addressbook
			FROM mail_aliases a
				JOIN domainnames d ON d.domain = a.domain
			WHERE lower(a.name) = lower(?)
				AND d.name = lower(?)
				AND a.person IS NULL', $local, $domain);
	}

	return $self->record2alias(@$res)->[0];
}

sub search_alias {
	my $query = shift;

	return [] unless defined $query;

	#my ($local, $domain) = split(/\@/, $query, 2);
	#return $self->search_exact($query) if defined $domain;

	my $escaped = $query;
	$escaped =~ s/([%_])/\\$1/g;

	my %parts; 
	@parts{split(' ', lc($query))} = ();
	return [] unless %parts;

	my %domain_parts;
	my @exact_parts;
	my @substring_where;
	my @substring_parts;

	foreach(keys %parts) {
		if(s/^\@//) {
			my $dom = $self->domain_by_name($_);
			undef $domain_parts{$dom->id} if defined $dom;
		} elsif(/\@/) {
			push @exact_parts, $_;
		} else {
			push @substring_parts, $_;
			s/([%_\\])/\\$1/g;
			push @substring_where, "%$_%";
		}	
	}

	my @domain_where = "a.domain IN (".join(', ', sort keys %domain_parts).")" if %domain_parts;

	# (substring AND substring AND (domain OR domain OR domain)) OR exact OR exact

	# 1) zoek exacte overeenkomsten

	my @res;

	foreach my $exact (@exact_parts) {
		my ($local, $domain) = split(/\@/, $exact, 2);
		next unless defined $domain;
		$domain = $self->domain_by_name($domain);

		if($domain) {
			$domain = $domain->id;
			$self->query(\@res, 'SELECT c.mail_alias, c.name, c.domain, NULL, p.person
				FROM mail_aliases a
					JOIN persons p ON p.person = a.person
					JOIN mail_aliases c ON c.mail_alias = p.canonical
				WHERE lower(a.name) = lower(?) AND a.domain = ?', $local, $domain);

			$self->query(\@res, 'SELECT a.mail_alias, a.name, a.domain, a.addressbook
				FROM mail_aliases a
				WHERE lower(a.name) = lower(?) AND a.domain = ? AND a.person IS NULL', $local, $domain);
		} else {
			$self->query(\@res, 'SELECT c.mail_alias, c.name, c.domain, NULL, p.person
				FROM persons p JOIN mail_aliases c ON c.mail_alias = p.canonical
				WHERE lower(p.mailaddress) = lower(?)', $exact);

			$self->query(\@res, 'SELECT a.mail_alias, a.name, a.domain, a.addressbook
				FROM mail_aliases a
					JOIN external_destinations e USING (mail_alias)
				WHERE lower(e.mailaddress) = lower(?)', $exact);
		}
	}

	# 2) zorg dat de meest obvious match bovenaan komt

	if(@substring_parts == 1) {
		my $substring_part = $substring_parts[0];

		my $where = join(' AND ', @domain_where, 'lower(a.name) = lower(?)');

		$self->query(\@res, "SELECT a.mail_alias, a.name, a.domain, a.addressbook
			FROM mail_aliases a WHERE $where AND a.person IS NULL ORDER BY a.mail_alias", $substring_part);

		$self->query(\@res, "SELECT c.mail_alias, c.name, c.domain, NULL, p.person
			FROM mail_aliases a
				JOIN persons p ON p.person = a.person
				JOIN mail_aliases c ON c.mail_alias = p.canonical
			WHERE $where ORDER BY a.mail_alias", $substring_part);
	}

	# 3) zoek fuzzy overeenkomsten

	my $where = join(' AND ', @domain_where, map { "lower(a.name) LIKE ?" } @substring_parts);
	if($where) {
		$self->query(\@res, "SELECT a.mail_alias, a.name, a.domain, a.addressbook
			FROM mail_aliases a WHERE $where AND a.person IS NULL
			ORDER BY a.mail_alias LIMIT 50", @substring_where);

		$self->query(\@res, "SELECT c.mail_alias, c.name, c.domain, NULL, p.person
			FROM mail_aliases a
				JOIN persons p ON p.person = a.person
				JOIN mail_aliases c ON c.mail_alias = p.canonical
			WHERE $where ORDER BY a.mail_alias LIMIT 50", @substring_where);
	}

	# ontdubbel

	do {
		my %uniq;
		@res = grep { !$uniq{$_->[0]}++ } @res;
	};

	# 4) zoek verwijzingen

	if(@res == 1) {
		my $res = $res[0];
		my ($a, undef, undef, undef, $p) = @$res;

		if($p) {
			$self->query(\@res, 'SELECT a.mail_alias, a.name, a.domain, a.addressbook
				FROM mail_aliases a JOIN personal_destinations r ON r.mail_alias = a.mail_alias
				WHERE r.person = ? AND a.person IS NULL', $p);
		} else {
			$self->query(\@res, 'SELECT a.mail_alias, a.name, a.domain, a.addressbook
				FROM mail_aliases a
					JOIN internal_destinations r ON r.mail_alias = a.mail_alias
				WHERE r.destination = ?', $a);
		}
	}

	# ontdubbel weer

	do {
		my %uniq;
		@res = grep { !$uniq{$_->[0]}++ } @res;
	};

	return $self->record2alias(@res);
}
