use strict;
use warnings FATAL => 'all';

package Class::Lazy::Stub;

use Scalar::Util ();
use Carp ();

# Our own implicit parent is UNIVERSAL; make sure we intercept method
# calls for methods in our parent class as well.
use subs keys %UNIVERSAL::;

sub AUTOLOAD {
	our $AUTOLOAD // die "AUTOLOAD called but \$AUTOLOAD not set";
	my $off = rindex($AUTOLOAD, '::');
	die "no package name in '$AUTOLOAD'" if $off == -1;
	#my $class = substr($AUTOLOAD, 0, $off);
	my $method = substr($AUTOLOAD, $off + 2);
	# don't shift - we want to leave @_ intact
	my ($self) = @_;
	my $class = Scalar::Util::blessed($self) // $self;

	# load the class and any lazy parents
	do {
		# protect $AUTOLOAD from being clobbered
		local $AUTOLOAD;
		my @bases = $class;
		my %bases = ('Class::Lazy::Stub' => undef);
		while(@bases) {
			my $class = shift @bases;
			next if exists $bases{$class};
			undef $bases{$class};
			my $isa = do { no strict qw(refs); \@{"${class}::ISA"} };
			# no @ISA? then it's not lazy either, and therefore not our concern
			next unless @$isa;
			@$isa = grep { $_ ne 'Class::Lazy::Stub' } @$isa;
			# do not call require() if other entries were in @ISA
			# (because that means the class was initialized independently already)
			unless(@$isa) {
				my $file = $class =~ s{::|'}{/}agr . '.pm';
#				warn "loading $class\n" unless exists $INC{$file};
				require $file unless exists $INC{$file};
			}
			push @bases, @$isa;
		}
	};

	my $sub = $self->can($method) || $self->can('AUTOLOAD');
	goto &$sub if $sub;

	my (undef, $filename, $line) = caller;
	die "Can't locate object method \"$method\" via package \"$class\" at $filename line $line.\n";
}

package Class::Lazy;

sub import {
	my $self = shift;
	if($ENV{PERL_CLASS_LAZY_DISABLED}) {
		foreach my $class (@_) {
			my $file = $class =~ s{::|'}{/}agr . '.pm';
			next if exists $INC{$file};
			# just load it eagerly
			require $file;
		}
	} else {
		foreach my $class (@_) {
			my $file = $class =~ s{::|'}{/}agr . '.pm';
			next if exists $INC{$file};
			my $isa = do { no strict qw(refs); \@{"${class}::ISA"} };
			if(@$isa > 1) {
				my (undef, $filename, $line) = caller;
				warn "$filename:$line: class '$class' does not seem to be suitable for loading with Class::Lazy\n";
				@$isa = grep { $_ ne 'Class::Lazy::Stub' } @$isa;
				require $file;
			} elsif(@$isa == 1 && $isa->[0] ne 'Class::Lazy::Stub') {
				my (undef, $filename, $line) = caller;
				warn "$filename:$line: class '$class' does not seem to be suitable for loading with Class::Lazy\n";
				require $file;
			} elsif(@$isa == 0) {
				@$isa = qw(Class::Lazy::Stub);
			}
		}
	}
}

1;

__END__

=encoding utf8

=head1 NAME

Class::Lazy - deferred loading of perl OO classes

=head1 SYNOPSIS

use Class::Lazy qw(Some::Class Some::Other::Class);

my $object = new Some::Class;

=head1 USAGE

Class::Lazy registers a perl OO class without loading the actual code.
The actual code will be loaded when the OO class is first used.

This method of class loading reduces loading time in a situation where you
need to have a large number of perl OO classes available even if you only
use a subset at a given time.

For this to work, the loaded class needs to fulfill a number of conditions:

=over

=item *

It must not contain any functions, only (class) methods;

=item *

It must not try to export anything;

=item *

It must not contain any code that is run when the package is loaded.

=back

If the package does contain functions, exports or tries to run code during
load time, such functionality will simply not be available until the
package is loaded. The only way to trigger a code load is to call one of
the methods of the class.

When a class is actually used and Class::Lazy is forced to load the code,
it does so using "require 'Some/Class.pm'". That means no import functions
are invoked.

=head1 AUTHOR

Wessel Dankers <wsl@uvt.nl>

=head1 COPYRIGHT

Copyright (c) 2015 Tilburg University. This program is free software; you
can redistribute it and/or modify it under the same terms as perl itself.

=cut
