<?php

/**
 * @file
 * Helper functions for cron.
 */

/**
 * Mail to owners for assets that are past retention date.
 */
function _mediamosa_sb_uvt_cron_retention($dry_run = TRUE) {
  global $base_url;

  if ((!$dry_run) && (!variable_get('mediamosa_sb_uvt_retention_cron_mail', FALSE))) {
    return;
  }

  $now = date('Y-m-d');
  $options = array(
    'offset' => 0,
    'limit' => variable_get('mediamosa_sb_uvt_cron_limit_number', 5),
    'is_app_admin' => TRUE,
    'calculate_total_count' => 'TRUE',
    'hide_empty_assets' => 'FALSE',
    'published' => 'ALL',
  );
  $options['cql'] = strtr("retention_date < '@now' and retention_date_mail = NULL",
                    array(
                      '@now' => $now,
                    )
  );
  $results = MediaMosaCkConnectorWrapper::search_asset($options);
  if (isset($results->xml->items->item)) {
    foreach ($results->xml->items->item as $asset) {
      if (isset($asset->asset_id)) {
        if (isset($asset->owner_id)) {
          $owner_id = strtolower(check_plain($asset->owner_id));
          $field = 'mail';
          if (variable_get('mediamosa_ck_asset_owner', 'mail') == 'mail') {
            $field = 'name';
          }
          $owner_mail = db_query("SELECT " . $field . " FROM {users} WHERE lower(mail) = :email",
                        array(
                          ':email' => strtolower(check_plain($owner_id)),
                        )
          )->fetchField();
        }

        $module = 'mediamosa_sb_uvt';
        $key = 'retention_notification';
        $from = variable_get('site_mail', 'admin@example.com');
        $language = language_default();

        $url = $base_url . '/' . mediamosa_sb::get_asset_detail_url((string) $asset->asset_id);

        // Get the asset title.
        $title = '';
        if (isset($asset->dublin_core->title)) {
          $title = (string) $asset->dublin_core->title;
        }

        $params = array(
          'asset_url' => l($url, $url, array('absolute' => TRUE)),
          'asset_id' => $asset->asset_id,
          'asset_title' => $title,
        );
        if (!$dry_run) {
          $result = drupal_mail($module, $key, $owner_mail, $language, $params, $from, TRUE);
          $metadata = array(
            'user_id' => (string) $asset->owner_id,
            'action' => 'update',
            'retention_date_mail' => date('Y-m-d'),
            'is_app_admin' => 'TRUE',
          );
          MediaMosaCkConnectorWrapper::update_metadata($asset->asset_id, $metadata);

          watchdog('uvt', 'Asset: @asset_id, retention notification mail sent to @mail', array('@asset_id' => (string) $asset->asset_id, '@mail' => $owner_mail), WATCHDOG_NOTICE);
          if (!$result) {
            watchdog('uvt', 'Error while sending email to @mail', array('@mail' => $owner_mail), WATCHDOG_NOTICE);
          }
        }
        else {
          drupal_set_message(strtr("Asset: @asset_id, retention notification mail will be sent to @mail", array('@asset_id' => (string) $asset->asset_id, '@mail' => $owner_mail)));
        }
      }
    }
  }
  if ($dry_run) {
    _mediamosa_sb_uvt_cron_retention_delete($dry_run);
  }
}

/**
 * Delete retention assets.
 *
 * Delete all assets that are past retention and mail sent 30 days ago (or
 * longer).
 */
function _mediamosa_sb_uvt_cron_retention_delete($dry_run = TRUE) {

  if ((!$dry_run) && (!variable_get('mediamosa_sb_uvt_retention_cron_delete', FALSE))) {
    return;
  }

  $now = date('Y-m-d');
  $numdays = variable_get('mediamosa_sb_uvt_retention_days', 30);
  $days_ago = date('Y-m-d', strtotime(strtr('-@numdays days', array('@numdays' => $numdays))));

  $options = array(
    'offset' => 0,
    'limit' => variable_get('mediamosa_sb_uvt_cron_limit_number', 5),
    'is_app_admin' => TRUE,
    'calculate_total_count' => 'TRUE',
    'hide_empty_assets' => 'FALSE',
    'published' => 'ALL',
  );
  $options['cql'] = strtr("retention_date < '@now' and retention_date_mail < '@days_ago'",
                    array(
                      '@now' => $now,
                      '@days_ago' => $days_ago,
                    )
  );
  $results = MediaMosaCkConnectorWrapper::search_asset($options);
  if (isset($results->xml->items->item)) {
    foreach ($results->xml->items->item as $asset) {
      if (isset($asset->asset_id)) {
        if (isset($asset->owner_id)) {

          if (!$dry_run) {
            // delete.
            MediaMosaCkConnectorWrapper::delete_asset((string) $asset->asset_id, (string) $asset->owner_id);
            watchdog('uvt', 'Asset: @asset_id deleted', array('@asset_id' => (string) $asset->asset_id), WATCHDOG_NOTICE);
          }
          else {
            drupal_set_message(strtr("Asset: @asset_id will be deleted", array('@asset_id' => $asset->asset_id)));
          }
        }
      }
    }
  }
  if ($dry_run) {
    drupal_set_message("Retention test cron has run.");
    drupal_goto('/admin/config/media/uvt_retention');
  }
}

/**
 * Menu hook callback for retention details.
 */
function _mediamosa_sb_uvt_retention_details($args) {

  $now = date('Y-m-d');
  $numdays = variable_get('mediamosa_sb_uvt_retention_days', 30);
  $days_ago = date('Y-m-d', strtotime(strtr('-@numdays days', array('@numdays' => $numdays))));

  $options = array(
    'offset' => 0,
    'limit' => 200,
    'is_app_admin' => TRUE,
    'calculate_total_count' => 'TRUE',
    'hide_empty_assets' => 'TRUE',
    'published' => 'ALL',
  );
  switch ((string) $args) {
    case 'to_sent':
      $options['cql'] = strtr("(retention_date < '@now') and retention_date_mail = NULL",
                        array(
                          '@now' => $now,
                        )
      );
      drupal_set_title('Mails that will be sent next cron');
      break;

    case 'waiting':
      $options['cql'] = strtr("retention_date < '@now' and retention_date_mail <> NULL",
                        array(
                          '@now' => $now,
                        )
      );
      drupal_set_title('rentention assets that wait for mail action.');
      break;

    case 'to_delete':
      // Determine how many assets will be deleted.
      $options['cql'] = strtr("retention_date < '@now' and retention_date_mail < '@days_ago'",
                        array(
                          '@now' => $now,
                          '@days_ago' => $days_ago,
                        )
      );
      drupal_set_title('rentention assets that will be deleted next cron.');
      break;

    default:
      return;
  }

  $header = array(
    'Asset ID', 'Title', 'Source', 'retention_date', 'retention_date_mail', 'link',
  );
  $rows = array();
  $results = MediaMosaCkConnectorWrapper::search_asset($options);
  foreach ($results->xml->items->item as $item) {
    $rows[] = array(
      isset($item->asset_id) ? $item->asset_id : '',
      isset($item->dublin_core->title) ? l($item->dublin_core->title, 'asset/detail/' . $item->asset_id) : '',
      isset($item->dublin_core->source) ? $item->dublin_core->source : '',
      isset($item->app_{$item->app_id}->retention_date) ? $item->app_{$item->app_id}->retention_date : '',
      isset($item->app_{$item->app_id}->retention_date_mail) ? $item->app_{$item->app_id}->retention_date_mail : '',
      l(t('view'), 'asset/detail/' . $item->asset_id),
    );
  }
  return theme('table', array('header' => $header, 'rows' => $rows));
}
