<?php

/**
 * @file
 * Helper functions for cron.
 */

/**
 * Cql defined queries for retention.
 */
function _mediamosa_sb_uvt_cql_to_sent($now = NULL) {
  if ($now == NULL) {
    $now = date('Y-m-d');
  }
  // Trick for 'and retention_date_mail == NULL'.
  return strtr("(retention_date < '@now') and not retention_date_mail > 0",
    array(
      '@now' => $now,
    ));
}

/**
 * Cql defined queries for retention.
 */
function _mediamosa_sb_uvt_cql_waiting($now = NULL) {
  if ($now == NULL) {
    $now = date('Y-m-d');
  }
  // Trick for 'and retention_date_mail <> NULL'.
  return strtr("(retention_date < '@now') and retention_date_mail > 0",
    array(
      '@now' => $now,
    ));
}

/**
 * Cql defined queries for retention.
 */
function _mediamosa_sb_uvt_cql_to_delete($now = NULL, $days_ago = NULL) {
  if ($now == NULL) {
    $now = date('Y-m-d');
  }
  if ($days_ago == NULL) {
    $numdays = variable_get('mediamosa_sb_uvt_retention_days', 30);
    $days_ago = date('Y-m-d', strtotime(strtr('-@numdays days', array('@numdays' => $numdays))));
  }

  return strtr("(retention_date < '@now') and retention_date_mail  < '@days_ago'",
    array(
      '@now' => $now,
      '@days_ago' => $days_ago,
    ));
}

/**
 * Mail to owners for assets that are past retention date.
 */
function _mediamosa_sb_uvt_cron_retention($dry_run = TRUE) {
  global $base_url;

  if ((!$dry_run) && (!variable_get('mediamosa_sb_uvt_retention_cron_mail', FALSE))) {
    return;
  }

  $options = array(
    'offset' => 0,
    'limit' => variable_get('mediamosa_sb_uvt_cron_limit_number', 5),
    'is_app_admin' => TRUE,
    'calculate_total_count' => 'TRUE',
    'hide_empty_assets' => 'FALSE',
    'published' => 'ALL',
  );
  $options['cql'] = _mediamosa_sb_uvt_cql_to_sent();
  $options['order_by'] = 'retention_date';
  $options['order_direction'] = 'asc';

  $results = MediaMosaCkConnectorWrapper::search_asset($options);
  if (isset($results->xml->items->item)) {
    foreach ($results->xml->items->item as $asset) {
      if (isset($asset->asset_id)) {
        if (isset($asset->owner_id)) {
          $owner_id = strtolower(check_plain($asset->owner_id));
          if (variable_get('mediamosa_ck_asset_owner', 'mail') == 'mail') {
            $owner_mail = $owner_id;
          }
          else {
            $owner_mail = db_query("SELECT mail FROM {users} WHERE lower(mail) = :email",
                          array(
                            ':email' => strtolower(check_plain($owner_id)),
                          )
            )->fetchField();
          }
        }

        $module = 'mediamosa_sb_uvt';
        $key = 'retention_notification';
        $from = variable_get('site_mail', 'admin@example.com');
        $language = language_default();

        $url = $base_url . '/' . mediamosa_sb::get_asset_detail_url((string) $asset->asset_id);

        // Get the asset title.
        $title = '';
        if (isset($asset->dublin_core->title)) {
          $title = (string) $asset->dublin_core->title;
        }

        $params = array(
          'asset_url' => l($url, $url, array('absolute' => TRUE)),
          'asset_id' => $asset->asset_id,
          'asset_title' => $title,
        );
        if (!$dry_run) {
          $result = drupal_mail($module, $key, $owner_mail, $language, $params, $from, TRUE);
          $metadata = array(
            'user_id' => (string) $asset->owner_id,
            'action' => 'update',
            'retention_date_mail' => date('Y-m-d'),
            'is_app_admin' => 'TRUE',
          );
          MediaMosaCkConnectorWrapper::update_metadata($asset->asset_id, $metadata);

          watchdog('uvt', 'Asset: @asset_id, retention notification mail sent to @mail', array('@asset_id' => (string) $asset->asset_id, '@mail' => $owner_mail), WATCHDOG_NOTICE);
          if (!$result) {
            watchdog('uvt', 'Error while sending email to @mail', array('@mail' => $owner_mail), WATCHDOG_NOTICE);
          }
        }
        else {
          drupal_set_message(strtr("testrun: retention notification mail will be sent to @mail", array('@mail' => $owner_mail)));
        }
      }
    }
  }
  if ($dry_run) {
    _mediamosa_sb_uvt_cron_retention_delete($dry_run);
  }
}

/**
 * Delete retention assets.
 *
 * Delete all assets that are past retention and mail sent 30 days ago (or
 * longer).
 */
function _mediamosa_sb_uvt_cron_retention_delete($dry_run = TRUE) {

  if ((!$dry_run) && (!variable_get('mediamosa_sb_uvt_retention_cron_delete', FALSE))) {
    return;
  }

  $options = array(
    'offset' => 0,
    'limit' => variable_get('mediamosa_sb_uvt_cron_limit_number', 5),
    'is_app_admin' => TRUE,
    'calculate_total_count' => 'TRUE',
    'hide_empty_assets' => 'FALSE',
    'published' => 'ALL',
  );
  $options['cql'] = _mediamosa_sb_uvt_cql_to_delete();
  $options['order_by'] = 'retention_date';
  $options['order_direction'] = 'asc';

  $results = MediaMosaCkConnectorWrapper::search_asset($options);
  if (isset($results->xml->items->item)) {
    foreach ($results->xml->items->item as $asset) {
      if (isset($asset->asset_id)) {
        if (isset($asset->owner_id)) {

          if (!$dry_run) {
            // delete.
            MediaMosaCkConnectorWrapper::delete_asset((string) $asset->asset_id, (string) $asset->owner_id);
            watchdog('uvt', 'Asset: @asset_id deleted', array('@asset_id' => (string) $asset->asset_id), WATCHDOG_NOTICE);
          }
          else {
            drupal_set_message(strtr("Asset: @asset_id will be deleted", array('@asset_id' => $asset->asset_id)));
          }
        }
      }
    }
  }
  if ($dry_run) {
    drupal_set_message("Retention test cron has run.");
    drupal_goto('/admin/config/media/uvt_retention');
  }
}

/**
 * Menu hook callback for retention details.
 */
function _mediamosa_sb_uvt_retention_details($args) {

  $limit = 50;
  $offset = 0;
  if (isset($_GET['page'])) {
    $offset = (int) $_GET['page'] * $limit;
  }

  $options = array(
    'offset' => $offset,
    'limit' => $limit,
    'is_app_admin' => TRUE,
    'calculate_total_count' => 'TRUE',
    'hide_empty_assets' => 'TRUE',
    'published' => 'ALL',
  );
  switch ((string) $args) {
    case 'to_sent':
      $options['cql'] = _mediamosa_sb_uvt_cql_to_sent();
      drupal_set_title('Mails that will be sent next cron');
      break;

    case 'waiting':
      $options['cql'] = _mediamosa_sb_uvt_cql_waiting();
      drupal_set_title('rentention assets that wait for mail action.');
      break;

    case 'to_delete':
      // Determine how many assets will be deleted.
      $options['cql'] = _mediamosa_sb_uvt_cql_to_delete();
      drupal_set_title('rentention assets that will be deleted next cron.');
      break;

    default:
      return;
  }

  $header = array(
    'Title',
    array('data' => 'Owner', 'field' => 'Owner', 'sort' => 'ASC'),
    'Source',
    array(
      'data' => 'Retention Date',
      'field' => 'retention_date',
      'sort' => 'asc',
    ),
    'retention_date_mail', 'link',
  );
  $rows = array();

  if ((isset($_GET['order'])) && ($_GET['order'] == 'Owner')) {
    $options['order_by'] = 'owner_id';
    if ($_GET['sort'] == 'asc') {
      $options['order_direction'] = 'asc';
    }
    else {
      $options['order_direction'] = 'desc';
    }
  }
  if ((isset($_GET['order'])) && ($_GET['order'] == 'Retention Date')) {
    $options['order_by'] = 'retention_date';
    if ($_GET['sort'] == 'asc') {
      $options['order_direction'] = 'asc';
    }
    else {
      $options['order_direction'] = 'desc';
    }
  }

  // Sort default by retention date.
  if (!isset($options['order_by'])) {
    $options['order_by'] = 'retention_date';
    $options['order_direction'] = 'asc';
  }

  $results = MediaMosaCkConnectorWrapper::search_asset($options);
  foreach ($results->xml->items->item as $item) {
    $retention_date = isset($item->{'app_' . $item->app_id}->retention_date) ? (string) $item->{'app_' . $item->app_id}->retention_date : '';
    $retention_date_email = isset($item->{'app_' . $item->app_id}->retention_date_mail) ? $item->{'app_' . $item->app_id}->retention_date_mail : '';
    $rows[] = array(
      isset($item->dublin_core->title) ? l($item->dublin_core->title, 'asset/detail/' . $item->asset_id) : '',
      isset($item->owner_id) ? $item->owner_id : '',
      isset($item->dublin_core->source) ? $item->dublin_core->source : '',
      l($retention_date, 'admin/config/media/uvt_retention/edit/' . $item->asset_id),
      l($retention_date_email, 'admin/config/media/uvt_retention/edit/' . $item->asset_id),
      l(t('view'), 'asset/detail/' . $item->asset_id),
    );
  }

  $total = $results->get_header_item_count_total();
  pager_default_initialize($total, $limit);

  return theme('table', array('header' => $header, 'rows' => $rows)) . theme('pager', array('quantity' => $limit));
}

/**
 * Menu hook callback for retention edit.
 */
function _mediamosa_sb_uvt_retention_edit($form, $form_state, $asset_id) {

  $form = array();

  $asset = MediaMosaSbConnectorWrapper::get_asset((string) $asset_id, array('show_stills' => FALSE, 'show_collections' => FALSE));
  $item = $asset->items->item;
  $retention_date = isset($item->{'app_' . $item->app_id}->retention_date) ? (string) $item->{'app_' . $item->app_id}->retention_date : '';
  $retention_date_mail = isset($item->{'app_' . $item->app_id}->retention_date_mail) ? $item->{'app_' . $item->app_id}->retention_date_mail : '';

  $form['retention_date'] = array(
    '#title' => t('Retention date'),
    '#default_value' => (string) $retention_date,
    '#value' => (string) $retention_date,
    '#type' => 'textfield',
    '#size' => 60,
    '#maxlength' => 60,
  );
  $form['retention_date_mail'] = array(
    '#title' => t('Retention date mailed'),
    '#default_value' => (string) $retention_date_mail,
    '#value' => (string) $retention_date_mail,
    '#type' => 'textfield',
    '#size' => 60,
    '#maxlength' => 60,
  );
  $form['asset_id'] = array(
    '#default_value' => (string) $asset_id,
    '#value' => (string) $asset_id,
    '#type' => 'hidden',
  );

  $form['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Save'),
  );

  return $form;
}

/**
 * Implements hook_submit().
 */
function _mediamosa_sb_uvt_retention_edit_submit($form, &$form_state) {

  $metadata = array(
    'action' => 'update',
    'retention_date' => $form_state['input']['retention_date'],
    'retention_date_mail' => $form_state['input']['retention_date_mail'],
    'is_app_admin' => 'TRUE',
    'user_id' => 'update',
  );

  MediaMosaCkConnectorWrapper::update_metadata((string) $form_state['input']['asset_id'], $metadata);

  drupal_goto('admin/config/media/uvt_retention/details');
}
