'''
Copyright 2014 Cisco Systems, Inc.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

'''
from UcsSdk import * 
import os
import re
import sys

try:
    import xml.etree.ElementTree as ET
    from xml.etree.ElementTree import Element, SubElement
except ImportError:
    import cElementTree as ET
    from cElementTree import Element, SubElement
from UcsNagiosGetEquipmentDetails import UcsNagiosGetEquipmentDetails
from base64 import b64encode
import csv
from PropertyHandler import PropertyHandlerClass, CheckForCommentOrBlank
from random import sample



# Class for handling all the file related activities

class UcsNagiosFileHandler(object):

    def __init__(self, ucs_handle, ucsDict):
        '''
        Constructor
        '''
        self.handle = ucs_handle
        self.ucs_ip = ucs_handle._name
        self.user = ucs_handle._username
        self.password = ucs_handle._password
        self.port = ucs_handle._port
        self.no_ssl = ucs_handle._noSsl
        self.proxy = ucs_handle._proxy
        self.ucs_dict = ucsDict
        self.NAGIOS_HOME = self.plugin_cfg_dict['NAGIOS_HOME']
        self.host_group_xml_file = self.replace_vars(self.plugin_cfg_dict['host_group_xml_file'])
        self.host_group_file = self.replace_vars(self.plugin_cfg_dict['host_group_file'])
        self.service_xml_file = self.replace_vars(self.plugin_cfg_dict['service_xml_file'])
        self.service_cfg_file = self.replace_vars(self.plugin_cfg_dict['service_cfg_file'])
        self.host_defination_file = self.replace_vars(self.plugin_cfg_dict['host_defination_file'])
        self.cmd_conf_file = self.replace_vars(self.plugin_cfg_dict['cmd_conf_file'])
        self.host_ext_info_file = self.replace_vars(self.plugin_cfg_dict['host_ext_info_file'])
        self.host_ext_info_xml_file = self.replace_vars(self.plugin_cfg_dict['host_ext_info_xml_file'])
        self.get_equipment_details_object = UcsNagiosGetEquipmentDetails(ucs_handle)
        self.user_password_string = "\"" + ucs_handle._username + "\"" + "!"\
        +"\""+b64encode(ucs_handle._password)+"\""+"!"+"\""
        self.default_class_list = "EquipmentChassis,ComputeBlade,EquipmentFex,NetworkElement,ComputeRackUnit,EquipmentIOCard,EquipmentSwitchIOCards"
        try:
            class_list_from_cfg = self.plugin_cfg_dict['HOST_CLASS_LIST']
            self.default_class_list+= class_list_from_cfg
        except:
            pass

    # method to resolve ENV and Code specific keywords
    # This will dynamically generate the variables and assign the appropriate 
    # values from the
    def replace_vars(self, filePath):
        error_config = False
        error_env = False
        updated_file_path = filePath
        var_list = re.findall(r'.*?<(.*?)>', filePath)
        for var in var_list:
            if var == 'UCS_IP':
                UCS_IP = self.ucs_ip
                updated_file_path = re.sub('<' + var + '>',
                                           eval(var),
                                           updated_file_path)
                continue
            if var == 'UCS_PASSWORD':
                UCS_PASSWORD = self.password
                updated_file_path = re.sub('<' + var + '>',
                                           eval(var),
                                           updated_file_path)
                continue
            try:
                val = self.plugin_cfg_dict[var]
                exec("%s='%s'") % (var, val)
                updated_file_path = re.sub('<' + var + '>',
                                           eval(var),
                                           updated_file_path)
            except:
                error_config = True
            # Find it in the env else look up in local
            try:
                val = os.environ[var]
                exec("%s='%s'") % (var, val)
                updated_file_path = re.sub('<' + var + '>',
                                           eval(var),
                                           updated_file_path)
            except KeyError: 
                error_env = True
            if error_config == True & error_env == True:
                print(" Unable to find the configuration variable in the Local or in the environment context.\n\
                Please set the variable %s as environment variable or define the same in the configuration file.") %var
                return None
        return updated_file_path.strip('"')


    def create_file(self, file_name):
        if file_name != None:
            file_name = file_name.strip('"')
        #print ("File Name %s ") % file_name
        dir_path = os.path.dirname(file_name)
        if not os.path.exists(dir_path):
            os.makedirs(dir_path)
        if not os.path.exists(file_name):
            f = file(file_name, 'w+')
        else:
            if file_name != self.host_ext_info_xml_file and file_name !=self.host_group_xml_file:
                f = file(file_name, 'w+')
            else:
                f = file(file_name, 'r+')
        return f

    # Creating the Base host file
    def create_base_host_file(self):
        key = self.ucs_ip
        self.ucs_dict[key, 'alias'] = ""
        self.ucs_dict[key, 'host_name'] = self.ucs_ip
        self.ucs_dict[key, 'host_ip'] = self.ucs_ip
        self.ucs_dict[key, 'parent'] = 'Cisco-UCS'
        self.ucs_dict[key, 'class_id'] = 'host'
        self.ucs_dict[key, 'Dn'] = 'sys'
        self.ucs_dict[self.ucs_ip, 'host_group'] = [['Cisco-UCS', self.ucs_ip]]

    # Method for providing the Host structure 
    def get_host_file_text(self, host_name, alias, parent, ip) :
        if parent == 'Cisco-UCS' :
            str = ('define host { \n \
            use                     cisco-ucs-host \n\
            host_name               %s \n\
            alias                   %s \n\
            address                 %s \n\
            }' % (host_name, self.ucs_ip + alias, ip))
        else :
            str = ('define host { \n \
            use                     cisco-ucs-host \n\
            host_name               %s \n\
            alias                   %s \n\
            parents                 %s \n\
            address                 %s \n\
            }' % (host_name, self.ucs_ip + alias, parent, ip))
        return str

    # Method for creating UCS Host File
    def create_host_file(self,host_nagiosxi):
        ucs_obj_list = []
        for key in self.ucs_dict.keys():
            ucs_object = key[0]
            host_type = self.ucs_dict[ucs_object, 'class_id']
            if ucs_object in ucs_obj_list:
                continue
            else:
                ucs_obj_list.append(ucs_object)
                ucs_file_cfg = self.NAGIOS_HOME + "/cisco/ucsObjs/" + "/" + self.ucs_ip + "/" + self.ucs_ip + self.ucs_dict[ucs_object, 'alias'] + ".cfg"
                if host_nagiosxi and self.ucs_dict[ucs_object, 'alias'] != "":
                    try:
                        os.remove(ucs_file_cfg)
                    except:
                        continue
                else:
                    if host_type not in self.default_class_list and host_type != "host":
                        continue
                    print "Creating UCS Host Configuration File : " + ucs_file_cfg
                    ucs_host_cfg_fd = self.create_file(ucs_file_cfg)
                    ucs_host_cfg_fd.write(self.get_host_file_text(self.ucs_dict[ucs_object, 'host_name'],
                                                                  self.ucs_dict[ucs_object, 'alias'],
                                                                  self.ucs_dict[ucs_object, 'parent'],
                                                                  self.ucs_dict[ucs_object, 'host_ip']))
                    ucs_host_cfg_fd.close()

    def get_wcard_filter_for_dn(self, dn, class_id):
        in_filter = FilterFilter()
        wcard_filter = WcardFilter()
        wcard_filter.Class = class_id
        wcard_filter.Property = UcsUtils.WordL(NamingId.DN)
        wcard_filter.Value = dn + "*"
        in_filter.AddChild(wcard_filter)
        return in_filter

    def get_ucs_service(self, dn, method_list):
        if method_list[1] == None:
            service_list = []
            service_name = "Check ping for UCS"
            service_cmd = "check-host-alive"
            service_list.append([service_name, service_cmd])
            return service_list
        else:
            class_id = method_list[1]
            in_filter = self.get_wcard_filter_for_dn(dn, class_id)
            service_list = []
            get_response = self.handle.ConfigResolveClass(class_id, in_filter=in_filter)
            for mo in get_response.OutConfigs.GetChild():
                moDn = mo.Dn
                service_name = method_list[0] + moDn.replace('sys/', '_').replace("/", '_')
                service_cmd = "check_cisco_ucs!" + self.user_password_string + moDn + "\""
                service_list.append([service_name, service_cmd])
            return service_list

    def get_service_for_type(self, hostType, dn):
        service_list = []
        if hostType == 'host':
            service_name = "Ping UCS"
            service_cmd = "check-host-alive"
            service_list.append([service_name, service_cmd])
            return service_list

        host_service = "Service_" + str(hostType.strip())
        try:
            service_list_for_host = self.plugin_cfg_dict[host_service].split(",")
        except Exception:
            print "." * 80
            print self.plugin_cfg_dict[host_service].split(",")
            print("WARNING : Skipping service definition for host type '%s' and dn '%s'")%(hostType, dn)
            print("Either no service is defined or check the service format definition in the auto-discovery configuration file.") 
            print("It should  be of the format <Service Name>:<Class or DN> :<[Optional] additional plugin cli options>")
            print "." * 80
            return service_list

        if service_list_for_host:
            for service_element in service_list_for_host:
                service_var_list = service_element.split(":")
                try:
                    service_name = service_var_list[0].strip("'").strip() + dn.replace('sys/',' - ')
                    if service_name == "":
                        raise Exception()
                    service_type = service_var_list[1].strip().strip("'")
                    if service_type == "":
                        raise Exception()
                except:
                    print "." * 80
                    print("ERROR : Skipping service definition for host type '%s' and dn '%s'")%(hostType, dn)
                    print ("Check the service format defined in the auto-discovery configuration file.")%dn
                    print("It should  be of the format <Service Name>:<Class or DN> :<[Optional] additional plugin cli options>") 
                    print "." * 80
                    return service_list
                try:
                    cli_options = service_var_list[2].strip().strip("'").strip("\"")
                except:
                    cli_options = ""

                if cli_options != "":
                    service_list_temp = self.get_equipment_details_object.create_ucs_service(service_name, service_type,dn)[0]
                    service_name = service_list_temp[0]
                    service_value = service_list_temp[1]
                    #action_value = service_list_temp[2]
                    service_value = service_value + " " + cli_options
                    #service_list.extend([[service_name, service_value, action_value]])
                    service_list.extend([[service_name, service_value]])
                else:
                    service_list.extend(self.get_equipment_details_object.create_ucs_service(service_name, service_type, dn))
        return service_list

    def get_host_ext_info_text(self, host_ext_info, icon_image ,icon_image_alt,status_map_image) :
        text_sring = ('define hostextinfo { \n\
            host_name              %s \n\
            icon_image             %s \n\
            icon_image_alt        %s \n\
            statusmap_image        %s \n\
            }\n' % (host_ext_info, icon_image, icon_image_alt, status_map_image))
        # print text_sring
        return text_sring

    def get_host_group_text(self, host_group_name, member_name):
        text_sring = ('define hostgroup { \n\
            hostgroup_name          %s \n\
            alias                   %s \n\
            members                 %s \n\
            }\n' % (host_group_name, host_group_name, member_name))
        # print text_sring
        return text_sring

    def get_service_text(self, host_name, service_name, check_cmd):
        rnd_time = sample(xrange(5,15), 1)
        use_parameter= "cisco-ucs-service"
        text_sring = ('define service{ \n\
            use                      %s \n\
            host_name                %s \n\
            service_description      %s \n\
            check_command            %s \n\
            check_interval            %s \n\
            }\n' % (use_parameter,host_name, service_name, check_cmd, rnd_time[0]))
        # print text_sring
        return text_sring

    # Input for this function is the hostname and its type
    def create_service_xml_file(self,host_nagiosxi):
        service_xml_fd = self.create_file(self.service_xml_file)
        xml_text = ""
        try:
            xml_text = service_xml_fd.read()
        except Exception, err:
            print ("Exception while creating : %s " % (self.service_xml_file)), str(err)
            service_xml_fd.close() 
        if xml_text == "":
            host_names_node = Element('hostnames')
            service_name_list = []
        else:
            tree = ET.parse(self.service_xml_file)
            host_names_node = tree.getroot()
            service_name_list = []
            for node in host_names_node.findall("hostname/service"):
                service_name_list.append(node.text) 

        for ucs_host in self.ucs_dict.keys():
            if len(ucs_host) != 2 :
                continue
            host_name_from_dict = ucs_host[0]
            host_type = self.ucs_dict[host_name_from_dict, 'class_id']
            dn = self.ucs_dict[host_name_from_dict, 'Dn']
            service_list = self.get_service_for_type(host_type,dn)
            for key1, value1 in service_list :
                if key1 in service_name_list :
                    # Key already exists we just need to add the service node
                    # print "Service already exist : " + key1
                    continue
                else:
                    service_name_list.append(key1)
                    #print "Hostname : " + host_name_from_dict + " \n Host Type  : " + host_type + " \n DN : " + dn
                    try:
                        matching_node = [node for node in host_names_node.findall("hostname") if node.text == host_name_from_dict][0]
                    except Exception, err:
                        #print "Exception while iterating the Matching node list:", str(err)
                        matching_node = SubElement(host_names_node, 'hostname')
                        if host_nagiosxi or (host_type not in self.default_class_list):
                            matching_node.text = host_name_from_dict.split("_")[0]
                        else:
                            matching_node.text = host_name_from_dict

                    service_node = SubElement(matching_node, 'service')
                    service_node.text = key1
                    cmd_node = SubElement(service_node, 'cmd')
                    if self.proxy == None:
                        if self.no_ssl:
                            cmd_node.text = value1+"!"+" --port "+str(self.port)+\
                            " --NoSsl "
                        else:
                            cmd_node.text = value1+"!"+" --port "+str(self.port)
                    else:
                        if self.no_ssl:
                            cmd_node.text = value1+"!"+" --port "+str(self.port)+\
                                        " --NoSsl --proxy "+self.proxy 
                        else:
                            cmd_node.text = value1+"!"+" --port "+str(self.port)+\
                                        " --proxy "+self.proxy
                    #action_node =SubElement(service_node, 'actionNode')
                    #if action == "":
                    #    action_node.text = "None"
                    #else:
                    #    action_node.text = host_name_from_dict.split("_")[0]+ action
                    #print tostring(host_names_node) 
        tree = ET.ElementTree(host_names_node)
        tree.write(self.service_xml_file)

    def convert_service_xml_to_nagios_file(self):
        tree = ET.parse(self.service_xml_file)
        host_names = tree.getroot()
        print "Creating Service Configuration File : " + self.service_cfg_file
        service_text = ""
        for host_name_node in host_names.findall("hostname"):
            for service_node in host_name_node.findall("service"):
                for cmd_node in service_node.findall("cmd"):
                   # for action_node in service_node.findall("actionNode"):
                        print "Adding Service For : " + service_node.text
                        #service_text += self.get_service_text(host_name_node.text, service_node.text, cmd_node.text,action_node.text)
                        service_text += self.get_service_text(host_name_node.text, service_node.text, cmd_node.text)
        service_cfg_fd = self.create_file(self.service_cfg_file)
        service_cfg_fd.write(service_text)
        service_cfg_fd.close()
        pass

    def create_host_group_xml_file(self, ucs_ip):
        host_group_fd = self.create_file(self.host_group_xml_file)
        xml_text = ""
        try:
            xml_text = host_group_fd.read()
        except Exception, err:
            print ("Exception while creating Host Group file %s:" % (self.host_group_xml_file)), str(err)
            host_group_fd.close() 
        if xml_text == "":
            hostgroups = Element('hostgroups')
            key_list = []
        else :
            tree = ET.parse(self.host_group_xml_file)
            hostgroups = tree.getroot()
            key_list = []
            for node in hostgroups.findall("hostgroup/hostgroup_name") :
                key_list.append(node.text) 
        for key, value in self.ucs_dict[ucs_ip, 'host_group'] :
            if key == "":
                continue
            if key == value:
                continue
            if key in key_list :
                # Key already exists we just need to add the member
                matching_node = [node for node in hostgroups.findall("hostgroup/hostgroup_name") if node.text == key][0]
                #for elem in matching_node.iterfind('members') :
                for elem in matching_node.getiterator():
                    if elem.tag != 'members' :
                        continue
                    add_node = True
                    for memberNode in elem.getiterator() :
                        if memberNode.text == value :
                            add_node = False
                    if add_node == True :    
                        member = SubElement(elem, 'member')
                        member.text = value
            else :
                key_list.append(key)
                hostgroup = SubElement(hostgroups, 'hostgroup',)
                hostgroup_name = SubElement(hostgroup, 'hostgroup_name',)
                hostgroup_name.text = key
                alias = SubElement(hostgroup_name, 'alias')
                alias.text = key
                members = SubElement(hostgroup_name, 'members')
                member = SubElement(members, 'member')
                member.text = value
        tree = ET.ElementTree(hostgroups)
        tree.write(self.host_group_xml_file)
    
    def update_host_group_xml_file(self,ucs_ip,disable_host_group=False):  
        host_group_fd = self.create_file(self.host_group_xml_file)
        xml_text = ""
        remove_list = list()
        try:
            xml_text = host_group_fd.read()
        except Exception, err:
            print ("Exception while updating Host Group file %s:" % (self.host_group_xml_file)), str(err)
            host_group_fd.close()
        if xml_text == "":
            #hostgroups = Element('hostgroups')
            key_list = []
        else :
            tree = ET.parse(self.host_group_xml_file)
            hostgroups = tree.getroot()
            key_list = []
            for node in hostgroups.findall("hostgroup/hostgroup_name") :
                key_list.append(node.text)
            matching = [s for s in key_list if ucs_ip in s]
            hostgroup = tree.findall("hostgroup")
            for host in hostgroup:
                for host_name in host:
                    if host_name.text in matching:
                        hostgroups.remove(host)
            tree = ET.ElementTree(hostgroups)
            tree.write(self.host_group_xml_file)
            if disable_host_group:
                self.remove_member_host_xml(self.host_group_xml_file,"Cisco-UCS",ucs_ip)
        host_group_fd.close()

    def remove_member_host_xml(self,host_group_file, hostgroup_text, member_text):
        tree = ET.parse(self.host_group_xml_file)        
        root = tree.getroot() 
        for hostgroup in root:
            for hostgroup_name in hostgroup:
                if hostgroup_name.text == hostgroup_text:
                    for members in hostgroup_name:
                        if members.tag == "members":
                            member_list = []
                            for member in members:
                                if member.text == member_text:
                                    member_list.append(member)
                            for member in member_list:
                                members.remove(member)
                            if len(members.findall("member")) == 0:
                                root.remove(hostgroup)
                                break
        tree = ET.ElementTree(root)
        tree.write(self.host_group_xml_file)
 
    
    def convert_host_group_xml_to_nagios_file(self):
        if not os.stat(self.host_group_xml_file).st_size==0:
            print "Creating Host Group Configuration File : " + self.host_group_file
            tree = ET.parse(self.host_group_xml_file)
            hostgroups = tree.getroot()
            host_group_text = ""
            for host_group_node in hostgroups.findall("hostgroup") :
                member_text = ""
                for node in host_group_node.findall("hostgroup_name/members/member") :
                    member_text += node.text + ","
                host_group_text += self.get_host_group_text(host_group_node[0].text, member_text.strip(","))
            host_group_fd = self.create_file(self.host_group_file)
            host_group_fd.write(host_group_text)
            host_group_fd.close()

    def convert_host_ext_info_xml_to_nagios_file(self):
        print "Creating Host Ext info Configuration File : " + self.host_ext_info_file
        tree = ET.parse(self.host_ext_info_xml_file)
        hostextinfo = tree.getroot()
        host_ext_info_text = ""
        for host_ext_info_node in hostextinfo.findall("hostextinfo_name") :
            # Find the member list
            member_text = ""
            for node in host_ext_info_node.findall("members/member") :
                member_text += node.text + ","
            # Get the Icon_Image
            icon_image = host_ext_info_node.find("icon_image").text
            #Get the Icon_Image_alt
            icon_image_alt = host_ext_info_node.find("icon_image_alt").text
            #Get the status_image
            status_map_image =host_ext_info_node.find("statusmap_image").text
            
            host_ext_info_text += self.get_host_ext_info_text(member_text.strip(","), icon_image ,icon_image_alt,status_map_image)
        host_ext_info_fd = self.create_file(self.host_ext_info_file)
        host_ext_info_fd.write(host_ext_info_text)
        host_ext_info_fd.close()

    def update_host_ext_xml_file(self,ucs_ip,disable_host_group=False):
        host_ext_fd = self.create_file(self.host_ext_info_xml_file)
        xml_text = ""
        remove_list = list()
        try:
            xml_text = host_ext_fd.read()
        except Exception, err:
            print ("Exception while updating Host Ext file %s:" % (self.host_ext_info_xml_file)), str(err)
            host_group_fd.close()
        if xml_text == "":
            #hostgroups = Element('hostgroups')
            key_list = []
        else :
            tree = ET.parse(self.host_ext_info_xml_file)
            hostextinfo = tree.getroot()
            key_list = []
            for node in hostextinfo.findall("hostextinfo_name/members/member"):
                key_list.append(node.text)
            matching_key = ucs_ip + "_"
            matching = [s for s in key_list if matching_key in s]
            extname_list = []
            for extname in hostextinfo:
                for members in extname:
                    if members.tag == "members":
                        member_list = []
                        for member in members:
                            if member.text in matching:
                                member_list.append(member)
                        for member in member_list:
                            members.remove(member)
                        if len(members.findall("member")) == 0:
                            extname_list.append(extname)
            for extname in extname_list:
                hostextinfo.remove(extname)
                               
            tree = ET.ElementTree(hostextinfo)
            tree.write(self.host_ext_info_xml_file)

    def get_host_ext_data_for_type(self, class_id):
        #if class_id 
        #TODO define this dict in the properties file
        type_dict = {'host': {'icon_image':'CISCO.UCS.R2.Domain.40x40Image.png',
                      'icon_image_alt': 'CiscoUCS' ,
                      'statusmap_image' : 'CISCO.UCS.R2.Domain.40x40Image.png' },
                    'NetworkElement':{'icon_image':'CISCO.UCS.R2.FabricInterconnect.40x40Image.png',
                                      'icon_image_alt': 'FabricInterconnect' ,
                                      'statusmap_image' : 'CISCO.UCS.R2.FabricInterconnect.40x40Image.png'},
                    'EquipmentChassis':{'icon_image':'CISCO.UCS.R2.Chassis.40x40Image.png',
                                        'icon_image_alt': 'CiscoUCSChassis' ,
                                        'statusmap_image' : 'CISCO.UCS.R2.Chassis.40x40Image.png'},
                    'ComputeBlade' :{'icon_image':'CISCO.UCS.R2.Blade.40x40Image.png',
                                     'icon_image_alt': 'CiscoUCSBlade' ,
                                     'statusmap_image' : 'CISCO.UCS.R2.Blade.40x40Image.png'},
                    'EquipmentFex':{'icon_image':'CISCO.UCS.R2.Fex.40x40Image.png',
                                    'icon_image_alt': 'FabricExtender' ,
                                    'statusmap_image' : 'CISCO.UCS.R2.Fex.40x40Image.png'},
                    'ComputeRackUnit' :{'icon_image':'CISCO.UCS.R2.RackUnit.40x40Image.png',
                                        'icon_image_alt': 'CiscoUCSRack' ,
                                        'statusmap_image' : 'CISCO.UCS.R2.RackUnit.40x40Image.png'},
                    'EquipmentIOCard' :{'icon_image':'CISCO.UCS.R2.IOM.40x40Image.png',
                                        'icon_image_alt': 'CiscoUCSIOM' ,
                                        'statusmap_image' : 'CISCO.UCS.R2.IOM.40x40Image.png'},
                     'EquipmentSwitchIOCard' :{'icon_image':'CISCO.UCS.R2.IOM.40x40Image.png',
                                        'icon_image_alt': 'CiscoUCSIOM' ,
                                        'statusmap_image' : 'CISCO.UCS.R2.IOM.40x40Image.png'},
                    }
        try:
            if type_dict[class_id]:
                pass
        except Exception, err:
            type_dict[class_id] = {'icon_image':'CISCO.UCS.R2.Domain.40x40Image.png',
                                    'icon_image_alt': 'CiscoUCSService' ,
                                    'statusmap_image' : 'CISCO.UCS.R2.Domain.40x40Image.png'}
        return type_dict[class_id]

    def remove_non_required_host_files(self):
        dir_path = self.NAGIOS_HOME + "/cisco/ucsObjs/" + "/" + self.ucs_ip + "/"
        file_names = [] 
        try:
            file_names = os.listdir(dir_path)
        except:
            return
        tree = ET.parse(self.service_xml_file)
        host_names = tree.getroot()
        file_names.remove("services.cfg")
        for host_name_node in host_names:
            if (host_name_node.text + ".cfg") in file_names:
                file_names.remove((host_name_node.text + ".cfg"))
        for names in file_names:
            try:
                full_path = dir_path + "/" + names
                os.remove(full_path)
            except:
                pass
        self.remove_non_required_hosts_from_host_ext_info_file(file_names)
        self.remove_non_required_hosts_from_host_group_info_file(file_names)
    
    
    def remove_non_required_hosts_from_host_group_info_file(self,file_names):
        matching =[]
        for names in file_names:
            matching.append(names.split(".cfg")[0])
        host_group_fd = self.create_file(self.host_group_xml_file)
        xml_text = ""
        remove_list = list()
        try:
            xml_text = host_group_fd.read()
        except Exception, err:
            print ("Exception while updating Host Group file %s:" % (self.host_group_xml_file)), str(err)
            host_group_fd.close()
        if xml_text == "":
            key_list = []
        else :
            tree = ET.parse(self.host_group_xml_file)
            root = tree.getroot()
            for hostgroup in root:
                for hostgroup_name in hostgroup:
                    for members in hostgroup_name:
                        if members.tag == "members":
                            member_list = []
                            for member in members:
                                if member.text in matching:
                                    member_list.append(member)
                            for member in member_list:
                                members.remove(member)
                            if len(members.findall("member")) == 0:
                                root.remove(hostgroup)
                                break
            tree = ET.ElementTree(root)
            tree.write(self.host_group_xml_file)

    def remove_non_required_hosts_from_host_ext_info_file(self,file_names):
        matching =[]
        for names in file_names:
            matching.append(names.split(".cfg")[0])
        host_ext_fd = self.create_file(self.host_ext_info_xml_file)
        xml_text = ""
        remove_list = list()
        try:
            xml_text = host_ext_fd.read()
        except Exception, err:
            print ("Exception while updating Host Ext file %s:" % (self.host_ext_info_xml_file)), str(err)
            host_ext_fd.close()
        if xml_text == "":
            key_list = []
        else :
            tree = ET.parse(self.host_ext_info_xml_file)
            hostextinfo = tree.getroot()
            key_list = []
            for node in hostextinfo.findall("hostextinfo_name/members/member"):
                key_list.append(node.text)
            extname_list = []
            for extname in hostextinfo:
                for members in extname:
                    if members.tag == "members":
                        member_list = []
                        for member in members:
                            if member.text in matching:
                                member_list.append(member)
                        for member in member_list:
                            members.remove(member)
                        if len(members.findall("member")) == 0:
                            extname_list.append(extname)
            for extname in extname_list:
                hostextinfo.remove(extname)

            tree = ET.ElementTree(hostextinfo)
            tree.write(self.host_ext_info_xml_file)
       

    def create_host_ext_info_xml_file (self, ucs_ip) :
        host_ext_info_xml_fd = self.create_file(self.host_ext_info_xml_file)
        xml_text = ""
        # Check if the hostextinfo file exists
        try:
            xml_text = host_ext_info_xml_fd.read()
        except Exception, err:
            print ("Exception while creating Host Group file %s:" % (self.host_ext_info_xml_file)), str(err)
            host_ext_info_xml_fd.close() 
        if xml_text == "" :
            hostextinfo = Element('hostextinfo')
            type_list = []
        else:
            tree = ET.parse(self.host_ext_info_xml_file)
            hostextinfo = tree.getroot()
            type_list = []
            for node in hostextinfo.findall("hostextinfo_name") :
                type_list.append(node.text) 
        for parent, child in self.ucs_dict[ucs_ip, 'host_group'] :
            class_id = self.ucs_dict[child, 'class_id']
            #print ("Type  : :%s   ") %class_id
            if class_id in type_list :
                # Key already exists we just need to add the member
                for node in hostextinfo.findall("hostextinfo_name") :
                    if node.text == class_id:
                        matching_node = node 
                        for elem in matching_node.getiterator():
                            if elem.tag != 'members':
                                continue
                            add_node = True
                            for member_node in elem.getiterator():
                                if member_node.text == child:
                                    add_node = False
                            if add_node == True:
                                member = SubElement(elem, 'member')
                                member.text = child
            else:
                type_list.append(class_id)
                hostextinfo_name = SubElement(hostextinfo, 'hostextinfo_name')
                #hostgroup_name = SubElement(hostgroup, 'hostextinfo_name',)
                hostextinfo_name.text = class_id
                host_ext_dict = self.get_host_ext_data_for_type(class_id)
                icon_image = SubElement(hostextinfo_name, 'icon_image')
                icon_image.text = host_ext_dict['icon_image']
                icon_image_alt = SubElement(hostextinfo_name, 'icon_image_alt')
                icon_image_alt.text = host_ext_dict['icon_image_alt']
                statusmap_image =SubElement(hostextinfo_name, 'statusmap_image')
                statusmap_image.text = host_ext_dict['statusmap_image']
                members = SubElement(hostextinfo_name, 'members')
                member = SubElement(members, 'member')
                member.text = child
        tree = ET.ElementTree(hostextinfo)
        tree.write(self.host_ext_info_xml_file)


    
def get_ucs_details_from_csv(file_path):
    '''Method for fetching UCS host information from a csv file 
    and converting the same to python dict'''    
    ucs_info_dict = dict()
    try:
        fd = CheckForCommentOrBlank(open(file_path, 'rb'))
        reader = csv.reader(fd)  # creates the reader object
        rownum = 0
        for row in reader:  # iterates the rows of the file in orders
            #print row  # prints each row
            if rownum == 0:
                header = row
            else:
                colnum = 0
                key = None
                val = []
                for col in row:
                    if colnum == 0 :
                        key = col
                    else :
                        val.append(col)
                    # print '%-8s: %s' % (header[colnum], col)
                    colnum += 1
                    # All the IP addresses are the key and rest of the 
                    # columns are its value in the dict
                ucs_info_dict.update({key:val}) 

            rownum += 1

    except IOError:
        print 'Unable to open the file ', file_path

    fd.close()
    return ucs_info_dict

