<?php

require_once( __DIR__ . "/edasproxy_client.php" );
require_once( __DIR__ . '/edasproxy_functions.php' );

/**
 * Test classes for Mediasite External Data Access Service (Edas)
 * These proxy classes were generated based on the Mediasite 6.0 EDAS WSDL definition.
 * PHP Version 5.3
 * tests are loosely grouped by functional areas so that certain areas (Presentations, Roles, &ct) can be tested together
 *
 * @group stateless indicates that the test is complete and does not rely on database state
 *             Generated by PHPUnit_SkeletonGenerator on 2012-10-03 at 13:08:25.
 * @group state-sensitive indicates that the test is complete but not self-contained and that test expectations must be updated
 *             to reflect database state
 * @group active indicates that there must be active (being watched) presentations for meaningful results.
 *             additionally, some expectations may need to be modified to reflect activity
 * @copyright  Copyright (c) 2013, Sonic Foundry
 * @license    http://opensource.org/licenses/gpl-license.php GNU Public License
 * @version    6.1.7
 * @package    SonicFoundry.Mediasite.Edas.PHPProxy
 * @subpackage Tests
 * @author     Cori Schlegel <coris@sonicfoundry.com>
 *             This software is provided "AS IS" without a warranty of any kind.
 * @since      6.1.1
 */
class ExternalAccessClientTest extends PHPUnit_Framework_TestCase
{
    /*  static variables    */
    /**
     * @var ExternalAccessClient $client
     */
    protected static $client;
    protected static $token;
    protected static $Username = "MediasiteAdmin"; //  this should be a member of MediasiteAdministrators
    protected static $Password = "New_Password";
    protected static $EdasEndPoint = "http://kevinb-3500.sonicfoundry.net/Mediasite/6_1_7/Services60/EdasSixOneThirteen.svc";

    /*  test values */
    protected $UpdateRoleId;
    protected $MediasiteAdminsRoleId;
    protected $UpdateRoleDirectoryEntry;
    protected $UpdateRoleName;
    protected $TestingPresentationId;
    protected $TestingPresentationGuid;
    protected $OtherTestingPresentationGuid;
    protected $TestingPresentationName;
    protected $TotalViewsExpectedViewCount;
    protected $AdminUserName;
    protected $OtherUserName;
    protected $LocalhostIp;
    protected $LocalhostHostName;
    protected $TestFolderId;
    protected $TestMediasiteUsersFolderId;
    protected $SourceTemplateId;
    protected $CurrentTimestamp;
    protected $ServicesVersion;
    protected $SiteVersion;
    protected $TestEncodingSettingId;
    protected $TestEncodingSettingName;
    protected $TestScheduleId;
    protected $TestPlayerId;
    protected $TestPlayerName;
    protected $TestingPresentationContentItemId;
    protected $EmailSearchString;
    protected $TestUserProfileId;
    protected $TestProjectId;

    function __construct() {
    }

    /**
     * Sets up the fixture, for example, opens a network connection.
     * This method is called before a test is executed.
     */
    protected function setUp() {

        $this->UpdateRoleId                     = "8aafd05d-0ca4-464d-9f54-5d2c1344109c";
        $this->MediasiteAdminsRoleId            = "476ca830-9ec4-4587-80b6-f8e32264901c"; //  this is a well-known id
        $this->UpdateRoleName                   = "EDAS PHP Client Test";
        $this->UpdateRoleDirectoryEntry         = ""; //  the DirectoryEntry of the UpdateRole
        $this->TestingPresentationId            = "9080750763334953bcf655250104215a1d";
        $this->TestingPresentationGuid          = "90807507-6333-4953-bcf6-55250104215a";
        $this->OtherTestingPresentationGuid     = "ace27d8b-bc46-4a18-bb75-7250ba6573a6";
        $this->TestingPresentationName          = "for testing";
        $this->TotalViewsExpectedViewCount      = 6;
        $this->AdminUserName                    = "MediasiteAdmin"; //  a member of the MediasiteAdministrators role
        $this->OtherUserName                    = "instructor"; //  *not* a member of the MediasiteAdministrators role
        $this->LocalhostIp                      = "10.0.70.65";
        $this->LocalhostHostName                = "moodle-dev.sonicfoundry.net";
        $this->TestFolderId                     = "e82b4302-a5c3-4880-b2c5-9ef32fa1ac11";
        $this->SourceTemplateId                 = "dd5b6933-d803-47f0-b547-340057354629";
        $this->ServicesVersion                  = 'http://www.SonicFoundry.com/Mediasite/Services60';
        $this->SiteVersion                      = '6.1.7.2170';
        $this->TestEncodingSettingId            = '1778977c-09ad-4592-a257-89f0c034f214';
        $this->TestEncodingSettingName          = 'Audio Only (20 Kbps Audio)';
        $this->TestScheduleId                   = "33f47f93ee4043a9b32f0e38beb35a3d18";
        $this->TestPlayerId                     = '46cf0690-818f-4bf9-8165-ce7c530533f9';
        $this->TestPlayerName                   = 'Mediasite - Full Experience';
        $date                                   = new DateTime( 'now', new DateTimeZone( 'America/Chicago' ) );
        $this->CurrentTimestamp                 = $date->getTimestamp();
        $this->TestingPresentationContentItemId = '4ba45e2e-502b-482f-84ca-57806fd2b3be';
        $this->EmailSearchString                = 'thedoctor@tard.is';
        $this->TestUserProfileId                = 'b695d710-73c0-4d37-9c3a-cea5aac69d0c';
        $this->TestProjectId                    = '3dd34825-7152-4890-bfed-accfbb8e08c1';

        date_default_timezone_set('America/Chicago');

    }

    /**
     * Tears down the fixture, for example, closes a network connection.
     * This method is called after a test is executed.
     */
    protected function tearDown() {
    }

    public static function setUpBeforeClass() {

        //  useful to send request traffic to fiddler or another debugger
        $proxyOptions = array( "proxy_host" => "127.0.0.1", "proxy_port" => 8888 );
        error_reporting  (E_ALL);
        ini_set ('display_errors', true);

        // in some environments ssl traffic in wsdl mode ends up hitting the wrong url, so you might need to specify the service location
        $sslOptions = array( "location" => self::$EdasEndPoint, "uri" => "http://www.tempuri.org/" );

        self::$client = new ExternalAccessClient(
            self::$EdasEndPoint
            , null
            , null
            , array_merge($sslOptions, array( "cache_wsdl" => WSDL_CACHE_BOTH ))
            //, array_merge($proxyOptions, $sslOptions, array( "cache_wsdl" => WSDL_CACHE_BOTH ))
        );

        self::$token = self::$client->Login(self::$Username, self::$Password, 'PHP Edas Proxy');

    }

    //region Roles
    /**
     * @covers ExternalAccessClient::CreateRole
     * @covers ExternalAccessClient::DeleteRole
     * @group Roles
     * @group stateless
     */
    public function testCreateAndDeleteRole() {

        $roleDetails = new CreateRoleDetails( "newRole", "New Role", "Role created by testCreateRole" );
        $createdRole = self::$client->CreateRole($roleDetails);
        $roleId      = $createdRole->RoleId;
        $this->assertStringMatchesFormat('%s-%s-%s-%s', $roleId, "Return value is not a role id");

        $role = self::$client->QueryRolesById(Array( $roleId ));
        $this->assertEquals("newRole", $role->RoleDetails[0]->DirectoryEntry, "Queried role name does not match");

        $delResult = self::$client->DeleteRole($role->RoleDetails[0]->Id);
        $this->assertNotNull($delResult, "Deleted result is null");

    }

    /**
     * @covers ExternalAccessClient::UpdateRole
     * @covers ExternalAccessClient::QueryRolesById
     * @covers ExternalAccessClient::QueryRolesByCriteria
     * @group Roles
     * @group state-sensitive
     */
    public function testUpdateRoleUpdatesRole() {
        $newName  = "A new name";
        $newDescr = "A new description";
        $newDE    = "DirectoryEntry";

        $roles = self::$client->QueryRolesById(Array( $this->UpdateRoleId ));
        $role  = $roles->RoleDetails[0];
        $this->assertEquals($this->UpdateRoleName, $role->Name, "Role name is not correct");

        $newRole = new UpdateRoleDetails( $role->Id, $newDescr, true, $newDE, true, $newName, true );
        self::$client->UpdateRole($newRole);

        $roleCriteria = new RoleQueryCriteria( $newDE );
        $updatedRoles = self::$client->QueryRolesByCriteria($roleCriteria);
        $updatedRole  = $updatedRoles->RoleDetails[0];
        $this->assertEquals($updatedRole->Name, $newName, "Actual and expected new role names do not match");
        $this->assertEquals($updatedRole->Description, $newDescr, "Actual and expected new role decriptions do not match");
        $this->assertEquals($updatedRole->DirectoryEntry, $newDE, "Actual and expected new role Directory Entries do not match");

        //  restore
        $newRole->Name           = $this->UpdateRoleName;
        $newRole->Description    = $role->Description;
        $newRole->DirectoryEntry = $role->DirectoryEntry;
        self::$client->UpdateRole($newRole);

    }

    /**
     * @covers ExternalAccessClient::UpdateRole
     * @group Roles
     * @group state-sensitive
     * @todo    If the isset flags for a given attribute are not true, that attribute shouldn't be updated.
     */
    public function testUpdateRoleWithUpdateFlagsNotSetDoesntUpdateRole() {

        $newName  = "A new name";
        $newDescr = "A new description";
        $newDE    = "DirectoryEntry";

        $roleResponse = self::$client->QueryRolesById(Array( $this->UpdateRoleId ));
        $this->assertEquals($this->UpdateRoleName, $roleResponse->RoleDetails[0]->Name, "Role name is not correct");

        $newRole = new UpdateRoleDetails( $roleResponse->RoleDetails[0]->Id, $newDescr, false, $newDE, false, $newName, false );
        self::$client->UpdateRole($newRole);

        $updatedRoleResponse = self::$client->QueryRolesById(Array( $this->UpdateRoleId ));
        $this->assertNotEquals($updatedRoleResponse->RoleDetails[0]->Name, $newName, "Actual and expected new role names do not match");
        $this->assertNotEquals($updatedRoleResponse->RoleDetails[0]->Description, $newDescr, "Actual and expected new role decriptionsdo not match");
        $this->assertNotEquals($updatedRoleResponse->RoleDetails[0]->DirectoryEntry, $newDE, "Actual and expected new role Directory Entries do not match");

    }

    //endregion

    //region QueryTotalViews

    /**
     * @covers ExternalAccessClient::QueryTotalViews
     * @group QueryTotalViews
     * @group state-sensitive   presentation must have had some views
     */
    public function testQueryTotalViewsForOnePresentationReturnsOneQueryResult() {
        $response = self::$client->QueryTotalViews(Array( $this->TestingPresentationId ), AnalyticsRequestType::Presentation);
        $this->assertNotNull($response->Results, "Response result are null");
        $this->assertEquals(1, $response->Results->TotalResults);

    }

    /**
     * @covers ExternalAccessClient::QueryTotalViews
     * @group QueryTotalViews
     * @group state-sensitive
     */
    public function testQueryTotalViewsForOnePresentationReturnsAnArrayOfLengthOne() {
        $response = self::$client->QueryTotalViews(Array( $this->TestingPresentationId ), AnalyticsRequestType::Presentation);
        $this->assertEquals(1, count($response->ViewsList), "IdNameTotalPair Array is not one long");
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViews
     * @group QueryTotalViews
     * @group state-sensitive
     */
    public function testQueryTotalViewsForTwoPresentationReturnsTwoIdNameTotalPairs() {
        $response = self::$client->QueryTotalViews(Array( $this->TestingPresentationId, $this->TestingPresentationId ), AnalyticsRequestType::Presentation);
        $this->assertTrue(is_array($response->ViewsList), "IdNameTotalPair is not an array");
        $this->assertEquals(2, count($response->ViewsList), "IdNameTotalPair doesn't have 2 items");
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViews
     * @group QueryTotalViews
     * @group state-sensitive
     */
    public function testQueryTotalViewsForOnePresentationReturnsAccurateViewsList() {
        $response = self::$client->QueryTotalViews(Array( $this->TestingPresentationId, $this->TestingPresentationId ), AnalyticsRequestType::Presentation);
        $this->assertEquals($this->TotalViewsExpectedViewCount, $response->ViewsList[0]->Total, "Count of views for requested presentation is incorrect");
        $this->assertEquals($this->TestingPresentationName, $response->ViewsList[0]->Name, "Queried presentation name does not match");
        $this->assertEquals($this->TestingPresentationId, $response->ViewsList[0]->Id, "Queried presentation id does not match");

    }

    /**
     * @covers ExternalAccessClient::QueryTotalViews
     * @group QueryTotalViews
     * @group state-sensitive
     */
    public function testQueryTotalViewsWithTwoUserNamesReturnsCorrectNumberOfViews() {
        $response = self::$client->QueryTotalViews(Array( $this->AdminUserName, $this->OtherUserName ), AnalyticsRequestType::User);
        $this->assertEquals(2, count($response->ViewsList), "Viewsd list is not correct length");
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViews
     * @group QueryTotalViews
     * @group state-sensitive
     */
    public function testQueryTotalViewsByUserNameReturnsAccurateViewsList() {
        $response                  = self::$client->QueryTotalViews(Array( $this->OtherUserName, $this->AdminUserName ), AnalyticsRequestType::User);
        $oun                       = $this->OtherUserName;
        $aun                       = $this->AdminUserName;
        $otherUserNameArrayMembers = array_filter($response->ViewsList, function ( $user ) use ( $oun ) {
            return $user->Name === $oun;
        });
        $adminUserNameArrayMembers = array_filter($response->ViewsList, function ( $user ) use ( $aun ) {
            return $user->Name === $aun;
        });
        $this->assertEquals(1, count($otherUserNameArrayMembers), "The number of users with the specified username is incorrect");
        $this->assertEquals(1, count($adminUserNameArrayMembers), "The number of users with the admin username is incorrect");
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViews
     * @group QueryTotalViews
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIpAddressReturnsAccurateViewsList() {
        $response = self::$client->QueryTotalViews(Array( $this->LocalhostIp ), AnalyticsRequestType::IPAddress);
        $this->assertEquals($this->TotalViewsExpectedViewCount, $response->ViewsList[0]->Total, "Count of views for requested user is incorrect");
        $this->assertEquals($this->LocalhostHostName, $response->ViewsList[0]->Name, "Queried user name does not match");
        $this->assertEquals($this->LocalhostIp, $response->ViewsList[0]->Id, "Queried user id does not match");

    }

    /**
     * @covers ExternalAccessClient::QueryTotalViews
     * @group QueryTotalViews
     * @expectedException SoapFault
     * @group state-sensitive
     */
    public function testQueryTotalViewsByServerIsInvalid() {
        $response = self::$client->QueryTotalViews(Array( $this->LocalhostHostName ), AnalyticsRequestType::Server);
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForPresentationAndUserReturnsMatchingTotalResultsAndViewsList() {
        $response = self::$client->QueryTotalViewsById($this->TestingPresentationId, AnalyticsRequestType::Presentation, AnalyticsRequestType::User);
        $this->assertNotNull($response->Results, "Response results are null");
        $this->assertEquals($response->Results->TotalResults, count($response->ViewsList), "Results and count of views list do not match");
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForPresentationAndIPAddressReturnsMatchingTotalResultsAndViewsList() {
        $response = self::$client->QueryTotalViewsById($this->TestingPresentationId, AnalyticsRequestType::Presentation, AnalyticsRequestType::IPAddress);
        $this->assertNotNull($response->Results, "Response results are null");
        $this->assertEquals($response->Results->TotalResults, count($response->ViewsList), "Results and count of views list do not match");
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @expectedException SoapFault
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForPresentationAndServerIsInvalid() {
        $response = self::$client->QueryTotalViewsById($this->TestingPresentationId, AnalyticsRequestType::Presentation, AnalyticsRequestType::Server);
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForUserAndPresentationReturnsMatchingTotalResultsAndViewsList() {
        $response = self::$client->QueryTotalViewsById($this->AdminUserName, AnalyticsRequestType::User, AnalyticsRequestType::Presentation);
        $this->assertNotNull($response->Results, "Response result are null");
        $this->assertEquals($response->Results->TotalResults, count($response->ViewsList), "Results and count of views list do not match");

    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForUserAndIPAddresReturnsMatchingTotalResultsAndViewsList() {
        $response = self::$client->QueryTotalViewsById($this->AdminUserName, AnalyticsRequestType::User, AnalyticsRequestType::IPAddress);
        $this->assertNotNull($response->Results, "Response result are null");
        $this->assertEquals($response->Results->TotalResults, count($response->ViewsList), "Results and count of views list do not match");

    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @expectedException SoapFault
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForUserAndServerIsInvalid() {
        $response = self::$client->QueryTotalViewsById($this->AdminUserName, AnalyticsRequestType::User, AnalyticsRequestType::Server);
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForIPAddressAndUserReturnsMatchingTotalViewAndViewsList() {
        $response = self::$client->QueryTotalViewsById($this->LocalhostIp, AnalyticsRequestType::IPAddress, AnalyticsRequestType::User);
        $this->assertNotNull($response->Results, "Response result are null");
        $this->assertEquals($response->Results->TotalResults, count($response->ViewsList), "Results and count of views list do not match");
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForIPAddressAndPresentationReturnsMatchingTotalViewAndViewsList() {
        $response = self::$client->QueryTotalViewsById($this->LocalhostIp, AnalyticsRequestType::IPAddress, AnalyticsRequestType::Presentation);
        $this->assertNotNull($response->Results, "Response result are null");
        $this->assertEquals($response->Results->TotalResults, count($response->ViewsList), "Results and count of views list do not match");
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @expectedException SoapFault
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForIPAddressAndServerIsInvalid() {
        $response = self::$client->QueryTotalViewsById($this->LocalhostIp, AnalyticsRequestType::IPAddress, AnalyticsRequestType::Server);
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @expectedException SoapFault
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForServerAndPresentationIsInvalid() {
        $response = self::$client->QueryTotalViewsById($this->LocalhostIp, AnalyticsRequestType::Server, AnalyticsRequestType::Presentation);
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @expectedException SoapFault
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForServerAndUserReturnsMatchingTotalViewAndViewsList() {
        $response = self::$client->QueryTotalViewsById($this->LocalhostIp, AnalyticsRequestType::Server, AnalyticsRequestType::User);
    }

    /**
     * @covers ExternalAccessClient::QueryTotalViewsById
     * @group QueryTotalViewsById
     * @expectedException SoapFault
     * @group state-sensitive
     */
    public function testQueryTotalViewsByIdForServerAndIPAddressReturnsMatchingTotalViewAndViewsList() {
        $response = self::$client->QueryTotalViewsById($this->LocalhostIp, AnalyticsRequestType::Server, AnalyticsRequestType::IPAddress);
    }

    /**   TODO:  handle subsequent query and options-based querying **/
    //endregion

    //region QueryDatesWatched
    /**
     * @covers ExternalAccessClient::QueryDatesWatched
     * @group QueryDatesWatched
     * @grop state-sensitive
     */
    public function testQueryDatesWatchedForPresentationContainsQueryResults() {
        $response = self::$client->QueryDatesWatched($this->TestingPresentationId, AnalyticsRequestType::Presentation);
        $this->assertNotNull($response->Results, "QueryResults is null");
    }

    /**
     * @covers ExternalAccessClient::QueryDatesWatched
     * @group QueryDatesWatched
     */
    public function testQueryDatesWatchedForPresentationContainsDates() {
        $response = self::$client->QueryDatesWatched($this->TestingPresentationId, AnalyticsRequestType::Presentation);
        $this->assertNotNull($response->DatesWatchedList, "Dates watched is null");
        $this->assertTrue(is_array($response->DatesWatchedList), "Dates watched is not an array");
    }

    /**
     * @covers ExternalAccessClient::QueryDatesWatched
     * @group QueryDatesWatched
     * @group state-sensitive
     */
    public function testQueryDatesWatchedForUserContainsQueryResults() {
        $response = self::$client->QueryDatesWatched($this->AdminUserName, AnalyticsRequestType::User);
        $this->assertNotNull($response->Results, "QueryResults is null");
    }

    /**
     * @covers ExternalAccessClient::QueryDatesWatched
     * @group QueryDatesWatched
     * @group state-sensitive
     */
    public function testQueryDatesWatchedForIAddressContainsDates() {
        $response = self::$client->QueryDatesWatched($this->LocalhostIp, AnalyticsRequestType::IPAddress);
        $this->assertNotNull($response->DatesWatchedList, "Dates watched is null");
        $this->assertTrue(is_array($response->DatesWatchedList), "Dates watched is not an array");
    }

    /**
     * @covers ExternalAccessClient::QueryDatesWatched
     * @group QueryDatesWatched
     * @group state-sensitive
     */
    public function testQueryDatesWatchedForIPAddressContainsQueryResults() {
        $response = self::$client->QueryDatesWatched($this->LocalhostIp, AnalyticsRequestType::IPAddress);
        $this->assertNotNull($response->Results, "QueryResults is null");
    }

    /**
     * @covers ExternalAccessClient::QueryDatesWatched
     * @group QueryDatesWatched
     * @expectedException SoapFault
     * @group state-sensitive
     */
    public function testQueryDatesWatchedForServerIsInvalid() {
        $result = self::$client->QueryDatesWatched($this->LocalhostHostName, AnalyticsRequestType::Server);
    }

    /**   TODO:  handle subsequent query and options-based querying **/
    //endregion

    //region QueryPlatformUsage
    /**
     * @covers ExternalAccessClient::QueryPlatformUsage
     * @group QueryPlatformUsage
     * @group state-sensitive
     */
    public function testQueryPlatformUsageForPresentationContainsCorrectlyFormattedUsage() {
        $response = self::$client->QueryPlatformUsage($this->TestingPresentationId, AnalyticsRequestType::Presentation);
        $this->assertNotNull($response->Usage, "Usage is null");
        $this->assertNotNull($response->Usage->BrowserList, "UsageBrowserList is null");
        $this->assertNotNull($response->Usage->MediaPluginList, "MediaPluginList is null");
        $this->assertNotNull($response->Usage->PlayerEngineList, "PlayerEngineList is null");
        $this->assertNotNull($response->Usage->SystemList, "SystemList is null");
    }

    /**
     * @covers ExternalAccessClient::QueryPlatformUsage
     * @group QueryPlatformUsage
     * @group state-sensitive
     */
    public function testQueryPlatformUsageForIPAddressContainsCorrectlyFormattedUsage() {
        $response = self::$client->QueryPlatformUsage($this->LocalhostIp, AnalyticsRequestType::IPAddress);
        $this->assertNotNull($response->Usage, "Usage is null");
        $this->assertNotNull($response->Usage->BrowserList, "UsageBrowserList is null");
        $this->assertNotNull($response->Usage->MediaPluginList, "MediaPluginList is null");
        $this->assertNotNull($response->Usage->PlayerEngineList, "PlayerEngineList is null");
        $this->assertNotNull($response->Usage->SystemList, "SystemList is null");
    }

    /**
     * @covers ExternalAccessClient::QueryPlatformUsage
     * @group QueryPlatformUsage
     * @group state-sensitive
     */
    public function testQueryPlatformUsageForUserContainsCorrectlyFormattedUsage() {
        $response = self::$client->QueryPlatformUsage($this->AdminUserName, AnalyticsRequestType::User);
        $this->assertNotNull($response->Usage, "Usage is null");
        $this->assertNotNull($response->Usage->BrowserList, "UsageBrowserList is null");
        $this->assertNotNull($response->Usage->MediaPluginList, "MediaPluginList is null");
        $this->assertNotNull($response->Usage->PlayerEngineList, "PlayerEngineList is null");
        $this->assertNotNull($response->Usage->SystemList, "SystemList is null");
    }

    /**
     * @covers ExternalAccessClient::QueryPlatformUsage
     * @group QueryPlatformUsage
     * @group state-sensitive
     */
    public function testQueryPlatformUsageForServerContainsCorrectlyFormattedUsage() {
        $response = self::$client->QueryPlatformUsage($this->LocalhostHostName, AnalyticsRequestType::Server);
        $this->assertNotNull($response->Usage, "Usage is null");
        $this->assertNotNull($response->Usage->BrowserList, "UsageBrowserList is null");
        $this->assertNotNull($response->Usage->MediaPluginList, "MediaPluginList is null");
        $this->assertNotNull($response->Usage->PlayerEngineList, "PlayerEngineList is null");
        $this->assertNotNull($response->Usage->SystemList, "SystemList is null");
    }

    /**   TODO:  handle subsequent query for remaining results and options-based querying **/
    //endregion

    //region QueryPresentationUsage
    /**
     * @covers ExternalAccessClient::QueryPresentationUsage
     * @group state-sensitive
     */
    public function testQueryPresentationUsageForUsersReturnsCorrectQuantityOfResults() {
        $userList = Array( $this->AdminUserName, $this->OtherUserName );
        $response = self::$client->QueryPresentationUsage($userList, AnalyticsRequestType::User, $this->TestingPresentationId);
        $this->assertEquals(count($userList), $response->Results->TotalResults);
        $this->assertEquals(count($userList), count($response->UsageList));
    }

    /**
     * @covers ExternalAccessClient::QueryPresentationUsage
     * @group state-sensitive
     * 2012-0-11-16 this throws if the presentation has had no views.
     */
    public function testQueryPresentationUsageForIPAddressReturnsCorrectQuantityOfResults() {
        $ipAddressList = Array( $this->LocalhostIp );
        $response      = self::$client->QueryPresentationUsage($ipAddressList, AnalyticsRequestType::IPAddress, $this->TestingPresentationId);
        $this->assertEquals(count($ipAddressList), $response->Results->TotalResults);
        $this->assertEquals(count($ipAddressList), count($response->UsageList));
    }

    /**
     * @covers ExternalAccessClient::QueryPresentationUsage
     * @expectedException SoapFault
     * @group stateless
     */
    public function testQueryPresentationUsageForServerIsInvalid() {
        $serverList = Array( $this->LocalhostHostName );
        $response   = self::$client->QueryPresentationUsage($serverList, AnalyticsRequestType::Server, $this->TestingPresentationId);
    }

    /**
     * @covers ExternalAccessClient::QueryPresentationUsage
     * @expectedException SoapFault
     * @group stateless
     */
    public function testQueryPresentationUsageForPresentationIsInvalid() {
        $presentationIdList = Array( $this->TestingPresentationId );
        $response           = self::$client->QueryPresentationUsage($presentationIdList, AnalyticsRequestType::Presentation, $this->TestingPresentationId);
    }

    /**   TODO:  handle subsequent query and options-based querying **/
    //endregion

    /**
     * @covers ExternalAccessClient::QueryServerUsage
     * @group state-sensitive
     */
    public function testQueryServerUsageReturnsProperlyFormattedResponse() {
        $response = self::$client->QueryServerUsage();
        $this->assertNotNull($response->Usage, "Usage is null");
        $this->assertNotNull($response->Usage->LivePresentationsWatched, "LivePresentationsWatched is null");
        $this->assertNotNull($response->Usage->LiveViews, "LiveViews is null");
        $this->assertNotNull($response->Usage->OnDemandPresentationsWatched, "LiveViews is null");
        $this->assertNotNull($response->Usage->OnDemandViews, "OnDemandViews is null");
        $this->assertNotNull($response->Usage->TotalPresentationsWatched, "TotalPresentationsWatched is null");
        $this->assertNotNull($response->Usage->TotalViews, "TotalViews is null");
    }

    //region QueryActiveConnections
    /**
     * @covers ExternalAccessClient::QueryActiveConnections
     * @group active
     */
    public function testQueryActiveConnectionsReturnsProperlyFormattedResponse() {
        $response = self::$client->QueryActiveConnections();
        $this->assertNotNull($response->Connections, "Usage is null");
        $this->assertNotNull($response->Connections->LiveConnections, "LiveConnections is null");
        $this->assertNotNull($response->Connections->LivePresentations, "LivePresentations is null");
        $this->assertNotNull($response->Connections->OnDemandConnections, "OnDemandConnections is null");
        $this->assertNotNull($response->Connections->OnDemandPresentations, "OnDemandPresentations is null");
        $this->assertNotNull($response->Connections->TotalConnections, "TotalConnections is null");
        $this->assertNotNull($response->Connections->TotalPresentations, "TotalPresentations is null");
    }

    /**
     * @covers ExternalAccessClient::QueryActiveConnections
     * @group active only run when user is connected and watching a presentation
     */
    public function testQueryActiveConnectionsReturnsFilledResultsWhenUserWatching() {
        $response = self::$client->QueryActiveConnections();
        $this->assertNotNull($response->Connections, "Usage is null");
        $this->assertEquals(0, $response->Connections->LiveConnections, "LiveConnections is not 0");
        $this->assertEquals(0, $response->Connections->LivePresentations, "LivePresentations is not 0");
        $this->assertEquals(1, $response->Connections->OnDemandConnections, "OnDemandConnections is not 1");
        $this->assertEquals(1, $response->Connections->OnDemandPresentations, "OnDemandPresentations is not1");
        $this->assertEquals(1, $response->Connections->TotalConnections, "TotalConnections is not 1");
        $this->assertEquals(1, $response->Connections->TotalPresentations, "TotalPresentations is not 1");
    }

    //endregion

    //region QueryActivePresentations
    /**
     * @covers ExternalAccessClient::QueryActivePresentations
     * @group active only run when user is connected and watching a presentation
     */
    public function testQueryActivePresentationsReturnsCorrectResultsWhenPresentationIsActive() {
        $presentationIdArray = Array( $this->TestingPresentationId );
        $options             = new QueryOptions( 10, 'testQueryActivePresentations', 0 );
        $response            = self::$client->QueryActivePresentations($presentationIdArray, $options);
        $this->assertEquals(count($presentationIdArray), count($response->ActivePresentationList), "Count of Ids and active presentations does not match");
        $this->assertEquals($presentationIdArray[0], $response->ActivePresentationList[0]->Id, "Submitrted presentation id and active presentation id do not match");
        $this->assertEquals(count($response->ActivePresentationList), $response->Results->TotalResults, "Total results and Number of connections do not match");
    }

    /**
     * @covers ExternalAccessClient::QueryActivePresentationConnections
     * @group active
     */
    public function testQueryActivePresentationConnectionsReturnsCorrectQuantityOfConnections() {
        $response = self::$client->QueryActivePresentationConnections($this->TestingPresentationId);
        $this->assertEquals(1, count($response->ConnectionsList), "Number of active connections is not 1");
        $this->assertEquals(count($response->ConnectionsList), $response->Results->TotalResults, "Total results and Number of connections do not match");
    }

    //endregion

    //region AuthTicket
    /**
     * @covers ExternalAccessClient::CreateAuthTicket
     * @covers ExternalAccessClient::QueryAuthTicketProperties
     * @group AuthTicket
     * @group stateless
     */
    public function testCreateAuthTicketReturnsValidTicket() {
        $response         = self::$client->CreateAuthTicket($this->LocalhostIp, 60, $this->TestingPresentationGuid, $this->OtherUserName);
        $AuthTicketId     = $response->AuthTicketId;
        $validateResponse = self::$client->QueryAuthTicketProperties($AuthTicketId, 100, false);
        $this->assertEquals($this->LocalhostIp, $validateResponse->Properties->ClientIpAddress, "IP Addresses do not match");
        $this->assertEquals($this->OtherUserName, $validateResponse->Properties->Username, "Usernames do not match");

        /*  clean up    */
        self::$client->RemoveAuthTicket($AuthTicketId);

    }

    /**
     * @covers ExternalAccessClient::RemoveAuthTicket
     * @expectedException SoapFault
     * @group AuthTicket
     * @group stateless
     */
    public function testRemoveAuthTicketRemovesTicketAndQueryAuthTicketPropertiesThrowsWhenNoAuthTicket() {
        $response     = self::$client->CreateAuthTicket($this->LocalhostIp, 60, $this->TestingPresentationId, $this->OtherUserName);
        $AuthTicketId = $response->AuthTicketId;

        /*  clean up    */
        self::$client->RemoveAuthTicket($AuthTicketId);
        $validateRemovalResponse = self::$client->QueryAuthTicketProperties($AuthTicketId, 100, false);

    }

    //endregion

    //region CreateDeleteFolder
    /**
     * @covers ExternalAccessClient::CreateSubFolder
     * @covers ExternalAccessClient::QueryFoldersById
     * @group state-dependant
     * delete must be manually verified because DeleteFolder runs as an async job
     */
    public function testMinimalCreateSubFolderCreatesFolderAndDeletes() {
        $folderName = "Folder name " . $this->CurrentTimestamp;
        $perms      = Array( new ResourcePermissionEntry( Array( ResourcePermissionMask::Read, ResourcePermissionMask::Write ), $this->MediasiteAdminsRoleId ) );

        $response = self::$client->CreateSubFolder($folderName, $perms);

        $validResponse = self::$client->QueryFoldersById(Array( $response->Id ), ResourcePermissionMask::Read);
        $this->assertEquals($folderName, $validResponse->FolderDetails[0]->Name, "Queried folder name is not the same as the folder name used to create");

        $deleteResponse = self::$client->DeleteFolder($response->Id);

        $anotherResponse = self::$client->QueryFoldersById(array( $response->Id ), ResourcePermissionMask::Read);

    }

    /**
     * @covers ExternalAccessClient::CreateSubFolder
     * @covers ExternalAccessClient::QueryFoldersById
     * @covers ExternalAccessClient::DeleteFolder
     * @covers ExternalAccessClient::QuerySubFolderDetails
     * @group stateless
     * delete must be manually verified because DeleteFolder runs as an async job
     * also since this is true if this test is run multiple times close together counts will be wrong
     * Updated to check creating Shared folder types in 6.1.7
     */
    public function testFullCreateSubFolderCreatesFolderWithMatchingCharacteristicsAndDeletes() {
        $folderName  = "Folder name " . $this->CurrentTimestamp;
        $description = "A Description";
        $perms       = Array( new ResourcePermissionEntry( Array( ResourcePermissionMask::Read, ResourcePermissionMask::Write, ResourcePermissionMask::Moderate ), $this->MediasiteAdminsRoleId ) );

        $response = self::$client->CreateSubFolder($folderName, $perms, $description, CreateFolderTypeDetails::Shared, $this->TestFolderId);

        $validResponse = self::$client->QueryFoldersById(Array( $response->Id ), ResourcePermissionMask::Read);

        $this->assertEquals($folderName, $validResponse->FolderDetails[0]->Name, "Queried folder name is not the same as the folder name used to create");
        $this->assertEquals($description, $validResponse->FolderDetails[0]->Description, "Queried folder description is not the same as the folder description used to create");
        $this->assertEquals($this->TestFolderId, $validResponse->FolderDetails[0]->ParentId, "Queried folder parent folder is not the same as the parent folder used to create");
        $this->assertTrue(substr_count($validResponse->FolderDetails[0]->Type, "Shared") > 0, "Queried folder is not of type 'Shared'");

        $folderDetailsResponse = self::$client->QuerySubFolderDetails(false, array( $this->TestFolderId ), ResourcePermissionMask::Read);

        $lastIndex = 0;
        for ( $i = 0; $i < count($folderDetailsResponse->Folders); $i++ ) {
            if ( $folderDetailsResponse->Folders[$i]->Id == $response->Id ) {
                $lastIndex = $i;
            }
        }
        $this->assertEquals($folderName, $folderDetailsResponse->Folders[$lastIndex]->Name, "Queried folder name is not the same as the folder name used to create");
        $this->assertEquals($description, $folderDetailsResponse->Folders[$lastIndex]->Description, "Queried folder description is not the same as the folder description used to create");
        $this->assertEquals($this->TestFolderId, $folderDetailsResponse->Folders[$lastIndex]->ParentId, "Queried folder parent folder is not the same as the parent folder used to create");

        $deleteResponse = self::$client->DeleteFolder($response->Id);

    }

    //endregion

    //region IdentityTicket
    /**
     * @covers ExternalAccessClient::CreateIdentityTicket
     * @covers ExternalAccessClient::QueryIdentityTicketProperties
     * @covers ExternalAccessClient::RemoveIdentityTicket
     * @group stateless
     */
    public function testCreateAndQueryIdentityTicket() {
        $ticketSettings = new CreateIdentityTicketSettings( $this->LocalhostIp, 60, $this->OtherUserName );
        $response       = self::$client->CreateIdentityTicket($ticketSettings);

        $validResponse = self::$client->QueryIdentityTicketProperties($response->Ticket, 60, false);
        $this->assertEquals($this->OtherUserName, $validResponse->Properties->Username, "Usernames do not match");
        $this->assertEquals($this->LocalhostIp, $validResponse->Properties->ClientIpAddress, "IP Addresses do not match");

    }

    /**
     * @covers ExternalAccessClient::CreateIdentityTicket
     * @covers ExternalAccessClient::QueryIdentityTicketProperties
     * @covers ExternalAccessClient::RemoveIdentityTicket
     * @expectedException SoapFault
     * @group stateless
     */
    public function testRemoveIdentityTicketAndQueryThrows() {

        $ticketSettings = new CreateIdentityTicketSettings( $this->LocalhostIp, 60, $this->OtherUserName );
        $response       = self::$client->CreateIdentityTicket($ticketSettings);

        $validResponse = self::$client->QueryIdentityTicketProperties($response->Ticket, 60, false);

        self::$client->RemoveIdentityTicket($response->Ticket);

        $throwsAfterDelete = self::$client->QueryIdentityTicketProperties($response->Ticket, 60, false);

    }

    /**
     * @covers ExternalAccessClient::CreateIdentityTicket
     * @covers ExternalAccessClient::QueryIdentityTicketProperties
     * @covers ExternalAccessClient::RemoveIdentityTicket
     * @group stateless
     */
    public function testQueryIdentityTicketPropertiesAndUpdateResetsExpirationTime() {
        $ticketSettings = new CreateIdentityTicketSettings( $this->LocalhostIp, 60, $this->OtherUserName );
        $response       = self::$client->CreateIdentityTicket($ticketSettings);

        $validResponse      = self::$client->QueryIdentityTicketProperties($response->Ticket, 120, true);
        $creationDateTime   = new DateTime( $validResponse->Properties->CreationTime );
        $expirationDateTime = new DateTime( $validResponse->Properties->ExpirationTime );

        $dateDiff = $expirationDateTime->diff($creationDateTime, true);
        $this->assertEquals(2, $dateDiff->h, "Creation and Expiration times do not differ by expected amount");

    }

    //endregion

    //region Presentations
    /**
     * @covers ExternalAccessClient::CreatePresentationFromTemplate
     * @covers ExternalAccessClient::QueryPresentationById
     * @covers ExternalAccessClient::UpdatePresentationDetails
     * @covers ExternalAccessClient::DeletePresentation
     * @todo    should test a more fully configured presentation
     * @group state-sensitive
     *          SourceTemplateId must exist
     */
    public function testMinimallyCreatePresentationFromTemplateCreatesCorrespondingTemplate() {
        $presentationDetails = new CreatePresentationFromTemplateDetails( $this->SourceTemplateId, "Presentation from Default Template" );
        $response            = self::$client->CreatePresentationFromTemplate($presentationDetails);

        $createdPresentation = $response->Presentation;
        $validResponse       = self::$client->QueryPresentationsById(Array( $createdPresentation->Id ));

        //  validate a few nominal properties
        $this->assertEquals(1, count($validResponse->Presentations), "Did not return a single Presentation");
        $this->assertEquals($createdPresentation->Name, $validResponse->Presentations[0]->Name, "Presentation names do not match");
        $this->assertEquals($createdPresentation->PlayerId, $validResponse->Presentations[0]->PlayerId, "Player ids do not match");
        $this->assertEquals($createdPresentation->Status, $validResponse->Presentations[0]->Status, "Statuses do not match");

        $updateDetails = new PresentationUpdateDetails( false, false, false, true, false, false, false, false, false,
            false, false, false, false, false, false, null, Array( EntityChangeTypesDetails::DefaultApprovalChanges ), null, 'A description' );
        self::$client->UpdatePresentationDetails($createdPresentation->Id, $updateDetails);
        $updatedResponse = self::$client->QueryPresentationsById(Array( $createdPresentation->Id ));
        $this->assertEquals($createdPresentation->ChangeTypes, $updatedResponse->Presentations[0]->ChangeTypes,
            'Change type was not indicated to be set so it should remaining unchanged');
        $this->assertNotEquals($createdPresentation->Description, $updatedResponse->Presentations[0]->Description,
            'Description was indicated to be set so it should be changed');

        self::$client->DeletePresentation($createdPresentation->Id);

    }

    /**
     * @covers ExternalAccessClient::QueryPresentationById
     * @covers ExternalAccessClient::DeletePresentation
     * @group state-sensitive
     */
    public function testDeletePresentationDeletesAndQueryForMissingPresentationThrows() {
        $presentationDetails = new CreatePresentationFromTemplateDetails( $this->SourceTemplateId, "Presentation from Default Template" );
        $response            = self::$client->CreatePresentationFromTemplate($presentationDetails);

        self::$client->DeletePresentation($response->Presentation->Id);
        $validResponse = self::$client->QueryPresentationsById(Array( $response->Presentation->Id ));
        $this->assertTrue((bool)$validResponse->Presentations[0]->Recycled, "Presentation is not recycled");
    }

    /**
     * @covers ExternalAccessClient::CreatePresentationFromSchedule
     * @todo   Create a fully-detailed test for CreatePresentationFromSchedule
     * @group state-sensitive
     * @group Presentation
     */
    public function testCreatePresentationFromScheduleWithDefaults_MatchesDefaults() {
        $presentationTitle   = "Presentation From Schedule";
        $recurrence          = 2;
        $presentationDetails = new CreatePresentationFromScheduleDetails( $this->TestScheduleId, $presentationTitle, $recurrence );
        $response            = self::$client->CreatePresentationFromSchedule($presentationDetails);

        $createdPresentation = self::$client->QueryPresentationsById(Array( $response->PresentationId ))->Presentations[0];

        $this->assertEquals("Scheduled", $createdPresentation->Status, "Created presentation status does not match default");
        $this->assertEquals($response->PresentationId, $createdPresentation->Id, "Created presentation id does not match result id");
        $this->assertEquals($presentationTitle, $createdPresentation->Name, "Created presentation name does not match presentation title");

        //  not testing this method
        self::$client->DeletePresentation($response->PresentationId);
    }

    /**
     * @covers ExternalAccessClient::CreatePresentationLike
     * @covers ExternalAccessClient::QueryTimeZonesByCriteria
     * @todo   test fully-detailed presentation like
     * @group stateless
     */
    public function test_MinimalCreatePresentationLike_CreatesExpectedPresentation() {
        $tzCriteria = new TimeZoneQueryCriteria( range(1, 94) );
        $tzResponse = self::$client->QueryTimeZonesByCriteria($tzCriteria);

        //  very simplistic check
        $this->assertTrue($tzResponse->TimeZones[19]->Id == 20, "Id is not one greater than the index in the TimeZones array");
        $this->assertEquals(count($tzCriteria->TimeZoneIdList), count($tzResponse->TimeZones), "Query did not return same number of timezones as requested array");

        $title               = "Presentation From Like";
        $presentationDetails = new CreatePresentationLikeDetails( $this->TestingPresentationGuid, $title, 19 );
        $response            = self::$client->CreatePresentationLike($presentationDetails);

        //  verify created presentation return values
        $createdPresentation = $response->Presentation;
        $this->assertEquals(1, $createdPresentation->Content[0]->ContentRevision, "ContentRevision is not initial");
        $this->assertEquals($createdPresentation->PresentationRootId, $createdPresentation->Id, "Id and Root Id are not equal");
        $this->assertEquals($title, $createdPresentation->Name, "Presentation Name is no equal to title");
        $this->assertEquals(self::$Username, $createdPresentation->Owner);

        self::$client->DeletePresentation($createdPresentation->Id);

    }

    //endregion

    //region ToBeImplemented
    /**
     * @covers ExternalAccessClient::CreatePresentationPoll
     * @group stateless
     *  would have to be manually verified before deleting the presentation. Manually verified on 2012-11-29, but
     *  leaving automated test to do the automatic delete and this test as an error check
     */
    public function test_CreatePresentationPoll_DoesntError() {
        $title               = "Presentation From Like";
        $presentationDetails = new CreatePresentationLikeDetails( $this->TestingPresentationGuid, $title, 19 );
        $response            = self::$client->CreatePresentationLike($presentationDetails);

        $noAnswer            = new PollAnswerDetails( "No", 1, 1, 0 );
        $yesAnswer           = new PollAnswerDetails( "Yes", 2, 1, 1 );
        $pollQuestionDetails = new PollQuestionDetails( Array( $noAnswer, $yesAnswer ), "1", "Is this the first question?" );
        $pollDetail          = new CreatePresentationPollDetails( Array( $pollQuestionDetails ), $response->Presentation->Id );
        self::$client->CreatePresentationPoll($pollDetail);

        $presoSAfterPoll = self::$client->QueryPresentationsById(Array( $response->Presentation->Id ));

        self::$client->DeletePresentation($response->Presentation->Id);
    }

    public function testCreateScheduleFromTemplate() {
        $name            = "Schedule From Template";
        $recurrence      = new ScheduleRecurrenceDetails( 1, 30, "12:00:00.000", "13:00:00.000", Array(), true,
            MonthOfTheYear::November, "12:00:00.000", 60, 1, RecurrencePattern::Yearly,
            RecurrencePatternType::Simple, true, WeekOfTheMonth::None, WeekDay::Friday );
        $recurrence2     = new ScheduleRecurrenceDetails( 1, 30, "12:00:00.000", "13:00:00.000", Array(), true,
            MonthOfTheYear::December, "12:00:00.000", 60, 1, RecurrencePattern::Daily,
            RecurrencePatternType::Simple, false, WeekOfTheMonth::None, WeekDay::Friday );
        $scheduleDetails = new CreateScheduleFromTemplateDetails( $this->SourceTemplateId, $name, Array( $recurrence,
                $recurrence2 ),
            ScheduleTitleType::ScheduleNameAndNumber, 15, 60, 2, false, false, false, false, false,
            "Description", "PublishingPoint", "RecipientsEmailAddresses",  "RecorderId", "SendersEmail", "TimeZoneId", "TimeZoneRegistryKey",
            TRUE, FALSE, TRUE);
        $response        = self::$client->CreateScheduleFromTemplate($scheduleDetails);
    }

    /**
     * @covers ExternalAccessClient::CreateScheduleFromTemplate
     * @covers ExternalAccessClient::QuerySchedulesByCriteria
     * @covers ExternalAccessClient::UpdateSchedule
     * @covers ExternalAccessClient::DeleteSchedule
     * @todo   test variations in QueryScheduleByCriteria
     * @todo   test variations in ScheduleRecurrence
     * @todo   test variations in CreateScheduleFromTemplateDetails
     * @todo   figure out why typemap isn't working properly
     * @todo   must test DaysOfWeek to ensure that you can pass an array of them and have it work
     * @group stateless
     */
    public function test_MinimalCreateUpdateDeleteScheduleFromTemplate() {

        $name            = "Schedule From Template";
        $recurrence      = new ScheduleRecurrenceDetails( 1, 30, "12:00:00.000", "13:00:00.000", Array(), true,
            MonthOfTheYear::November, "12:00:00.000", 60, 1, RecurrencePattern::Yearly,
            RecurrencePatternType::Simple, true, WeekOfTheMonth::None, WeekDay::Friday );
        $recurrence2     = new ScheduleRecurrenceDetails( 1, 30, "12:00:00.000", "13:00:00.000", Array(), true,
            MonthOfTheYear::December, "12:00:00.000", 60, 1, RecurrencePattern::Daily,
            RecurrencePatternType::Simple, false, WeekOfTheMonth::None, WeekDay::Friday );
        $scheduleDetails = new CreateScheduleFromTemplateDetails( $this->SourceTemplateId, $name, Array( $recurrence,
                $recurrence2 ),
            ScheduleTitleType::ScheduleNameAndNumber, 15, 60, 2, false, false, false, false, false );
        $response        = self::$client->CreateScheduleFromTemplate($scheduleDetails);

        $queryCriteria   = new PresentationScheduleQueryCriteria( false, QueryScheduleBy::ScheduledId, null, null, null, null, null, $response->ScheduleId );
        $createdSchedule = self::$client->QuerySchedulesByCriteria($queryCriteria)->ScheduleList[0];

        //  lots to test here, so just testing a few arbitrary and easily-verified properties
        $this->assertEquals($name, $createdSchedule->Name, "Created name does not match provided name");
        $this->assertEquals($scheduleDetails->TitleType, $createdSchedule->TitleType,
            "Created TitleType does not match provided TitleType");

        //  RecurrencePattern is a little complicated
        $this->assertEquals($recurrence->MonthOfTheYear,
            $createdSchedule->RecurrenceList->ScheduleRecurrenceDetails[0]->MonthOfTheYear,
            "Created MonthOfTheYear does not match provided MonthOfTheYear");
        $this->assertEquals($recurrence->RecurrencePatternType,
            $createdSchedule->RecurrenceList->ScheduleRecurrenceDetails[0]->RecurrencePatternType,
            "Created RecurrencePatternType does not match provided RecurrencePatternType");
        $this->assertEquals('false',
            $createdSchedule->RecurrenceList->ScheduleRecurrenceDetails[0]->WeekDayOnly,
            "WeekDayOnly should be disregarded (forced to false) if RecurrencePattern is not Daily");
        $this->assertEquals(MonthOfTheYear::None,
            $createdSchedule->RecurrenceList->ScheduleRecurrenceDetails[1]->MonthOfTheYear,
            "Created MonthOfTheYear should be None if RecurrencePattern is not Monthly or Yearly");

        //  updating is a bear - there are a lot of properties to set
        //  the booleans right after Id are all flags to tell the server which parameters have been changed in the updated Schedule
        $scheduleUpdate  = new UpdateScheduleDetails( $createdSchedule->Id, false, false, true, false, false, false, false,
            false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false, false,
            $createdSchedule->AdvanceCreationTime, $createdSchedule->AdvanceLoadTimeInSeconds + 1, !$createdSchedule->AutoStart,
            $createdSchedule->AutoStop, $createdSchedule->CreatePresentation, $createdSchedule->DeleteInactive, $createdSchedule->IsForumEnabled,
            $createdSchedule->IsLive, $createdSchedule->IsOnDemand, $createdSchedule->IsPollsEnabled, $createdSchedule->IsUploadAutomatic,
            $createdSchedule->LoadPresentation, $createdSchedule->NextNumberInSchedule, $createdSchedule->NotifyPresenter,
            $createdSchedule->ReplaceAclWithPolicy, $createdSchedule->ReviewEditApproveEnabled, $createdSchedule->TimeZoneRegistryKey
        );
        $updateId        = self::$client->UpdateSchedule($scheduleUpdate)->ScheduleId;
        $queryCriteria   = new PresentationScheduleQueryCriteria( false, QueryScheduleBy::ScheduledId, null, null, null, null, null, $response->ScheduleId );
        $updatedSchedule = self::$client->QuerySchedulesByCriteria($queryCriteria)->ScheduleList[0];

        $this->assertEquals($createdSchedule->AdvanceLoadTimeInSeconds, $updatedSchedule->AdvanceLoadTimeInSeconds,
            'Since the update object indicated that the AdvancedLoadTimeInSeconds was not set, it should not be updated');
        $this->assertNotEquals($createdSchedule->AutoStart, $updatedSchedule->AutoStart,
            'Since the update object indicated that the AutoStart value was set and it was changed in the request, it should have been updated');

        self::$client->DeleteSchedule($createdSchedule->Id);
    }

    /**
     * @covers ExternalAccessClient::GetVersion
     * @group stateless
     */
    public function testGetVersion() {
        $version = self::$client->GetVersion()->Version;
        $this->assertEquals($this->ServicesVersion, $version);
    }

    /**
     * @covers ExternalAccessClient::Logout
     * retest after updating impersonation stuff
     * @group broken
     */
    public function testLogout() {
        self::$client->Logout();
        $valid = self::$client->QueryIdentityTicketProperties(self::$client->Ticket, 60, false);
    }

    /**
     * @covers ExternalAccessClient::QueryCatalogShares
     * @todo   set up a not-state-sensitive test
     * @group state-sensitive
     *         the quantity of catalogs returned will depend on how many the logged-in user has access to
     */
    public function testQueryCatalogShares() {
        $catalogResponse = self::$client->QueryCatalogShares(array( ResourcePermissionMask::Read,
            ResourcePermissionMask::Write ));
        $this->assertEquals(5, count($catalogResponse->Shares));
    }

    /**
     * @covers ExternalAccessClient::QueryChapterPoints
     * @group state-sensitive
     *  chapter details must match expected values. Since there's no api for adding chapter details,
     *  there's no way to make this stateless
     */
    public function testQueryChapterPoints() {
        //  test environment doesn't have presentation with chapters, find some in another Mediasite Instance
        $client          = new ExternalAccessClient( self::$EdasEndPoint );
        $ticket          = $client->Login(self::$Username, self::$Password);
        $chapterResponse = $client->QueryChapterPoints(5, '71d28a84-dd35-4f92-98f9-e9a2907dcd7f', 0, $ticket->UserTicket);
        $this->assertEquals(3, count($chapterResponse->ChapterPoints), 'Incorrect number of chapters');
        $this->assertEquals('Untitled', $chapterResponse->ChapterPoints[0]->Title);
    }

    /**
     * @covers ExternalAccessClient::QueryClientIpAddress
     * @group stateless
     */
    public function testQueryClientIpAddressByIp() {
        $result = self::$client->QueryClientIpAddress(true);
        $this->assertEquals($this->LocalhostIp, $result->IpAddress);
        $this->assertEquals($this->LocalhostHostName, $result->DnsName);
    }

    /**
     * @covers ExternalAccessClient::QueryContentServersByCriteria
     * @group state-sensitive
     *  ContentServer layout must match expectations
     */
    public function testQueryContentServersByPresentationIdReturnsCorrectInfo() {
        $criteria = new ContentServerQueryCriteria( ContentServerQueryBy::Presentation, true, $this->TestingPresentationGuid, null, null );
        $result   = self::$client->QueryContentServersByCriteria($criteria);
        $this->assertEquals(4, count($result->ContentServers));

        $this->assertEquals(ContentServerEndpointType::Distribution, $result->ContentServers[0]->ServerConnections->ContentServerEndpoint[0]->EndpointType,
            'First content server endpoint is a distribution endpoint');
        $this->assertNotNull($result->ContentServers[0]->ServerConnections->ContentServerEndpoint[0]->EndpointType, 'First content server endpoint has a distribution url');
        $this->assertObjectNotHasAttribute('Password', $result->ContentServers[0]->ServerConnections->ContentServerEndpoint[0], 'First content server endpoint does not have a password');
        $this->assertEquals(ContentServerEndpointType::Storage, $result->ContentServers[0]->ServerConnections->ContentServerEndpoint[1]->EndpointType,
            'First content server endpoint is a storage');
        $this->assertNotNull($result->ContentServers[0]->ServerConnections->ContentServerEndpoint[1]->Password, 'Second content server endpoint has a password');
        $this->assertObjectNotHasAttribute('DistributionUrl', $result->ContentServers[0]->ServerConnections->ContentServerEndpoint[1], 'Second content server endpoint does not have a distribution url');
        $this->assertEquals(ContentServerTypeDetails::IIsProgressiveServer, $result->ContentServers[0]->ServerType);
    }

    /**
     * @covers ExternalAccessClient::QueryFoldersWithPresentations
     * @group state-sensitive   //  must have the correct quantity of system folders containing presentations
     */
    public function testQueryFoldersWithPresentations() {
        $resp = self::$client->QueryFoldersWithPresentations();
        $this->assertEquals(8, count($resp->Folders), 'Response folder quantity does not match number of folders with prsentations');
    }

    /**
     * @covers ExternalAccessClient::QueryContentEncodingSettingsById
     * @group state-sensitive
     */
    public function testQueryContentEncodingSettingsById() {
        $response         = self::$client->QueryContentEncodingSettingsById(array( $this->TestEncodingSettingId ));
        $encodingSettings = self::$client->QueryContentEncodingSettingsById(array( $this->TestEncodingSettingId ));
        $this->assertEquals($this->TestEncodingSettingName, $encodingSettings->ContentEncodingSettings[0]->Name);
    }

    /**
     * @covers ExternalAccessClient::QueryContentEncodingSettingsByCriteria
     * @group state-sensitive
     */
    public function testQueryContentEncodingSettingsByCriteria() {
        //  default audio encoding setting should be of mimetype video., so checing that type will return 1 for a single id
        //  checking same id for an image type will return none
        $validCriteria   = new ContentEncodingSettingsQueryCriteria( array( $this->TestEncodingSettingId ), array( 'video/x-ms-wmv' ) );
        $validSettings   = self::$client->QueryContentEncodingSettingsByCriteria($validCriteria)->ContentEncodingSettings;
        $invalidCriteria = new ContentEncodingSettingsQueryCriteria( array( $this->TestEncodingSettingId ), array( 'image/jpeg' ) );
        $invalidSettings = self::$client->QueryContentEncodingSettingsByCriteria($invalidCriteria)->ContentEncodingSettings;
        $this->assertEquals(1, count($validSettings), 'incorrect quantity of valid encoding settings returned');
        $this->assertEquals(0, count($invalidSettings), 'incorrect quantity of invalid encoding settings returned');
    }

    /**
     * @covers ExternalAccessClient::QueryPlayers
     * @covers ExternalAccessClient::CreatePlayerLike
     * @covers ExternalAccessClient::UpdatePlayer
     * @covers ExternalAccessClient::DeletePlayer
     * @group stateless
     */
    public function testQueryPlayers() {
        $details  = new CreatePlayerLikeDetails( $this->TestPlayerId, 'From CreatePlayerLike', $this->TestFolderId );
        $playerId = self::$client->CreatePlayerLike($details)->Id;

        /**
         * @param $ar
         * @param $name
         * @param $field
         *
         * @return bool
         */
        $playerFieldExists = function ( $ar, $name, $field ) {
            foreach ( $ar as $val ) {
                if ( isset( $val->$field ) && $val->$field == $name ) {
                    return true;
                }
            }

            return false;
        };

        $playerResponse = self::$client->QueryPlayers();
        $this->assertTrue($playerFieldExists($playerResponse->Players, $this->TestPlayerName, 'Name'), 'Players does not contain created Player');

        $playerUpdate = new UpdatePlayerDetails( $playerId, null, false, null, false, 'A new player name', true );
        $resp         = self::$client->UpdatePlayer($playerUpdate);

        $playerResponse = self::$client->QueryPlayers();
        $this->assertTrue($playerFieldExists($playerResponse->Players, 'A new player name', 'Name'), 'Players does not contain updated Player');

        self::$client->DeletePlayer($playerId);
    }

    /**
     * @covers ExternalAccessClient::QueryPresentationsByCriteria
     * @group stateless
     */
    public function testQueryPresentationsByCriteria() {
        //  validate that presentation was, indeed, created
        $today                = new DateTime( 'now', new DateTimeZone( 'America/Chicago' ) );
        $presentationCriteria = new PresentationQueryCriteria( '2012-01-01', $today->format('c'),
            array( ResourcePermissionMask::Read, ResourcePermissionMask::Write ), false );
        $opts                 = new QueryOptions( 3, 'testQueryPresentationsByCriteria', 0 );
        $response             = self::$client->QueryPresentationsByCriteria($presentationCriteria, $opts);

        //  to keep this as stateless as possible, just checking to see if the query is paying attention to the
        //      QueryOptions
        $this->assertEquals($opts->BatchSize, count($response->Presentations));
    }

    /**
     * @covers ExternalAccessClient::QueryPresentersByCriteria
     * @covers ExternalAccessClient::QueryPresentersById
     * @group stateless
     */
    public function testQueryPresentersByCriteria() {
        $criteria       = new PresenterQueryCriteria();
        $allPresenters  = self::$client->QueryPresentersByCriteria($criteria);
        $testPresenter  = null;
        $presenterIndex = -1;

        //  need to make sure we have an email address
        foreach ( $allPresenters->Presenters as $index => $presenter ) {
            if ( is_string($presenter->Email) ) {
                $testPresenter  = $presenter;
                $presenterIndex = $index;
                break;
            }
        }
        $this->assertEquals($allPresenters->Presenters[$presenterIndex]->Email, $testPresenter->Email, 'PresenterByCriteria email address not the same as PresenterById email addresss');

        //  less trivial example of criteria query
        $newCriteria   = new PresenterQueryCriteria( $testPresenter->Email, $testPresenter->FirstName, $testPresenter->Id, $testPresenter->LastName );
        $newPresenters = self::$client->QueryPresentersByCriteria($newCriteria);

        $this->assertEquals($testPresenter->ImageUrl, $newPresenters->Presenters[0]->ImageUrl, 'First presenter image location is not the same as specific presenter image');
    }

    /**
     * @covers ExternalAccessClient::QueryResourcePermissionList
     * @covers ExternalAccessClient::QueryResourcePermissions
     * @covers ExternalAccessClient::UpdateResourcePermissions
     * @group state-sensitive
     * In a default configuration, and logging in as a member of MediasiteAdministrators, this should be stateless
     */
    public function testQueryResourcePermissionList() {

        $resource             = new ResourceIdentifier( $this->TestingPresentationId, ResourceType::Presentation );
        $firstResponse        = self::$client->QueryResourcePermissionList($resource);
        $mediasiteAdminsPerms = '';
        foreach ( $firstResponse->ResourcePermission->PermissionList as $perm ) {
            if ( $perm->RoleId == $this->MediasiteAdminsRoleId ) {
                $mediasiteAdminsPerms = $perm;
                break;
            }
        }

        $this->assertEquals('Read Write Execute Moderate', $mediasiteAdminsPerms->PermissionMask, 'PermissionMask for MediasiteAdministrators is incorrect');
        $this->assertEquals($this->TestingPresentationId, $firstResponse->ResourcePermission->Resource->Id, 'Returned resource Id does not equal testing presentation id');

        $secondResponse = self::$client->QueryResourcePermissions(array( $firstResponse->ResourcePermission->Resource ));

        $this->assertEquals($this->TestingPresentationGuid, $secondResponse->ResourceList[0]->Id, 'Id of returned resource does not match testing presentation guid');
        $this->assertEquals($mediasiteAdminsPerms->PermissionMask, $secondResponse->ResourceList[0]->PermissionMask, 'PermissionMask of returned resource does not match MediasiteAdministrators PermissionMask');

        $permissionMasks  = Array( ResourcePermissionMask::Execute, ResourcePermissionMask::Read, ResourcePermissionMask::Write );
        $permissionsEntry = new ResourcePermissionEntry( $permissionMasks, $this->MediasiteAdminsRoleId );
        $permissionUpdate = new UpdateResourcePermissionsDetails( Array( $permissionsEntry ), Array( $firstResponse->ResourcePermission->Resource ), false, false, self::$Username );
        self::$client->UpdateResourcePermissions($permissionUpdate);

        $thirdResponse = self::$client->QueryResourcePermissions(array( $firstResponse->ResourcePermission->Resource ));

        $this->assertEquals($this->TestingPresentationGuid, $thirdResponse->ResourceList[0]->Id, 'Id of returned resource does not match testing presentation guid');
        $this->assertEquals("Read Write Execute", $thirdResponse->ResourceList[0]->PermissionMask, 'PermissionMask of returned resource does not match updated mask');

        $permissionsEntry->PermissionMask[] = ResourcePermissionMask::Moderate;
        $permissionUpdate                   = new UpdateResourcePermissionsDetails( Array( $permissionsEntry ), Array( $firstResponse->ResourcePermission->Resource ), false, false, self::$Username );
        self::$client->UpdateResourcePermissions($permissionUpdate);

    }

    /**
     * @covers ExternalAccessClient::QuerySiteProperties
     * @group state-sensitive
     */
    public function testQuerySiteProperties() {
        $response = self::$client->QuerySiteProperties();
        $this->assertEquals($this->SiteVersion, $response->Properties->Version, 'Returned Site version incorrect');
    }

    /**
     * @covers ExternalAccessClient::QuerySlides
     * @group state-sensitive
     */
    public function testQuerySlides() {
        $slideCount = 5;
        $slideStart = 10;
        $response   = self::$client->QuerySlides($slideCount, $this->TestingPresentationId, $slideStart);
        $this->assertEquals($slideCount, count($response->Slides), 'Returned slide count does not equal expected count');
        $this->assertEquals($slideStart + 1, $response->Slides[0]->Number, 'First returned slide number does not equal expected value');
    }

    /**
     * @covers ExternalAccessClient::Test
     * @group stateless
     */
    public function testTest() {
        $result = self::$client->Test();
        $this->assertEquals(1, $result->Value);
    }

    /**
     * @covers ExternalAccessClient::Search
     * @group stateless
     *  although the query string might need to be updated for your environment
     *  This test illustrates the usage of the QueryOptions class and iterating over a series of queries until all the results are retrieved
     */
    public function testSearch() {
        $batchSize    = 15;
        $opt          = new QueryOptions( $batchSize, 'test', 0 );
        $results      = array();
        $searchString = 'cori';
        do {
            $response  = self::$client->Search(array( SupportedSearchField::Owner, SupportedSearchField::Name ),
                                               $searchString,
                                               array( SupportedSearchType::Presentation, SupportedSearchType::Folder ),
                                               $opt,
                                               TRUE,
                                               TRUE);
            $results[] = $response;
            //  NextQueryOptions is not returned as a recognizable QueryOptions object, so we have to map it over
            $opt = TypeMapFunctions::objectToObject($response->Results->NextQueryOptions, 'QueryOptions');
        } while ( $response->Results->MoreResultsAvailable );

        $totalResults     = $response->Results->TotalResults;
        $lastResultsArray = array_slice($results, -1, 1);
        $lastResults      = $lastResultsArray[0];
        $this->assertEquals($totalResults, ( count($results) - 1 ) * $batchSize + count($lastResults->DetailList->SearchResponseDetails));
    }

    //  region Highly State Dependent Tests
    //  the following tests are highly state dependent and largely call for specific configuration to be put into place manually before running
    //  this is mostly due to the fact that none of these Delete* methods have a corresponding Create* mechanism

    /**
     * @covers ExternalAccessClient::DeletePodcast
     * @group state-sensitive
     */
    public function testDeletePodcast() {
        $feedId   = '7b07b161-3b6b-4af5-a6f4-494e13930eac';
        $response = self::$client->DeletePodcast($feedId);
        $this->assertEquals('Your item has been deleted.', $response->Message, 'Message is incorrect');
    }

    /**
     * @covers ExternalAccessClient::DeleteMediaImportProject
     * @group state-sensitive
     */
    public function testDeleteMediaImportProject() {
        $projectId = '62d98ce9-d669-47f0-9f40-c62ad43eddd0';
        $response  = self::$client->DeleteMediaImportProject($projectId);
        $this->assertEquals('Your item has been deleted.', $response->Message, 'Message is incorrect');
    }

    /**
     * @covers ExternalAccessClient::DeleteContentEncodingSettings
     * @group state-sensitive
     */
    public function testDeleteContentEncodingSettings() {
        $settingId = 'bff37849-52a2-444a-9969-672e1d6309fe';
        $response  = self::$client->DeleteContentEncodingSettings($settingId);
        $this->assertEquals('Your item has been deleted.', $response->Message, 'Message is incorrect');
    }

    /**
     * @covers ExternalAccessClient::DeleteContentServer
     * @group state-sensitive
     */
    public function testDeleteContentServer() {
        $serverId = '6380358c-0df8-4c4b-9da1-a6b6b31dc1ff';
        $response = self::$client->DeleteContentServer($serverId);
        $this->assertEquals('Your item has been deleted.', $response->Message, 'Message is incorrect');
    }

    //endregion

    /**
     * @covers ExternalAccessClient::CreateCatalogFromFolder
     * @covers ExternalAccessClient::QueryCatalogById
     * @covers ExternalAccessClient::DeleteCatalog
     * @group stateless
     */
    public function test_CreateCatalogFromFolder() {

        $name           = 'Catalog From Folder';
        $desc           = 'Created by testCreateCatalogFromFolder';
        $catalogDetails = new CreateCatalogFromFolderDetails( $this->TestFolderId, false, $name, $desc );
        $response       = self::$client->CreateCatalogFromFolder($catalogDetails);

        $createdCatalog = self::$client->QueryCatalogsById(array( $response->CatalogId ))->Shares[0];

        $this->assertEquals($name, $createdCatalog->Name, 'Created catalog name does not match provided name');
        $this->assertEquals($desc, $createdCatalog->Description, 'Created catalog name does not match provided name');
        $this->assertEquals('false', $createdCatalog->Recycled, 'Created catalog should not be recycled');

        self::$client->DeleteCatalog($response->CatalogId);
        $catalogAfterDelete = self::$client->QueryCatalogsById(array( $response->CatalogId ))->Shares[0];

        $this->assertEquals('true', $catalogAfterDelete->Recycled, 'After deletion, catalog should be recycled');
    }

    //endregion

    /*  New for 6.1.1    */

    //region UserProfiles

    /**
     * @covers ExternalAccessClient::CreateUserProfile
     * @covers ExternalAccessClient::QueryUserProfilesById
     * @group stateless
     */
    public function testCreateUserProfileAndQueryProducesMatchingProfiles() {
        $username       = "tup";
        $profileDetails = new UserProfileCreateDetails( "test user profiles", "user.profile@sonicfoundry.net", $username );
        $response       = self::$client->CreateUserProfiles(Array( $profileDetails ));
        $this->assertEquals(1, count($response->ProfileMappings), "Returned profile mappings count incorrect");
        $this->assertEquals($username, $response->ProfileMappings[0]->Value, "Usernames do not match");

        $validResponse = self::$client->QueryUserProfilesById(Array( $response->ProfileMappings[0]->Id ));
        $this->assertEquals($username, $validResponse->UserProfiles[0]->Value, "Profile names do not match");

    }

    /**
     * @covers ExternalAccessClient::CreateUserProfile
     * @covers ExternalAccessClient::QueryUserProfilesById
     * @todo    should test multiple
     * @group stateless
     */
    public function testCreateUserProfilesFromEmailsAndQueryReturnsMatchingProfiles() {
        $email    = "test@sonicfoundry.com";
        $response = self::$client->CreateUserProfilesFromEmails(Array( $email ));

        $this->assertEquals($email, $response->ProfileMappings[0]->Value, "Profile emails do not match");
        $this->assertEquals(1, count($response->ProfileMappings), "Returned the wrong quantity of ProfileMappings");

        $validResponse = self::$client->QueryUserProfilesById(Array( $response->ProfileMappings[0]->Id ));

        $this->assertEquals($email, $validResponse->UserProfiles[0]->Value, "Profile emails do not match");
        $this->assertEquals(1, count($validResponse->UserProfiles), "Returned the wrong quantity of ProfileMappings");
    }

    /**
     * @covers ExternalAccessClient::QueryUserProfilesByCriteria
     * @todo    fails
     * @group state-sensitive   profile must exist
     */
    public function testQueryUserProfilesByEmailCriteriaFindsCorrectProfiles() {

        $criteria      = new UserProfileQueryCriteria( Array( "user.profile@sonicfoundry.net", "coris@sonicfoundry.net" ), null, null, null );
        $validResponse = self::$client->QueryUserProfilesByCriteria($criteria);
        $this->assertEquals(1, count($validResponse->UserProfiles));

    }

    /**
     * @covers ExternalAccessClient::QueryUserProfilesByCriteria
     * @group UserProfiles
     * @group state-sensitive   profile must exist
     */
    public function testQueryUserProfilesByEmailContainsCriteriaFindsCorrectProfiles() {

        $criteria      = new UserProfileQueryCriteria( null, "sonicfoundry.net", null, null );
        $validResponse = self::$client->QueryUserProfilesByCriteria($criteria);
        $this->assertEquals(7, count($validResponse->UserProfiles));

    }

    /**
     * @covers ExternalAccessClient::QueryUserProfilesByCriteria
     * @group UserProfiles
     * @group state-sensitive
     *  profile must exist
     */
    public function testQueryUserProfilesByUsernameContainsCriteriaFindsCorrectProfiles() {

        $criteria      = new UserProfileQueryCriteria( null, null, "up", null );
        $validResponse = self::$client->QueryUserProfilesByCriteria($criteria);
        $this->assertEquals(4, count($validResponse->UserProfiles));

    }

    /**
     * @covers ExternalAccessClient::QueryUserProfilesByCriteria
     * @group state-sensitive   profile must exist
     */
    public function testQueryUserProfilesByUsernameCriteriaFindsCorrectProfiles() {

        $criteria      = new UserProfileQueryCriteria( null, null, null, Array( "cori", "msa" ) );
        $validResponse = self::$client->QueryUserProfilesByCriteria($criteria);
        $this->assertEquals(2, count($validResponse->UserProfiles));

    }

    /**
     * @covers ExternalAccessClient::QueryUserProfilesByCriteria
     * @group state-sensitive   profile must exist
     */
    public function testQueryUserProfilesByUsernameAndEmailContainsCriteriaFindsCorrectProfiles() {

        $criteria      = new UserProfileQueryCriteria( null, "sonicfoundry.net", null, Array( "cori", "MediasiteAdmin" ) );
        $validResponse = self::$client->QueryUserProfilesByCriteria($criteria);
        $this->assertEquals(1, count($validResponse->UserProfiles));

    }

    /**
     * @covers  ExternalAccessClient::UpdateUserProfiles
     * @group UserProfiles
     * @group stateless
     */
    public function testUpdateUserProfilesDisplayNameUpdatesUserProfiles() {
        $now            = new DateTime();
        $userName       = $now->getTimestamp();
        $email          = $userName . 'sofo.net';
        $createProfile  = new UserProfileCreateDetails( $userName, $email, $userName );
        $createResponse = self::$client->CreateUserProfiles(Array( $createProfile ));
        $profileId      = $createResponse->ProfileMappings[0]->Id;

        $newUsername = 'NewUsername';

        $profileUpdates = new UserProfileUpdateDetails( $profileId, null, false, null, false, false, $newUsername, true );
        self::$client->UpdateUserProfiles(Array( $profileUpdates ));

        $updatedProfile = self::$client->QueryUserProfilesById(Array( $profileId ));
        $this->assertEquals($newUsername, $updatedProfile->UserProfiles[0]->Value, "Updated username does not match");

    }

    //endregion

    /**
     * @covers ExternalAccessClient::CheckJobStatus
     * @group state-sensitive   parent folder must exist
     */
    public function testCheckJobStatusReturnsQueuedWhenJobsNotRunning() {
        $newFolder      = self::$client->CreateSubFolder("test folder", Array( new ResourcePermissionEntry( Array( ResourcePermissionMask::Write ), $this->UpdateRoleId ) ));
        $deleteResponse = self::$client->DeleteFolder($newFolder->Id);
        $checkResponse  = self::$client->CheckJobStatus($deleteResponse->JobId);
        $this->assertEquals("Queued", $checkResponse->Status, "New job status is not queued");
    }

    /**
     * @covers ExternalAccessClient::AddTagToMediasiteObject
     * @covers ExternalAccessClient::QueryTagsByMediasiteId
     * @covers ExternalAccessClient::RemoveTagFromMediasiteObject
     * @group state-sensitive   presentation must exist
     */
    public function testAddTagToMediasiteObjectAddsTags() {

        $originalTags = self::$client->QueryTagsByMediasiteId($this->TestingPresentationGuid)->Tags;
        $tagName      = "TagName" . $this->CurrentTimestamp;

        //  add a tag and get new tag set and check count
        self::$client->AddTagToMediasiteObject($this->TestingPresentationGuid, $tagName);
        $queryResponse = self::$client->QueryTagsByMediasiteId($this->TestingPresentationGuid);
        $this->assertEquals(count($originalTags) + 1, count($queryResponse->Tags));

        //  remove previously added tag an check counts
        self::$client->RemoveTagFromMediasiteObject($this->TestingPresentationGuid, $tagName);
        $afterQueryResponse = self::$client->QueryTagsByMediasiteId($this->TestingPresentationGuid);
        $this->assertEquals(count($originalTags), count($afterQueryResponse->Tags));

    }

    /**
     * @covers ExternalAccessClient::CreateTemplateLike
     * @covers ExternalAccessClient::QueryPresentationTemplatesByCriteria
     * @covers ExternalAccessClient::DeletePresentationTemplate
     * @group state-sensitive   source template must exist
     */
    public function testCreateTemplateLike() {

        //  get source template; CreateTemplateLikeDetails requires a playerid and parentfolderid
        $sourceQueryOpts    = new PresentationTemplateQueryCriteria( false, null, Array( $this->SourceTemplateId ) );
        $sourceTemplateResp = self::$client->QueryPresentationTemplatesByCriteria($sourceQueryOpts);
        $sourceTemplate     = $sourceTemplateResp->PresentationTemplates[0];

        //  set up new template details
        $templateName    = "New Template_" . $this->CurrentTimestamp;
        $details         = new CreateTemplateLikeDetails( $this->SourceTemplateId, $templateName, $sourceTemplate->PlayerId, $sourceTemplate->ParentFolderId );
        $newTemplateId   = self::$client->CreateTemplateLike($details)->TemplateId;
        $newQueryOpts    = new PresentationTemplateQueryCriteria( false, null, Array( $newTemplateId ) );
        $newTemplateResp = self::$client->QueryPresentationTemplatesByCriteria($newQueryOpts);
        $newTemplate     = $newTemplateResp->PresentationTemplates[0];

        //  validate
        $this->assertEquals($sourceTemplate->PlayerId, $newTemplate->PlayerId, 'Template Player Id is not the same as the Source template player Id');
        $this->assertEquals($templateName, $newTemplate->Name, 'Template Name is not correct');
        $this->assertEquals(1, $newTemplateResp->Results->TotalResults, 'Queried template results returned the wrong Total Results');

        //  delete created template and revalidate
        self::$client->DeletePresentationTemplate($newTemplate->Id);
        $afterDeleteTemplateResp = self::$client->QueryPresentationTemplatesByCriteria($newQueryOpts);
        $this->assertEquals(0, count($afterDeleteTemplateResp->PresentationTemplates), 'Queried results after delete found templates');
        $this->assertEquals(0, $afterDeleteTemplateResp->Results->TotalResults, 'Queried results after delete returned incorrect Total Results');

    }

    //region MediasiteKeyValue
    //  TODO:   test queries when multiple MediasiteObjects have the same key/value
    //  TODO:   try to use typemapping functions to hoist these values better
    /**
     * @covers ExternalAccessClient::CreateMediasiteKeyValue
     * @covers ExternalAccessClient::QueryMediasiteKeyValuesById
     * @covers ExternalAccessClient::QueryMediasiteKeyValueByKeyAndId
     * @covers ExternalAccessClient:DeleteMediasiteKeyValueByIdAndKey
     * @group MediasiteKeyValue
     * @group stateless
     */
    public function test_ByIdAndKey_CreateMediasiteKeyValueAddKeyAndDeleteByIdRemovesIt() {

        $arrayHasKVP = function ( $key, $ar ) {
            foreach ( $ar as $k => $v ) {
                if ( $v->Key == $key ) {
                    return true;
                }
            }

            return false;
        };

        $key = "Key_" . $this->CurrentTimestamp;
        $val = "Value_" . $this->CurrentTimestamp;
        $kv  = new MediasiteKeyValue( $this->SourceTemplateId, $key, $val );

        //  add the KeyValue and verify that the object contains it
        $resp                 = self::$client->CreateMediasiteKeyValue($kv);
        $queriedKeysAndValues = self::$client->QueryMediasiteKeyValuesById($resp->Id);
        $queriedKv            = $queriedKeysAndValues->KeyValues;
        $this->assertTrue($arrayHasKVP($key, $queriedKv), 'Queried MediasiteKeyValues does not contain an object with Key: ' . $key);

        //  delete the KeyValue and verify that the object does not contain it
        self::$client->DeleteMediasiteKeyValueByIdAndKey($resp->Id, $key);
        $postQueriedKv        = self::$client->QueryMediasiteKeyValuesById($resp->Id);
        $queryKVByIdAndKeyKVP = self::$client->QueryMediasiteKeyValuesByIdAndKey($resp->Id, $key)->KeyValues;

        $this->assertFalse($arrayHasKVP($key, $postQueriedKv->KeyValues), 'Queried MediasiteKeyValues contains an object with Key: ' . $key);
        $this->assertEquals(0, count($queryKVByIdAndKeyKVP), 'Count of queried KVP with this Id and Key is not 0');

    }

    /**
     * @covers ExternalAccessClient::CreateMediasiteKeyValue
     * @covers ExternalAccessClient::QueryMediasiteKeyValuesByKeyValue
     * @covers ExternalAccessCLient::QueryMediaisteKeyValuesByCriteria
     * @covers ExternalAccessClient:DeleteMediasiteKeyValueByKeyValue
     * @group MediasiteKeyValue
     * @group state-sensitive
     *  source template must exist
     */
    public function test_ByKeyValue_CreateMediasiteKeyValueAddKeyAndDeleteRemovesIt() {

        $arrayHasKVP = function ( $kv, $ar ) {
            if ( is_array($ar) ) {
                foreach ( $ar as $k => $v ) {
                    if ( serialize($v) == serialize($kv) ) {
                        return true;
                    }
                }
            } else {
                return $kv->Key == $ar->Key && $kv->Value == $ar->Value && $kv->Id == $ar->Id;
            }

            return false;
        };

        $key = "Key_" . $this->CurrentTimestamp;
        $val = "Value_" . $this->CurrentTimestamp;
        $kv  = new MediasiteKeyValue( $this->SourceTemplateId, $key, $val );

        //  add the KeyValue and verify that the object contains it
        self::$client->CreateMediasiteKeyValue($kv);
        $queriedKv = self::$client->QueryMediasiteKeyValuesByKeyValue($kv);
        $this->assertTrue($arrayHasKVP($kv, $queriedKv->KeyValues), 'Queried MediasiteKeyValues does not contain an object like provided KeyValue');

        $criteriaQueriedKVById = self::$client->QueryMediasiteKeyValuesByCriteria($queriedKv->KeyValues[0]->Id);
        $this->assertTrue($arrayHasKVP($kv, $criteriaQueriedKVById->KeyValues));
        $criteriaQueriedKVByKey = self::$client->QueryMediasiteKeyValuesByCriteria(null, $key);
        $this->assertTrue($arrayHasKVP($kv, $criteriaQueriedKVByKey->KeyValues));
        $criteriaQueriedKVByVal = self::$client->QueryMediasiteKeyValuesByCriteria(null, null, $val);
        $this->assertTrue($arrayHasKVP($kv, $criteriaQueriedKVByVal->KeyValues));

        //  delete the KeyValue and verify that the object does not contain it
        self::$client->DeleteMediasiteKeyValueByKeyValue($kv);
        $postQueriedKv = self::$client->QueryMediasiteKeyValuesByKeyValue($kv);
        $this->assertEquals(0, count($postQueriedKv->KeyValues), 'Count of queried KVP with this Id and Key is not 0');

    }

    /**
     * @covers ExternalAccessClient::CreateMediasiteKeyValue
     * @covers ExternalAccessClient::UpdateMediasiteKeyValuesByIdsAndKey
     * @covers ExternalAccessClient:DeleteMediasiteKeyValueByKeyValue
     * @group state-sensitive
     *  source template must exist
     */
    public function test_UpdateMediasiteKeyValueByIdsAndAddKey() {

        $arrayHasKVP = function ( $kv, $ar ) {
            if ( is_array($ar) ) {
                foreach ( $ar as $k => $v ) {
                    if ( serialize($v) == serialize($kv) ) {
                        return true;
                    }
                }
            } else {
                return $kv->Key == $ar->Key && $kv->Value == $ar->Value && $kv->Id == $ar->Id;
            }

            return false;
        };

        $key            = "Key_" . $this->CurrentTimestamp;
        $val            = "Value_" . $this->CurrentTimestamp;
        $templateKv     = new MediasiteKeyValue( $this->SourceTemplateId, $key, $val );
        $presentationKv = new MediasiteKeyValue( $this->TestingPresentationGuid, $key, $val );

        //  add the KeyValue and verify that the object contains it
        $respTemplate          = self::$client->CreateMediasiteKeyValue($templateKv);
        $respPresentation      = self::$client->CreateMediasiteKeyValue($presentationKv);
        $queriedTemplateKv     = self::$client->QueryMediasiteKeyValuesByKeyValue($templateKv);
        $queriedPresentationKv = self::$client->QueryMediasiteKeyValuesByKeyValue($presentationKv);
        $this->assertEquals($queriedTemplateKv->KeyValues[0]->Value, $queriedPresentationKv->KeyValues[0]->Value, 'KeyValue values are not equal');

        //update and verify
        $newVal = strrev($val);
        self::$client->UpdateMediasiteKeyValueByIdsAndKey(array( $this->SourceTemplateId, $this->TestingPresentationGuid ), $key, $newVal);

        $updatedTemplateKv     = self::$client->QueryMediasiteKeyValuesByIdAndKey($this->SourceTemplateId, $key);
        $updatedPresentationKv = self::$client->QueryMediasiteKeyValuesByIdAndKey($this->TestingPresentationGuid, $key);
        $this->assertEquals($newVal, $updatedPresentationKv->KeyValues[0]->Value, 'KeyValue values are not equal');
        $this->assertEquals($updatedTemplateKv->KeyValues[0]->Value, $updatedPresentationKv->KeyValues[0]->Value, 'KeyValue values are not equal');

        //  delete the KeyValue and verify that the object does not contain it
        self::$client->DeleteMediasiteKeyValueByKeyValue($updatedPresentationKv->KeyValues[0]);
        self::$client->DeleteMediasiteKeyValueByKeyValue($updatedTemplateKv->KeyValues[0]);
        $deletedPresentationKv = self::$client->QueryMediasiteKeyValuesByKeyValue($presentationKv);
        $deletedTemplateKv     = self::$client->QueryMediasiteKeyValuesByKeyValue($templateKv);
        $this->assertEquals(0, count($deletedPresentationKv->KeyValues), 'Queried KVP with this Id and Key is not null');
        $this->assertEquals(0, count($deletedTemplateKv->KeyValues), 'Queried KVP with this Id and Key is not null');

    }

    //endregion

    //region    new for 6.1.5

    /**
     * @covers UpdatePresentationContentDetails
     * @group state-sensitive
     *  requires specific presentation and presentation content ids to be present and correctly configuired
     */
    public function test_UpdatePresentationContentDetails() {

        $testId = $this->TestingPresentationContentItemId;

        /**
         * helper callback function
         *
         * @internal
         * @param $testId
         *
         * @return callable
         */
        function create_id_callback( $testId ) {
            return function ( $contentItem ) use ( $testId ) {
                return $contentItem->PresentationContentId == $testId;
            };
        }

        ;

        $origPresentation = self::$client->QueryPresentationsById(array( $this->TestingPresentationGuid ))->Presentations[0];
        $callback         = create_id_callback($testId);
        $origContents     = array_filter($origPresentation->Content->PresentationContentDetails, $callback);
        $origContent      = array_pop($origContents);

        $origFileName       = $origContent->FileNameWithExtension;
        $newContentFileName = "testing new name.mp4";

        $deets = new PresentationContentUpdateDetails( null, false, null, false, null, false, null, false, $newContentFileName, true, null, false, null, false, 5, true );

        self::$client->UpdatePresentationContentDetails($origContent->PresentationContentId, $this->TestingPresentationGuid, $deets);
        $revisedPresentation = self::$client->QueryPresentationsById(array( $this->TestingPresentationGuid ))->Presentations[0];
        $revisedContents     = array_filter($revisedPresentation->Content->PresentationContentDetails, $callback);
        $revisedContent      = array_pop($revisedContents);

        $this->assertEquals($newContentFileName, $revisedContent->FileNameWithExtension, 'Updated file name does not match');

        $deets->FileName = $origFileName;
        self::$client->UpdatePresentationContentDetails($origContent->PresentationContentId, $this->TestingPresentationGuid, $deets);

    }

    /**
     * @covers DeletePresentationContent
     * @covers CreatePresentationExternalLink
     * @group stat-specific
     *  requires specific presentation and presentation content ids to be present and correctly configured
     */
    public function test_DeletePresentationContent() {

        $createDeets = new CreatePresentationLikeDetails( $this->TestingPresentationId, "New preso for content deletion", 19 );

        $newPreso = self::$client->CreatePresentationLike($createDeets);
        self::$client->CreatePresentationExternalLink("LinkName", 1, $newPreso->Presentation->Id, "http://mediasite.com");
        $preDeleteRevisedPresos = self::$client->QueryPresentationsById(array( $newPreso->Presentation->Id ));

        $content        = $preDeleteRevisedPresos->Presentations[0]->Content->PresentationContentDetails;
        $nonLinkContent = array();
        $foundLink      = false;

        //find new external link
        foreach ( $content as $contentItem ) {
            if ( isset( $contentItem->ExternalLinks ) ) {
                $this->assertTrue($contentItem->ExternalLinks->ExternalLinkDetails[0]->Url == "http://mediasite.com", 'Link url is not the provided url');
                $foundLink = true;
            } else {
                array_push($nonLinkContent, $contentItem);
            }
        }
        $this->assertTrue($foundLink, 'No links were found');
        $contentCount    = count($content);
        $contentToRemove = array_pop($nonLinkContent);

        //  can't remove the transcode source
        if ( $contentToRemove->IsTranscodeSource ) {
            $contentToRemove = array_pop($nonLinkContent);
        }
        $deleteReq = self::$client->DeletePresentationContent($contentToRemove->PresentationContentId, $newPreso->Presentation->Id);

        $revisedPresos = self::$client->QueryPresentationsById(array( $newPreso->Presentation->Id ));

        $this->assertCount($contentCount - 1, $revisedPresos->Presentations[0]->Content->PresentationContentDetails, 'content count is not one less than original');
        self::$client->DeletePresentation($newPreso->Presentation->Id);

    }

    /**
     * @covers QueryPresentationUsageById
     * @group state-sensitive
     *  presentation id must exist
     */
    public function test_QueryPresentationUsageById() {
        $response = self::$client->QueryPresentationUsageById($this->TestingPresentationGuid, AnalyticsRequestType::Presentation, AnalyticsRequestType::User);
        $this->assertEquals($response->Results->TotalResults, count($response->UsageList), 'Usage list count and Total results do not match');
    }

    /**
     * @covers QueryPresentationUsageById
     * @group state-sensitive
     *  matching profile must exist
     */
    public function test_QueryUserProfileDetailsByEmailContainsCriteria_AllProfileEmailsShouldContainSearchString() {
        $criteria = new UserProfileQueryCriteria( null, $this->EmailSearchString );
        $resp     = self::$client->QueryUserProfileDetailsByCriteria($criteria);
        foreach ( $resp->UserProfiles as $profile ) {
            $this->assertContains($this->EmailSearchString, $profile->Email, 'Email address of User Profile does not contain search string', true);
        }
    }

    /**
     * @covers QueryUserProfileDetailsById
     * @group state-sensitive
     *  matching profile must exist
     */
    public function test_QueryUserProfileDetailsById_ReturnsProfileWithRequestedId() {
        $upList = array( $this->TestUserProfileId );
        $resp   = self::$client->QueryUserProfileDetailsById($upList);
        $this->assertEquals($this->TestUserProfileId, $resp->UserProfiles[0]->Id, 'Returned Profile Id does not match requested Profile Id');
    }

    /**
     * @covers CreatePresentationForMediaUpload
     * @covers CreateMediaUpload
     * @group state-sensitive
     *  must be able to find referenced file and php's curllib must be available
     */
    public function test_CreatePresentationForMediaUpload_UploadMedia() {
        $uploadFullPath = "c:\\temp\\small-file.wmv";
        $pathbits       = pathinfo($uploadFullPath);
        $fileName       = $pathbits['filename'] . '.' . $pathbits['extension'];
        $resp           = self::$client->CreatePresentationForMediaUpload('Upload Presentation V');
        $fh             = fopen($uploadFullPath, 'rb');
        $ch             = curl_init();
        curl_setopt($ch, CURLOPT_PUT, true);
        curl_setopt($ch, CURLOPT_HTTPAUTH, 'CURLAUTH_BASIC');
        curl_setopt($ch, CURLOPT_USERPWD, self::$Username . ':' . self::$Password);
        curl_setopt($ch, CURLOPT_INFILE, $fh);
        curl_setopt($ch, CURLOPT_INFILESIZE, filesize($uploadFullPath));
        curl_setopt($ch, CURLOPT_URL, $resp->Details->UploadUrl . '/' . $fileName);
        curl_exec($ch);
        curl_close($ch);

        $jobResult = self::$client->CreateMediaUpload($fileName, $resp->Details->PresentationId, MediaUploadTranscodeOptionDetails::Always)->CreateMediaUploadResult;
        $jobState  = self::$client->CheckJobStatus($jobResult->JobId);
        $this->assertEquals('Queued', $jobState->Status, 'Transcode Job status is not "Queued"');
        self::$client->DeletePresentation($resp->Details->PresentationId);
    }

    /**
     * @cover RefreshResponseData
     * @group stateless
     */
    public function test_RefreshReportData_ReturnsValidJobId() {
        $resp = self::$client->RefreshReportData();

        $jobResp = self::$client->CheckJobStatus($resp->JobId);
        $this->assertNotNull($jobResp->Status, "Resulting job status is null");
    }

    /**
     * @cover CreateRegistration
     * @group stateless
     */
    public function test_CreateRegistration() {
        $resourceRegistrationCustomFields[] = new ResourceRegistrationCustomFieldDefinition(false, "Label");
        $registrationDetails = new RegistrationDetails( $resourceRegistrationCustomFields, false, "", "", "", $this->TestingPresentationGuid);
        try {
            self::$client->CreateRegistration($registrationDetails);
        } catch (Exception $e) {
            print($e->getMessage());
        }
    }

    /**
     * @cover DisableRegistration
     * @group stateless
     */
    public function test_DisableRegistration() {
        self::$client->DisableRegistration($this->TestingPresentationGuid);
    }

    /**
     * @cover AddRegistrationToResource
     * @group stateless
     */
    public function test_AddRegistrantsToResource() {
        $customFields[] = new ResourceRegistrationCustomField( "", "" );
        $resourceDetails[] = new ResourceRegistrationDetail($customFields, "KevinB@sonicfoundry.com", "Kevin", "Burton", $this->TestingPresentationGuid);
        self::$client->AddRegistrantsToResource($resourceDetails);
    }

    /**
     * @cover RemoveRegistrationFromResource
     * @group stateless
     */
    public function test_RemoveRegistrantsFromResource() {
        $customFields[] = new ResourceRegistrationCustomField( "", "" );
        $registrationDetails[] = new ResourceRegistrationDetail($customFields, "KevinB@sonicfoundry.com", "Kevin", "Burton", $this->TestingPresentationGuid);
        self::$client->RemoveRegistrantsFromResource($registrationDetails);
    }

    /**
     * @cover QueryResourceRegistrants
     * @group stateless
     */
    public function test_QueryResourceRegistrants() {
        $response = self::$client->QueryResourceRegistrants($this->TestingPresentationGuid);
        $this->assertNotNull($response->RegistrationDetails, "Registration details is null");
        // $this->assertNotNull($response->RegistrationDetails->ResourceRegistrationDetail, "Resource registration details is null");
    }

    /**
     * @cover CreateMp3Content
     * @group stateless
     */
    public function test_CreateMp3Content() {
        // Bit rate should be an acceptable value or you get 'Object reference not set to an instance of an object'
        try {
            self::$client->CreateMp3Content(128, $this->TestingPresentationGuid);
        } catch (Exception $e) {
            print($e->getMessage());
        }
    }

    /**
     * @cover CreateMediaImportProject
     * @group stateless
     */
    public function test_CreateMediaImportProject() {
        $dropboxDetails = new DropboxDetails("Location", "Password", "Username");
        $importOptions = new PresentationImportOptionDetails(true, true, true, '2013-07-18', true);
        $createDetails = new CreateMediaImportProjectDetails($dropboxDetails, $importOptions, "Name", $this->TestFolderId, '2013-07-18', $this->SourceTemplateId, 15);
        $response = self::$client->CreateMediaImportProject($createDetails);
        $this->assertNotNull($response->MediaImportProjectId, "Media import project id is null");
    }

    /**
     * @cover QueryContentImportProjectProgress
     * @group stateless
     */
    public function test_QueryContentImportProjectProgress() {
        $fileNames[] = "first file";
        $fileNames[] = "second file";
        $response = self::$client->QueryContentImportProjectProgress($fileNames, $this->TestProjectId);
        $this->assertNotNull($response->ProgressList, "Progress list is null");
        $this->assertNotNull($response->ProgressList->ImportProjectItemProgress, "Progress item list is null");
    }
    /**
     * @cover CreatePublishToGoContent
     * @group state-sensitive
     */
    public function test_CreatePublishToGoContent() {
        $response = self::$client->CreatePublishToGoContent();
        $this->assertNotNull($response->Results, "CreatePublishToGoContent is null");
    }
    /**
     * @cover QueryUserViews
     * @group stateless
     */
    public function test_QueryUserViews() {
        $response = self::$client->QueryUserViews(array('MediasiteAdmin'));
        $this->assertNotNull($response->Results, "QueryUserViews is null");
        $this->assertNotCount(0, $response->ViewsList, "No results");
    }
    /**
     * @cover QueryUserPresentationViews
     * @group stateless
     */
    public function test_QueryUserPresentationViews() {
        $response = self::$client->QueryUserPresentationViews('MediasiteAdmin');
        $this->assertNotNull($response->Results, "QueryUserPresentationViews is null");
        $this->assertNotCount(0, $response->ViewsList, "No results");
    }
    /**
     * @cover QueryUserIpAddressViews
     * @group stateless
     */
    public function test_QueryUserIpAddressViews() {
        $response = self::$client->QueryUserIpAddressViews('MediasiteAdmin');
        $this->assertNotNull($response->Results, "QueryUserIpAddressViews is null");
        $this->assertNotCount(0, $response->ViewsList, "No results");
    }
    /**
     * @cover QueryPresentationIpAddressDetails
     * @group state-sensitive
     */
    public function test_QueryPresentationIpAddressDetails() {
        $response = self::$client->QueryPresentationIpAddressDetails($this->TestingPresentationId);
        $this->assertNotNull($response->Results, "QueryPresentationIpAddressDetails is null");
    }
    /**
     * @cover QueryPresentationIpAddressDetails
     * @group stateless
     */
    public function test_QueryUserPresentationDetails() {
        $response = self::$client->QueryUserPresentationDetails('MediasiteAdmin');
        $this->assertNotNull($response->Results, "QueryUserPresentationDetails is null");
        $this->assertNotCount(0, $response->UsageList, "No results");
    }
}
