#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdint.h>
#include <stdbool.h>
#include <stdarg.h>
#include <syslog.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <net/if.h>
#include <net/if_arp.h>

#ifndef VERSION
#define VERSION "$Id: send_na.c 41671 2014-06-05 10:28:48Z wsl $"
#endif

static bool usesyslog = false;
static const char *progname;

static void error(const char *fmt, ...) {
	va_list ap;
	va_start(ap, fmt);
	fprintf(stderr, "%s: ", progname);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	if(usesyslog) {
		va_start(ap, fmt);
		vsyslog(LOG_ERR, fmt, ap);
		va_end(ap);
	}
	exit(2);
}

static unsigned long csum_buf(const unsigned char *buf, size_t len) {
	unsigned long t = 0;
	size_t i;
	union {
		unsigned char b[32];
		uint16_t w[16];
	} sa;
	while(len > sizeof sa.b) {
		memcpy(sa.b, buf, sizeof sa.b);
		for(i = 0; i < sizeof sa.b; i += 2)
			t += sa.w[i / 2];
		buf += sizeof sa.b;
		len -= sizeof sa.b;
	}
	if(len & 1)
		sa.b[len] = 0;
	memcpy(sa.b, buf, len);
	for(i = 0; i < len; i += 2)
		t += sa.w[i / 2];
	return t;
}

static unsigned long csum_u32(uint32_t len) {
	union {
		uint32_t u;
		uint16_t w0, w1;
	} sa;
	sa.u = htonl(len);
	return (unsigned long)sa.w0 + (unsigned long)sa.w1;
}

static const struct option long_options[] = {
	{"help\0               Print this message to stdout", 0, 0, 'h'},
	{"version\0            Print the program version", 0, 0, 'v'},
	{"syslog\0             Log errors to syslog", 0, 0, 's'},
	{"foreground\0         Do not background after the first packet", 0, 0, 'f'},
	{"repeat\0 <count>     Send <count> additional packets", 2, 0, 'r'},
	{0, 0, 0, 0}
};

static void get_short_options(const struct option *lo, char *buf) {
	int i;
	*buf++ = ':';
	while(lo->name) {
		i = lo->val;
		if(i) {
			*buf++ = (char)i;
			i = lo->has_arg;
			while(i--)
				*buf++ = ':';
		}
		lo++;
	}
	*buf++ = '\0';
}

static void usage(FILE *fh) {
	int i;
	fprintf(fh, "usage: %s [-", progname);
	for(i = 0; long_options[i].name; i++)
		if(long_options[i].val && !long_options[i].has_arg)
			fputc(long_options[i].val, fh);
	fprintf(fh, "] [-r <count>] <interface> <address>\n");
	for(i = 0; long_options[i].name; i++)
		fprintf(fh, "\t-%c, --%s%s\n",
			long_options[i].val,
			long_options[i].name,
			long_options[i].name + strlen(long_options[i].name) + 1);
}


int main(int argc, char **argv) {
	struct sockaddr_in6 sockaddr = {0};
	struct ifreq ifr;
	unsigned long csum_state = 0;
	uint16_t csum;
	char short_options[32];
	int option_index;
	int fd;
	int ifindex;
	int hops = 255;
	int i;
	const char *ifname, *ip;
	unsigned char buf[32] = "\x08\x00\x00\x00\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x01\x00\x00\x00\x00\x00\x00";
	int count = 1;
	char *var;
	bool daemonize = true;

	progname = strchr(argv[0], '/');
	if(progname)
		progname++;
	else
		progname = argv[0];

	setlinebuf(stderr);

	get_short_options(long_options, short_options);

	opterr = 0;
	while((i = getopt_long(argc, argv, short_options, long_options, &option_index)) != EOF) {
		switch(i) {
			case 'h':
				puts("unsort - reorder files semi-randomly");
				usage(stdout);
				exit(0);
			case 'v':
				printf("send_na %s\nCopyright 2014 Tilburg University\n"
					"\tThis program is free software: you can redistribute it and/or modify\n"
					"\tit under the terms of the GNU General Public License as published by\n"
					"\tthe Free Software Foundation, either version 3 of the License, or\n"
					"\t(at your option) any later version.\n"
					"\n"
					"\tThis program is distributed in the hope that it will be useful,\n"
					"\tbut WITHOUT ANY WARRANTY; without even the implied warranty of\n"
					"\tMERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
					"\tGNU General Public License for more details.\n"
					"\n"
					"\tYou should have received a copy of the GNU General Public License\n"
					"\talong with this program.  If not, see <http://www.gnu.org/licenses/>.\n",
					VERSION);
				exit(0);
			case 's':
				usesyslog = true;
				break;
			case 'r':
				count = atoi(optarg);
				break;
			case 'f':
				daemonize = false;
				break;
			case '?':
				usage(stderr);
				error("unknown option: -%c", optopt);
			case ':':
				usage(stderr);
				error("option -%c requires an argument", optopt);
			default:
				usage(stderr);
				error("unknown option: -%c", i);
		}
	}

	if(usesyslog)
		openlog(progname, LOG_NDELAY|LOG_PID, LOG_DAEMON);

	if(argc - optind != 2)
		error("usage: %s [options] <interface> <IPv6 address>\n", progname);

	ifname = argv[optind];

	if(strlen(ifname) > sizeof ifr.ifr_name - 1)
		error("interface name '%s' too long\n", ifname);
	strcpy(ifr.ifr_name, ifname);

	ip = argv[optind + 1];

	sockaddr.sin6_family = AF_INET6;
	switch(inet_pton(AF_INET6, ip, &sockaddr.sin6_addr)) {
		case 0:
			error("'%s' is not a valid IPv6 address\n", ip);
		case -1:
			error("'%s' is not a valid IPv6 address: %s\n", ip, strerror(errno));
	}

	fd = socket(PF_INET6, SOCK_RAW, IPPROTO_ICMPV6|SOCK_CLOEXEC);
	if(fd == -1)
		error("socket(PF_INET6, SOCK_RAW, IPPROTO_ICMPV6): %s\n", strerror(errno));

	if(ioctl(fd, SIOCGIFINDEX, &ifr) == -1)
		error("ioctl(SIOCGIFINDEX, %s): %s\n", ifname, strerror(errno));
	ifindex = ifr.ifr_ifindex;
	sockaddr.sin6_scope_id = (uint32_t)ifindex;

	if(ioctl(fd, SIOCGIFHWADDR, &ifr) == -1)
		error("ioctl(SIOCGIFHWADDR, %s): %s\n", ifname, strerror(errno));

	if(ifr.ifr_hwaddr.sa_family != ARPHRD_ETHER)
		error("%s is not an ethernet device\n", ifname);
	memcpy(buf + 26, ifr.ifr_hwaddr.sa_data, 6);

	if(setsockopt(fd, SOL_IPV6, IPV6_MULTICAST_IF, &ifindex, sizeof ifindex) == -1)
		error("setsockopt(IPV6_MULTICAST_IF, %s): %s\n", ifname, strerror(errno));

	if(setsockopt(fd, SOL_IPV6, IPV6_MULTICAST_HOPS, &hops, sizeof hops) == -1)
		error("setsockopt(IPV6_MULTICAST_HOPS, %d): %s\n", hops, strerror(errno));

	if(bind(fd, &sockaddr, sizeof sockaddr) == -1)
		error("bind(%s): %s\n", ip, strerror(errno));

	if(count < 1)
		return 0;

	memcpy(buf + 8, sockaddr.sin6_addr.s6_addr, sizeof sockaddr.sin6_addr.s6_addr);
	csum_state += csum_buf(sockaddr.sin6_addr.s6_addr, sizeof sockaddr.sin6_addr.s6_addr);
	csum_state += csum_u32(sizeof buf);
	csum_state += csum_u32(IPPROTO_ICMPV6);
	csum_state += csum_buf(buf, sizeof buf);

	inet_pton(AF_INET6, "ff02::1", &sockaddr.sin6_addr);
	sockaddr.sin6_scope_id = 0;

	csum_state += csum_buf(sockaddr.sin6_addr.s6_addr, sizeof sockaddr.sin6_addr.s6_addr);
	while(csum_state > UINT16_MAX)
		csum_state = (csum_state & UINT16_MAX) + (csum_state >> 16);
	csum = (uint16_t)~(uint16_t)csum_state;
	memcpy(buf + 2, &csum, sizeof csum);

	if(sendto(fd, buf, sizeof buf, 0, &sockaddr, sizeof sockaddr) != sizeof buf)
		error("sendto(ff02::1): %s\n", strerror(errno));

	if(count > 1) {
		if(daemonize) {
			if(daemon(0, 0) == -1)
				error("daemon(): %s\n", strerror(errno));
			usesyslog = true;
		}

		for(i = 1; i < count; i++) {
			usleep(100000);
			if(sendto(fd, buf, sizeof buf, 0, &sockaddr, sizeof sockaddr) != sizeof buf)
				error("daemon(): %s\n", strerror(errno));
		}
	}

	close(fd);

	return 0;
}
