use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::Crypto;

use MIME::Base64;
use Encode;
use Digest::SHA qw(hmac_sha256);
use Crypt::OpenSSL::Random;
use Time::HiRes qw(time);
use Xyzzy::Util qw(utf8_testandset_inplace utf8_disable);

use Clarity -self;

field cfg;
field secret => sub { shift->cfg->cryptosecret };
field jitter => sub { shift->cfg->cryptoclockjitter };

sub random_bytes {
	die "Insufficient entropy\n"
		unless Crypt::OpenSSL::Random::random_status;
	return Crypt::OpenSSL::Random::random_bytes(@_);
}

sub random_hex {
	my $hex = unpack('H*', $self->random_bytes($@));
	$hex =~ tr/a-z/A-Z/;
	return $hex;
}

sub hmac {
	return hmac_sha256(map { utf8_disable($_) } @_, $self->secret);
}

sub create_token_data {
	my ($hidden, @keys) = @_;

	foreach(@keys) {
		Encode::_utf8_off($_);
		die "Bad characters in request\n"
			unless index($_, "\0") < 0;
	}

	my $time = int(time * 1e6);
	my $salt = $self->random_bytes(16);
	my $token = @keys && $keys[-1] eq ''
		? pack('Q>a*(Z*)*', $time, $salt, @keys)
		: pack('Q>a*a*', $time, $salt, join("\0", @keys));

	my $hash = $self->hmac($hidden, "\0", $token);

	return $time, $salt, $hash.$token;
}

sub create_token {
	my ($time, $salt, $raw) = $self->create_token_data(@_);

	my $encoded = encode_base64($raw, '');
	$encoded =~ s/=+$//;
	$encoded =~ tr|/+|._|;

	return $time, $salt, $encoded;
}

sub check_token_data {
	my ($hidden, $token, $expiry) = @_;

	my $hash = substr($token, 0, 32, '');
	die "Short token\n"
		unless length($hash) == 32;

	die "Bad signature on token\n"
		unless $self->hmac($hidden, "\0", $token) eq $hash;

	my ($time, $salt, @keys) = unpack('Q>a16(Z*)*', $token);

	# work around a quirk in unpack():
	@keys = () if length($token) == 24;

	$time /= 1e6;

	my $now = time;

	die "Future token\n"
		if $time > $now + $self->jitter;

	die "Expired token\n"
		if $expiry && $time + $expiry < $now;

	utf8_testandset_inplace(@keys);

	if(wantarray) {
		return $time, $salt, @keys;
	} else {
		return (shift @keys) // 1;
	}
}

sub check_token {
	my ($hidden, $token, $expiry) = @_;

	die "Undefined token\n"
		unless defined $token;

	die "Invalid token\n"
		unless $token =~ /^[a-zA-Z0-9_.]+$/;

	$token =~ tr|._|/+|;
	$token .= '=' x -(length($token) % -4);
	$token = decode_base64($token);

	return $self->check_token_data($hidden, $token, $expiry);
}

__END__

=pod

=encoding utf8

=head1 Xyzzy::Crypto

This module provides functions for creating signed tokens that can be used
in lieu of persistent session storage.

Each token has an optional public part that is stored inside the token, a
hidden part that is not stored inside the token but is required for
validation, and a timestamp to allow for expiry.

Tokens are signed using a passphrase that can be configured using
Xyzzy::Crypto::Config.

Ascii encoded tokens use a base64 alphabet using A-Za-z0-9 and the
characters '.' and '-'.

=head2 random_bytes($num)

Returns $num cryptographically random bytes.

=head2 random_hex($num)

Like random_bytes() but returns the result hex-encoded.

=head2 hmac($data ...)

Calculates a HMAC of the specified data, using the configured secret.

=head2 create_token($hidden, ...$public...)

Creates an ascii encoded token from a hidden part and an optional public
part. Returns the timestamp, salt and (ascii encoded) token string.

The public parts may not contain \0 characters (because those are used as
separators).

Example:

	# Create a session token, tied to the IP-address:
	my $session = $crypto->create_token('s'.$remote_addr, $uid);

	# Create an anti-CSRF nonce based on the session:
	my $nonce = $crypto->create_token('n'.$session);

In scalar context, this function returns just the token string.

=head2 check_token($hidden, $token, $lifetime)

Checks whether the (ascii encoded) cookie is valid. Returns the timestamp,
salt and any public data if the token was valid. It will die() if the token
was invalid somehow. The $hidden parameter must be exactly the same as when
the token was created or the check will fail. Example:

	# check a session token (bound to IP address, valid for one hour)
	my (undef, undef, $uid) = $crypto->check_token('s'.$remote_addr, $session, 3600);

	# same as above, but catch errors more gracefully:
	my (undef, undef, $uid) = eval { $crypto->check_token('s'.$remote_addr, $session, 3600) }
	or warn "session token: $@";

In scalar context, check_token will return the first public data item. If
no public data items are attached to the token, it will return 1.

=head2 create_token_data($hidden, ...$public...)

Same as create_token, but the token data will be in binary form.

=head2 check_token_data($hidden, $token, $lifetime)

Same as check_token, but the token data must be in binary form.

=cut
