package Xyzzy::Header;

use Clarity -self;

use Scalar::Util qw(blessed);

# A Xyzzy::Header represents the value of an HTTP header, that is, the part
# after the colon in for example ‘Location: http://example.com’.

# Many header definitions allow multiple values, separated with semicolons.
# Often these are in key=value format.
# Xyzzy::Header is able to parse and represent this format so that you
# can retrieve the keys and values seperately.

# The data structure for a Xyzzy::Header is an array; if the number of
# elements is odd, the first element is a literal value that is not subject
# to further interpretation. Any subsequent values are interpreted as
# key-value pairs. An undefined key means that only the value is present
# (there will be no equals sign in the serialization).

# The arguments to the new method follow the same convention.

sub new {
	# make a copy, so we don't ruin the values when we check it (below)
	my @vals = @_;

	# literal values
	if(@_ & 1) {
		my $val = shift;
		confess("header value contains newline characters")
			if $val =~ /[\x0A\x0D]/a;
	}

	# key-value pairs
	while(@_) {
		my $key = shift;
		my $val = shift;
		confess("malformed header token '$key'")
			if defined $key && !istoken($key);
		confess("header value contains newline characters")
			if defined $val && $val =~ /[\x0A\x0D]/a;
	}

	return bless \@vals, blessed($self) // $self;
}

sub unslash() {
	my $str = shift;
	return $str unless defined $str;
	local $1;
	$str =~ s/\\(.)/$1/ga;
	return $str;
}

# Do a forgiving ‘tag soup’ like parse of a header. Deal with possible
# broken or malicious headers from untrusted sources.
sub parse {
	my $str = shift;
	$str =~ s/[\x0A\x0D]+/ /ga;
	my @vals;
	push @vals, unslash($1) // $2
		if $str =~ s{^\s*+(?:"((?:\\.|[^"])*+)"|([^=;]*?))\s*(?:;\s*+|$)}{}a;
	$str =~ s{([^=;"]+)\s*+(?:=\s*+(?:"((?:\\.|[^"])*+)"|([^;]*?)))?\s*(?:;\s*+|$)}
		{push @vals, $1, unslash($2) // $3; ''}ega;
	# Don't use new, as it might flag bad keys/values we have no control over.
	return bless \@vals, blessed($self) // $self;
}

sub value {
	return undef unless @$self & 1;
	return $self->[0];
}

sub attributes {
	my @attrs = @$self;
	shift @attrs if @attrs & 1;
	return wantarray ? @attrs : \@attrs;
}

sub istoken() {
	return $_[0] =~ /^[!\#-'*.0-9A-Z^_\`a-z|-]+$/a;
}

sub isnasty() {
	return $_[0] =~ /[^ -~]|^\s|\s$|\s\s|[;"]/a;
}

sub toString {
	my @vals = @{$self};

	my @out;

	push @out, shift @vals
		if @vals & 1;

	while(@vals) {
		my $key = shift @vals;
		my $val = shift @vals;
		my $out = defined $key ? $key : '';
		if(defined $val) {
			if(isnasty($val)) {
				$val =~ s/(["\\])/\\$1/ga;
				$val = "\"$val\"";
			}

			$out .= '='
				if defined $key;

			$out .= $val;
		}
		push @out, $out;
	}

	return join('; ', @out);
}

use overload
	'bool' => sub { 1 },
	'""' => \&toString,
	fallback => 1;
