use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::Util;

use Encode;
use URI::Escape;
use Exporter 'import';

our @EXPORT_OK = qw(iso8601 iso8601_date iso8601_strict utf8_testandset_inplace utf8_testandset utf8_disable_inplace utf8_disable uri_escape_auto uri_escape_plus uri_unescape_plus htmlentities parse_time parse_size parse_bool parse_path);
our %EXPORT_TAGS = (all => \@EXPORT_OK);

sub iso8601 {
	my ($sec, $min, $hour, $mday, $mon, $year) = @_;
	return sprintf('%04d-%02d-%02d %02d:%02d:%02d',
		$year + 1900, $mon + 1, $mday, $hour, $min, $sec)
}

sub iso8601_date {
	my (undef, undef, undef, $mday, $mon, $year) = @_;
	return sprintf('%04d-%02d-%02d',
		$year + 1900, $mon + 1, $mday)
}

sub iso8601_strict {
	my ($sec, $min, $hour, $mday, $mon, $year) = @_;
	return sprintf('%04d-%02d-%02dT%02d:%02d:%02d',
		$year + 1900, $mon + 1, $mday, $hour, $min, $sec)
}

sub utf8_testandset_inplace {
	foreach my $val (@_) {
		# upgrade to UTF-8 if possible, without changing any bytes
		Encode::_utf8_on($val);
		Encode::_utf8_off($val) unless utf8::valid($val);
	}
	return;
}

sub utf8_testandset {
	my $str = shift;
	return undef unless defined $str;
	utf8_testandset_inplace($str);
	return $str;
}

sub utf8_disable_inplace {
	foreach my $val (@_) {
		Encode::_utf8_off($val);
	}
	return;
}

sub utf8_disable {
	my $str = shift;
	return undef unless defined $str;
	Encode::_utf8_off($str);
	return $str;
}

sub uri_escape_auto {
	my $val = shift;
	Encode::_utf8_off($val);
	return uri_escape($val, @_);
}

sub uri_escape_plus {
	my $str = uri_escape_auto(@_);
	$str =~ s/%20/+/g;
	return $str;
}

sub uri_unescape_plus {
	return uri_unescape(map { my $str = $_; $str =~ tr/+/ /; $str } @_);
}

{
	my %htmlentities = (
		'"' => '&quot;',
		"'" => '&apos;',
		'<' => '&lt;',
		'>' => '&gt;',
		'&' => '&amp;',
	);

	my $escapes = quotemeta(join('', keys %htmlentities));

	sub htmlentities {
		my $html = shift;

		$html =~ s/([$escapes])/$htmlentities{$1}/ego;
		$html =~ s/([^\t\n -~])/'&#'.ord($1).';'/eg;

		return $html;
	}
}

{
	my %units = (
		'' => 1,
		ns => 0.000000001,
		us => 0.000001,
		ms => 0.001,
		s => 1,
		m => 60,
		h => 3600,
		d => 86400,
		w => 604800,
		l => 31 * 86400,
		q => 92 * 86400,
		y => int(365.2425 * 86400),
	);

	sub parse_time {
		my ($time) = @_;
		$time =~ /^(\d+)\s*(\w*)$/i;
		my ($scalar, $unit) = ($1, $2);
		die "can't parse time value '$time'\n"
			unless defined $scalar;
		my $multiplier = $units{lc $unit}
			or die "unknown unit $unit\n";
		return $scalar * $multiplier;
	}
}

{
	my %units = (
		'' => 1,
		k => 1024,
		m => 1048576,
		g => 1073741824,
		t => 1099511627776,
		p => 1125899906842624,
		e => 1152921504606846976,
		z => 1180591620717411303424,
		y => 1208925819614629174706176,
	);

	sub parse_size {
		my ($size) = @_;
		$size =~ /^(\d+)\s*([a-z]?)$/i
			or die "can't parse size value '$size'\n";
		my ($bytes, $unit) = ($1, $2);
		my $multiplier = $units{lc $unit}
			or die "unknown unit $unit\n";
		return $bytes * $multiplier;
	}
}

sub parse_bool {
	local $_ = shift;
	die "undefined boolean value\n" unless defined;
	return 1 if /^(?:1|y|yes|true|on|enabled?)$/i;
	return 0 if /^(?:0|n|no|false|off|disabled?)$/i;
	die "unknown boolean value '$_'\n";
}

sub parse_path {
	return [map {
		utf8_testandset($_);
	} grep {
		$_ ne ''
	} split(qr{/+}, shift)];
}
